/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SequenceUtils.h"

#include <U2Core/DNATranslation.h>
#include <U2Core/TextUtils.h>

namespace U2 {

QList<QByteArray> SequenceUtils::extractRegions(const QByteArray& seq, const QList<LRegion>& regions, DNATranslation* complTT) {
    QList<LRegion> safeLocation = regions;
    LRegion::bound(0, seq.length(), safeLocation);

    QList<QByteArray> resParts;
    for (int i = 0, n = safeLocation.size(); i < n; i++) {
        const LRegion& oReg = safeLocation.at(i);
        if (complTT == NULL) {
            resParts.append(seq.mid(oReg.startPos, oReg.len));
        } else {
            QByteArray arr = seq.mid(oReg.startPos, oReg.len);
            TextUtils::reverse(arr.data(), arr.length());
            complTT->translate(arr.data(), arr.length());
            resParts.prepend(arr);
        }
    }
    return resParts;
}

QByteArray SequenceUtils::joinRegions(const QList<QByteArray>& parts) {
    if (parts.size() == 1) {
        return parts.first();
    }
    QByteArray res;
    foreach(const QByteArray& p, parts) {
        res.append(p);
    }
    return res;
}

QList<QByteArray> SequenceUtils::translateRegions(const QList<QByteArray>& origParts, DNATranslation* aminoTT, bool join) {
    QList<QByteArray> resParts;
    assert(aminoTT != NULL);
    if (join) {
        resParts.append(SequenceUtils::joinRegions(origParts));
    }  else {
        resParts.append(origParts);
    }
    for (int i = 0, n = resParts.length(); i < n; i++) {
        const QByteArray& d = resParts[i];
        int translatedLen = d.size() / 3;
        QByteArray translated(translatedLen, '?');
        aminoTT->translate(d.constData(), d.length(), translated.data(), translatedLen);
        resParts[i] = translated;
    }
    return resParts;
}


QList<QByteArray> SequenceUtils::extractSequence(const QByteArray& seq, const QList<LRegion>& origLocation, 
                                         DNATranslation* complTT, DNATranslation* aminoTT, bool join, bool circular)
{
    QList<QByteArray> resParts = extractRegions(seq, origLocation, complTT);
    if (circular && resParts.size() > 1) {
        const LRegion& firstL = origLocation.first();
        const LRegion& lastL = origLocation.last();
        if (firstL.startPos == 0 && lastL.endPos() == seq.length()) { 
            QByteArray lastS = resParts.last();
            QByteArray firstS = resParts.first();
            resParts.removeLast();
            resParts[0] = lastS.append(firstS);
        }
    }
    if (aminoTT != NULL) {
        resParts = translateRegions(resParts, aminoTT, join);
    } else if (join) {
        QByteArray joined = joinRegions(resParts);
        resParts.clear();
        resParts.append(joined);
    }
    return resParts;
}


QList<LRegion> SequenceUtils::toJoinedRegions(const QList<QByteArray>& seqParts) {
    QList<LRegion>  res;
    int prevEnd = 0;
    foreach(const QByteArray& seq, seqParts) {
        res.append(LRegion(prevEnd, seq.size()));
        prevEnd += seq.size();
    }
    return res;
}

}//namespace
