/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <core_api/AppContext.h>
#include <core_api/L10n.h>
#include <distributed_computing/SerializeUtils.h>

#include "MuscleLocalTask.h"

namespace GB2 {

/*************************************************
* MuscleLocalTaskSettings
*************************************************/

MuscleLocalTaskSettings::MuscleLocalTaskSettings() {
}

MuscleLocalTaskSettings::~MuscleLocalTaskSettings() {
}

MuscleLocalTaskSettings::MuscleLocalTaskSettings( const MAlignment & ma, const MuscleTaskSettings & s ) 
: alignment( ma ), settings( s ) {
}

QVariant MuscleLocalTaskSettings::serializeMuscleTaskSettings() const {
    QVariantList res;
    res << SerializeUtils::serializeValue( (int)settings.op );
    res << SerializeUtils::serializeValue( settings.maxIterations );
    res << SerializeUtils::serializeValue<unsigned long long>( settings.maxSecs );
    res << SerializeUtils::serializeValue( settings.stableMode );
    res << SerializeUtils::serializeValue( settings.alignRegion );
    res << SerializeUtils::serializeValue( settings.regionToAlign.startPos );
    res << SerializeUtils::serializeValue( settings.regionToAlign.len );
    res << SerializeUtils::serializeValue( settings.profile );
    res << SerializeUtils::serializeValue( settings.nThreads );
    return res;
}

QVariant MuscleLocalTaskSettings::serialize() const {
    QVariantList res;
    res << SerializeUtils::serializeValue( alignment );
    res << serializeMuscleTaskSettings();
    return res;
}

bool MuscleLocalTaskSettings::deserializeMuscleTaskSettings( const QVariant & data ) {
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    if( SERIALIZED_MUSCLE_TASK_SETTINGS_LIST_SZ != args.size() ) {
        return false;
    }
    
    {
        int tmp = 0;
        if( !SerializeUtils::deserializeValue( args[0], &tmp ) ) { return false; }
        settings.op = (MuscleTaskOp)tmp;
    }
    
    if( !SerializeUtils::deserializeValue( args[1], &settings.maxIterations ) ) {  return false; }
    
    {
        unsigned long long tmp = 0;
        if( !SerializeUtils::deserializeValue<unsigned long long>( args[2], &tmp ) ) { return false; }
        settings.maxSecs = (unsigned long)tmp;
    }
    
    if( !SerializeUtils::deserializeValue( args[3], &settings.stableMode ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[4], &settings.alignRegion ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[5], &settings.regionToAlign.startPos ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[6], &settings.regionToAlign.len ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[7], &settings.profile ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[8], &settings.nThreads ) ) { return false; }
    return true;
}

bool MuscleLocalTaskSettings::deserialize( const QVariant & data ) {
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    if( SERIALIZED_LIST_SZ != args.size() ) {
        return false;
    }
    
    if( !SerializeUtils::deserializeValue( args[0], &alignment ) ) { return false; }
    if( !deserializeMuscleTaskSettings( args[1] ) ) { return false; }
    return true;
}

MAlignment MuscleLocalTaskSettings::getMAlignment() const {
    return alignment;
}

MuscleTaskSettings MuscleLocalTaskSettings::getMuscleSettings() const {
    return settings;
}

/*************************************************
* MuscleLocalTaskResult
*************************************************/

MuscleLocalTaskResult::MuscleLocalTaskResult() {
}

MuscleLocalTaskResult::~MuscleLocalTaskResult() {
}

MuscleLocalTaskResult::MuscleLocalTaskResult( const MAlignment & m, const MAlignment & s ) 
: ma( m ), subMa( s ) {
}

QVariant MuscleLocalTaskResult::serialize() const {
    QVariantList res;
    res << SerializeUtils::serializeValue( ma );
    res << SerializeUtils::serializeValue( subMa );
    return res;
}

bool MuscleLocalTaskResult::deserialize( const QVariant & data ) {
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    if( SERIALIZED_LIST_SZ != args.size() ) {
        return false;
    }
    
    if( !SerializeUtils::deserializeValue( args[0], &ma ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[1], &subMa ) ) { return false; }
    return true;
}

void MuscleLocalTaskResult::setResult( const MAlignment & m, const MAlignment & s ) {
    ma = m;
    subMa = s;
}

MAlignment MuscleLocalTaskResult::getMa() const {
    return ma;
}

MAlignment MuscleLocalTaskResult::getSubMa() const {
    return subMa;
}

/*************************************************
* MuscleLocalTask
*************************************************/

MuscleLocalTask::MuscleLocalTask( MuscleLocalTaskSettings * s )
: LocalTask( "", TaskFlags( TaskFlag_NoRun ) | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled ), 
  settings( s ), muscleTask( NULL ) {
      
    if( NULL == settings ) {
        setTaskName( tr( "Muscle local task" ) );
        setError( tr( "No settings given" ) );
        return;
    }
    setTaskName( tr( "Muscle local task. Alignment: %1" ).arg( settings->getMAlignment().getName() ) );
}

MuscleLocalTask::~MuscleLocalTask() {
    delete settings;
    settings = NULL;
}

void MuscleLocalTask::prepare() {
    if( hasErrors() ) {
        return;
    }
    assert( NULL != settings );
    muscleTask = new MuscleTask( settings->getMAlignment(), settings->getMuscleSettings() );
    addSubTask( muscleTask );
}

Task::ReportResult MuscleLocalTask::report() {
    if( hasErrors() ) {
        return ReportResult_Finished;
    }
    if( muscleTask->hasErrors() ) {
        setError( muscleTask->getError() );
        return ReportResult_Finished;
    }
    result.setResult( muscleTask->resultMA, muscleTask->resultSubMA );
    return ReportResult_Finished;
}

const LocalTaskResult * MuscleLocalTask::getResult() const {
    return &result;
}

QString MuscleLocalTask::generateReport() const {
    QString res;
    res += "<table>";
    res+="<tr><td width=200><b>" + tr("Alien Muscle task from remote machine.") + "</b></td><td>";
    res += "<tr><td><b>" + tr("Alignment") + "</b></td><td>" + settings->getMAlignment().getName() + "</td></tr>";
    
    if( hasErrors() || isCanceled() ) {
        res += "<tr><td width=200><b>" + tr("Task finished with error") + "</b></td><td></td></tr>";
        res += "</table>";
        return res;
    }
    res += "</table>";
    return res;
}

/*************************************************
* MuscleRemoteToGobjectTask
*************************************************/

MuscleRemoteToGobjectTask::MuscleRemoteToGobjectTask( MAlignmentObject * o, const MuscleTaskSettings & set, RemoteMachineSettings * s ) 
: Task( "Muscle align on remote machine", TaskFlags( TaskFlag_NoRun ) | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled ),
obj( o ), lock( NULL ), muscleTask( NULL ), machineSettings( s ), machine( NULL ), config( set ) {
    
    if( obj.isNull() ) {
        setError( tr( "Malignment object removed" ) );
        return;
    }
    if( NULL == machineSettings ) {
        setError( L10N::badArgument( tr( "remote machine settings" ) ) );
        return;
    }
}

MuscleRemoteToGobjectTask::~MuscleRemoteToGobjectTask() {
    assert( lock == NULL );
    delete machine;
    machine = NULL;
}

void MuscleRemoteToGobjectTask::prepare() {
    if( hasErrors() ) {
        return;
    }
    
    if( obj->isStateLocked() ) {
        setError( tr( "MAlignment object is state locked" ) );
        return;
    }
    lock = new StateLock( "musle lock" );
    obj->lockState( lock );
    
    machine = AppContext::getProtocolInfoRegistry()->getProtocolInfo( machineSettings->getProtocolId() )->
        getRemoteMachineFactory()->createInstance( machineSettings );
    if( NULL == machine ) {
        setError( tr( "Cannot create remote machine" ) );
        return;
    }
    
    MuscleLocalTaskSettings localSettings( obj->getMAlignment(), config );
    muscleTask = new RemoteTask( MuscleLocalTaskFactory::ID, localSettings, machine );
    addSubTask( muscleTask );
}

Task::ReportResult MuscleRemoteToGobjectTask::report() {
    if( NULL != lock ) {
        obj->unlockState( lock );
        delete lock;
        lock = NULL;
    }
    propagateSubtaskError();
    if( hasErrors() || isCanceled() ) {
        return ReportResult_Finished;
    }
    assert( !obj.isNull() );
    if( obj->isStateLocked() ) {
        setError( tr( "MAlignment object is state locked" ) );
        return ReportResult_Finished;
    }
    
    MuscleLocalTaskResult * result = dynamic_cast< MuscleLocalTaskResult* >( muscleTask->getResult() );
    if( NULL == result ) {
        setError( tr( "remote task didn't produced result" ) );
        return ReportResult_Finished;
    }
    
    obj->setMAlignment( result->getMa() );
    return ReportResult_Finished;
}

QString MuscleRemoteToGobjectTask::generateReport() const {
    QString res;
    res += "<table>";
    res+="<tr><td width=200><b>" + tr("Muscle task runned on remote machine.") + "</b></td><td>" + 
        machineSettings->toString() + "</td></tr>";
    res += "<tr><td><b>" + tr("Alignment") + "</b></td><td>" + obj->getMAlignment().getName() + "</td></tr>";
    
    if( hasErrors() || isCanceled() ) {
        res += "<tr><td width=200><b>" + tr("Task finished with error") + "</b></td><td></td></tr>";
        res += "</table>";
        return res;
    }
    res += "</table>";
    return res;
}

/*************************************************
* MuscleLocalTaskFactory
*************************************************/

template<> const QString MuscleLocalTaskFactory::ID = "Muscle3 align task";

} // GB2
