/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "BowtieConstants.h"
#include "BowtieAdapter.h"
#include "BowtieIOAdapter.h"
#include "BowtieContext.h"

#include <core_api/DNAAlphabet.h>
#include <core_api/Task.h>
#include <core_api/Log.h>
#include <datatype/MAlignment.h>
#include <datatype/DNASequence.h>

#include <QtCore/QVector>

#include <stdlib.h>
#include <stdexcept>
#include <vector>

using namespace std;
using namespace seqan;

extern void prepareSearchOptions();

void driverAdapter(const QList<GB2::DNASequence>& reads,
				   GB2::MAlignment& resultMA,
				   const char * type,
				   const string& ebwtFileBase,
				   const string& query,
				   const vector<string>& queries,
				   const vector<string>& qualities,
				   const string& outfile);

int bowtieBuildAdapter(const string& infile, const string& outfile);

namespace GB2 {

static LogCategory log(ULOG_CAT_BOWTIE);

void BowtieAdapter::doBowtie(const QString& ebwtFileName, const QList<GB2::DNASequence> &reads, MAlignment& output, TaskStateInfo& ti) {
	BowtieContext *ctx = BowtieContext::getContext();
	try {
		prepareSearchOptions();
		//driver<String<Dna, Alloc<> > >("DNA", ebwtFileName, query, queries, qualities, outfile);
		vector<string> queries;
		queries.push_back("reads/reads");
		driverAdapter(reads, output, "DNA", ebwtFileName.toAscii().constData(), "", queries, vector<string>(), "");
		if(output.getRows().count() == 0) {
			ti.setError(tr("Reference assembly failed - no possible alignment found"));
			return;
		}
		output.setAlphabet(reads.first().alphabet);
		//CHUD_STOP();
	}catch(exception& e) {
		ti.setError(QString("Bowtie exits with error \"%1\"").arg(e.what()));
		ctx->hasError = true;
	} catch(int e) {
		if(e != 0) {
			ti.setError(QString("Bowtie exits with error code %1").arg(e));
			ctx->hasError = true;
		}
	}
	if(ctx->hasError) {
		for(int i=0;i<ctx->search.nthreads-1;i++) {
			ctx->workerList[i]->start.release();
		}
	}
}



void BowtieAdapter::doBowtieBuild(const QString& infile, const QString& outfile, TaskStateInfo& ti )
{
	try{
		bowtieBuildAdapter(infile.toLocal8Bit().constData(), outfile.toLocal8Bit().constData());
	} catch(std::exception& e) {
		if(!(ti.cancelFlag && QString("Canceled") == e.what()))
		ti.setError(QString("Bowtie-build exits with error: %1").arg(e.what()));
	} catch(int e) {
		ti.setError(QString("Bowtie-build exits with error code %1").arg(e));
	}
}

void BowtieAdapter::doBowtieWorker(int id, TaskStateInfo& ti)
{
	BowtieContext* ctx = BowtieContext::getContext();
	ctx->workerList[id]->start.acquire();
	if(ctx->hasError || ctx->isCanceled())
	{
		return;
	}
	BowtieContext::BowtieWorkerArg arg = ctx->jobList[id];
	try {
		arg.start_routine(arg.arg);
	} 
	catch(int e) {
		if(e == 2) {
			log.trace(QString("Exiting worker %1").arg(id));
		} else {
			ti.setError("Bowtie worker throws exception");
			ctx->hasError = true;
		}
	}
	catch(...) {
		ti.setError("Bowtie worker throws exception");
		ctx->hasError = true;
	}
	ctx->workerList[id]->stop.release();
}
}