/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RemoteQueryWorker.h"

#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>
#include <core_api/Log.h>

#include <util_tasks/FailTask.h>

namespace GB2 {
namespace LocalWorkflow {


static LogCategory log(ULOG_CAT_WD);

const QString RemoteQueryWorkerFactory::ACTOR_ID("remote.query");

const QString DATABASE("Database");
const QString EXPECT("Expect_value");
const QString MAX_HITS("Max_hits");
const QString SHORT_SEQ("Short_sequence");
const QString ANNOTATION_NAME("Annotation_name");
const QString ORIGINAL_OUT("Blast_output");

void RemoteQueryWorkerFactory::init() {
    QList<PortDescriptor*> p; 
    QList<Attribute*> a;
    Descriptor ind(CoreLibConstants::IN_PORT_ID, RemoteQueryWorker::tr("Input sequence"), 
        RemoteQueryWorker::tr("Sequence for which annotations is searched"));
    Descriptor outd(CoreLibConstants::OUT_PORT_ID, RemoteQueryWorker::tr("Annotations"), 
        RemoteQueryWorker::tr("Found annotations"));

    p << new PortDescriptor(ind,BioDataTypes::DNA_SEQUENCE_TYPE(),true);
    p << new PortDescriptor(outd,BioDataTypes::ANNOTATION_TABLE_TYPE(), false, true);

    Descriptor db(DATABASE,RemoteQueryWorker::tr("Database"),
        RemoteQueryWorker::tr("Select database in witch search should be made."));
    Descriptor evalue(EXPECT,RemoteQueryWorker::tr("Expected value"),
        RemoteQueryWorker::tr("This setting specifies the statistical significance threshold for reporting matches against database sequences."));
    Descriptor hits(MAX_HITS,RemoteQueryWorker::tr("Max hits"),
        RemoteQueryWorker::tr("Maximum number of hits."));
    Descriptor short_seq(SHORT_SEQ,RemoteQueryWorker::tr("Short sequence"),
        RemoteQueryWorker::tr("Optimize search for short sequences."));
    Descriptor annotateAs(ANNOTATION_NAME,RemoteQueryWorker::tr("Annotate as"),
        RemoteQueryWorker::tr("Name for annotations"));
    Descriptor output(ORIGINAL_OUT, RemoteQueryWorker::tr("BLAST output"),
        RemoteQueryWorker::tr("Location of BLAST output file. This parameter insignificant for cdd search."));

    a << new Attribute(db,CoreDataTypes::STRING_TYPE(),true,"ncbi-blastn");
    a << new Attribute(evalue,CoreDataTypes::STRING_TYPE(),false,10);
    a << new Attribute(hits,CoreDataTypes::NUM_TYPE(),false,10);
    a << new Attribute(short_seq,CoreDataTypes::BOOL_TYPE(),false,false);
    a << new Attribute(annotateAs,CoreDataTypes::STRING_TYPE(),false);
    a << new Attribute(output, CoreDataTypes::STRING_TYPE(),false);

    Descriptor desc(ACTOR_ID, RemoteQueryWorker::tr("Request to remote database"), 
        RemoteQueryWorker::tr("Finds annotations for DNA sequence in remote database")
        );
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates; 

    {
        QVariantMap m;
        m["minimum"] = 1;
        m["maximum"] = 500;
        delegates[MAX_HITS] = new SpinBoxDelegate(m);
    }

    {
        QVariantMap m;
        m["ncbi-blastn"] = "ncbi-blastn";
        m["ncbi-blastp"] = "ncbi-blastp";
        m["ncbi-cdd"] = "ncbi-cdd";
        delegates[DATABASE] = new ComboBoxDelegate(m);
    }

    {
        QVariantMap m;
        m["1e-100"] = 1e-100;
        m["1e-10"] = 1e-10;
        m["1"] = 1;
        m["10"] = 10;
        m["100"] = 100;
        m["1000"] = 1000;
        delegates[EXPECT] = new ComboBoxDelegate(m);
    }

    delegates[ORIGINAL_OUT] = new URLDelegate("(*.xml)","xml file");

    proto->setPrompter(new RemoteQueryPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":remote_query/images/remote_db_request.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_BASIC(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new RemoteQueryWorkerFactory());
}

QString RemoteQueryPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* producer = input->getProducer(CoreLibConstants::IN_PORT_ID);
    QString producerName = producer ? tr(" from %1").arg(producer->getLabel()) : "";

    QString doc = tr("For sequence <u>%1</u> find annotations in database <u>%2</u>")
        .arg(producerName).arg(getParameter(DATABASE).toString());

    return doc;
}

void RemoteQueryWorker::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
}

bool RemoteQueryWorker::isReady() {
    return (input && input->hasMessage());
}

Task* RemoteQueryWorker::tick() {
    if((actor->getParameter(ANNOTATION_NAME)->getAttributeValue<QString>()).isEmpty()){
        log.error(tr("Annotations name is empty, default name used"));
    }
    
    Message inputMessage = getMessageAndSetupScriptValues(input);
    //cfg.minrl = 0;
    //cfg.maxrl = 3000;
    cfg.dbChoosen = actor->getParameter(DATABASE)->getAttributeValue<QString>().split("-").last();
    cfg.aminoT = NULL;

    int evalue = actor->getParameter(EXPECT)->getAttributeValue<int>();
    int maxHits = actor->getParameter(MAX_HITS)->getAttributeValue<int>();
    bool shortSeq = actor->getParameter(SHORT_SEQ)->getAttributeValue<bool>();

    if(evalue <= 0 ){
        log.error(tr("Incorrect value for 'e-value' parameter, default value passed to schema"));
        evalue = 10;
    }

    if(cfg.dbChoosen == "cdd") {
        cfg.params = "db=cdd";
        addParametr(cfg.params,ReqParams::cdd_hits,maxHits);
        addParametr(cfg.params,ReqParams::cdd_eValue,evalue);
    }
    else {
        cfg.params = "CMD=Put";
        addParametr(cfg.params,ReqParams::database,"nr");
        addParametr(cfg.params, ReqParams::program, cfg.dbChoosen);
        QString filter;
        QString wordSize;
        if(shortSeq) {
            evalue = 1000;
            filter = "";
            if(cfg.dbChoosen == "blastn") {
                addParametr(cfg.params, ReqParams::wordSize, 7);
            }
        }
        else {
            filter = "L";
        }
        addParametr(cfg.params, ReqParams::expect, evalue);
        addParametr(cfg.params, ReqParams::filter, filter);
        addParametr(cfg.params, ReqParams::hits, maxHits);
    }
    DNASequence seq = inputMessage.getData().value<DNASequence>();
    
    seq.info.clear();
    DNAAlphabet *alp = AppContext::getDNAAlphabetRegistry()->findAlphabet(seq.seq);
    if(seq.length()>MAX_BLAST_SEQ_LEN) {
        log.error(tr("Sequence too long"));
        return NULL;
    }
    if(alp == AppContext::getDNAAlphabetRegistry()->findById(BaseDNAAlphabetIds::AMINO_DEFAULT)) {
        if(cfg.dbChoosen == "blastn") {
            log.info(tr("Selected nucleotide database"));
            return NULL;
        }
    }
    else {
        if(cfg.dbChoosen != "blastn") {
            log.info(tr("Selected amino acid database"));
            return NULL;
        }
    }
    cfg.query = seq.seq;
    cfg.retries = 60;
    Task* t = new RemoteQueryTask(cfg);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

bool RemoteQueryWorker::isDone() {
    return !input || input->isEnded();
}

void RemoteQueryWorker::sl_taskFinished() {
    RemoteQueryTask * t = qobject_cast<RemoteQueryTask*>(sender());
    if (t->getState() != Task::State_Finished || t->hasErrors()) {
        return;
    }

    if(output) {
        if(actor->getParameter(DATABASE)->getAttributeValue<QString>() != "ncbi-cdd") {
            QString url = actor->getParameter(ORIGINAL_OUT)->getAttributeValue<QString>();
            if(!url.isEmpty()) {
                IOAdapterFactory * iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById( BaseIOAdapters::LOCAL_FILE );
                IOAdapter * io = iof->createIOAdapter();
                if(io->open( url, IOAdapterMode_Write )) {
                    QByteArray output = t->getOutputFile();
                    io->writeBlock(output);
                    io->close();
                }
            }
        }

        QList<SharedAnnotationData> res = t->getResultedAnnotations();
        QString annName = actor->getParameter(ANNOTATION_NAME)->getAttributeValue<QString>();
        if(!annName.isEmpty()) {
            for(int i = 0; i<res.count();i++) {
                res[i]->name = annName;
            }
        }
        QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(res);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
        if (input->isEnded()) {
            output->setEnded();
        }
    }
}

}
}
