/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <QtNetwork/QTcpSocket>
#include <core_api/AppContext.h>

#include "DirectSocketUtils.h"

#include "DirectSocketPlugin.h"
#include "DirectSocketRemoteMachine.h"

namespace GB2 {

// DirectSocketRemoteMachine

DirectSocketRemoteMachine::DirectSocketRemoteMachine(const DirectSocketRemoteMachineSettings &settings):
    settings(settings)
{
}

DirectSocketRemoteMachine::~DirectSocketRemoteMachine()
{
}

RemoteTaskError DirectSocketRemoteMachine::runRemoteTask( const QString & taskFactoryId, const QVariant & settings, qint64 * taskId )
{
    assert(NULL != taskId);
    QVariantList args;
    args << "runRemoteTask";
    args << taskFactoryId;
    args << settings;
    QVariant result;
    {
        RemoteTaskError error = sendRequest( args, &result);
        if(!error.getOk())
        {
            return error;
        }
    }
    if(!result.canConvert(QVariant::LongLong))
    {
        return RemoteTaskError(false, "invalid response");
    }
    *taskId = result.toLongLong();
    return RemoteTaskError(true, "");
}

RemoteTaskError DirectSocketRemoteMachine::cancelRemoteTask( qint64 taskId )
{
    QVariantList args;
    args << "cancelRemoteTask";
    args << taskId;
    return sendRequest(args);
}

RemoteTaskError DirectSocketRemoteMachine::deleteRemoteTask( qint64 taskId )
{
    QVariantList args;
    args << "deleteRemoteTask";
    args << taskId;
    return sendRequest(args);
}

RemoteTaskError DirectSocketRemoteMachine::getRemoteTaskCancelFlag( qint64 taskId, bool * cancelFlag )
{
    assert(NULL != cancelFlag);
    QVariantList args;
    args << "getRemoteTaskCancelFlag";
    args << taskId;
    QVariant result;
    {
        RemoteTaskError error = sendRequest(args, &result);
        if(!error.getOk())
        {
            return error;
        }
    }
    if(!result.canConvert(QVariant::Bool))
    {
        return RemoteTaskError(false, "invalid response");
    }
    *cancelFlag = result.toBool();
    return RemoteTaskError(true, "");
}

RemoteTaskError DirectSocketRemoteMachine::getRemoteTaskState( qint64 taskId, Task::State * state )
{
    assert(NULL != state);
    QVariantList args;
    args << "getRemoteTaskState";
    args << taskId;
    QVariant result;
    {
        RemoteTaskError error = sendRequest(args, &result);
        if(!error.getOk())
        {
            return error;
        }
    }
    if(!result.canConvert(QVariant::Int))
    {
        return RemoteTaskError(false, "invalid response");
    }
    *state = (Task::State)result.toInt();
    return RemoteTaskError(true, "");
}

RemoteTaskError DirectSocketRemoteMachine::getRemoteTaskProgress( qint64 taskId, int * progress )
{
    assert(NULL != progress);
    QVariantList args;
    args << "getRemoteTaskProgress";
    args << taskId;
    QVariant result;
    {
        RemoteTaskError error = sendRequest(args, &result);
        if(!error.getOk())
        {
            return error;
        }
    }
    if(!result.canConvert(QVariant::Int))
    {
        return RemoteTaskError(false, "invalid response");
    }
    *progress = result.toInt();
    return RemoteTaskError(true, "");
}

RemoteTaskError DirectSocketRemoteMachine::getRemoteTaskResult( qint64 taskId, QVariant * result )
{
    assert(NULL != result);
    QVariantList args;
    args << "getRemoteTaskResult";
    args << taskId;
    return sendRequest(args, result);
}

RemoteTaskError DirectSocketRemoteMachine::getRemoteTaskError( qint64 taskId, QString * errMsg )
{
    assert(NULL != errMsg);
    QVariantList args;
    args << "getRemoteTaskError";
    args << taskId;
    QVariant result;
    {
        RemoteTaskError error = sendRequest(args, &result);
        if(!error.getOk())
        {
            return error;
        }
    }
    if(!result.canConvert(QVariant::String))
    {
        return RemoteTaskError(false, "invalid response");
    }
    *errMsg = result.toString();
    return RemoteTaskError(true, "");
}

RemoteTaskError DirectSocketRemoteMachine::getUuid( QUuid * uuid )
{
    assert(NULL != uuid);
    QVariantList args;
    args << "getUuid";
    QVariant result;
    {
        RemoteTaskError error = sendRequest(args, &result);
        if(!error.getOk())
        {
            return error;
        }
    }
    if(!result.canConvert(QVariant::String))
    {
        return RemoteTaskError(false, "invalid response");
    }
    *uuid = QUuid(result.toString());
    return RemoteTaskError(true, "");
}

RemoteTaskError DirectSocketRemoteMachine::getServicesList( QStringList * list ) {
    assert( NULL != list );
    QVariantList args;
    args << "getServicesList";
    QVariant result;
    {
        RemoteTaskError error = sendRequest( args, &result );
        if( !error.getOk() )
        {
            return error;
        }
    }
    if( !result.canConvert( QVariant::StringList ) )
    {
        return RemoteTaskError( false, "invalid response" );
    }
    *list = result.toStringList();
    return RemoteTaskError( true, "" );
}

RemoteTaskError DirectSocketRemoteMachine::getHostName( QString * hostname ) {
    assert( NULL != hostname );
    QVariantList args;
    args << "getHostName";
    QVariant result;
    {
        RemoteTaskError error = sendRequest( args, &result );
        if( !error.getOk() )
        {
            return error;
        }
    }
    if( !result.canConvert( QVariant::String ) )
    {
        return RemoteTaskError( false, "invalid response" );
    }
    *hostname = result.toString();
    return RemoteTaskError( true, "" );
}

RemoteTaskError DirectSocketRemoteMachine::sendRequest(const QVariant & args, QVariant * result)
{
    QTcpSocket socket;
    socket.connectToHost(settings.getHost(), settings.getPort());
    if(!socket.waitForConnected())
    {
        return RemoteTaskError(false, "can't connect to remote host");
    }
    
    if( !DirectSocketUtils::writeToSocket( socket, args ) )
    {
        return RemoteTaskError(false, "can't send request");
    }
    
    QVariant response;
    if( !DirectSocketUtils::readFromSocket( socket, &response ) )
    {
        return RemoteTaskError(false, "can't receive response");
    }
    
    if(!response.canConvert(QVariant::List))
    {
        return RemoteTaskError(false, "invalid response");
    }
    
    QVariantList resultList = response.toList();
    if(2 != resultList.size())
    {
        return RemoteTaskError(false, "invalid response");
    }
    {
        RemoteTaskError error(false, "unknown error");
        if(!error.deserialize(resultList[0]))
        {
            return RemoteTaskError(false, "invalid response");
        }
        if(!error.getOk())
        {
            return error;
        }
    }
    if(NULL != result)
    {
        *result = resultList[1];
    }
    return RemoteTaskError(true, "");
}

// DirectSocketRemoteMachineSettings

DirectSocketRemoteMachineSettings::DirectSocketRemoteMachineSettings():
RemoteMachineSettings(AppContext::getProtocolInfoRegistry()->getProtocolInfo( DirectSocketPlugin::DIRECT_SOCKET_PROTOCOL_ID )),
host(""), port(0)
{
}

DirectSocketRemoteMachineSettings::DirectSocketRemoteMachineSettings(
        const QString &host, quint16 port):
RemoteMachineSettings(AppContext::getProtocolInfoRegistry()->getProtocolInfo( DirectSocketPlugin::DIRECT_SOCKET_PROTOCOL_ID ) ),
host(host), port(port)
{
}

QString DirectSocketRemoteMachineSettings::toString() const
{
    return host + ":" + QString::number( port );
}

QString DirectSocketRemoteMachineSettings::getHost()const
{
    return host;
}

quint16 DirectSocketRemoteMachineSettings::getPort()const
{
    return port;
}

bool DirectSocketRemoteMachineSettings::operator ==( const RemoteMachineSettings & m ) const {
    const RemoteMachineSettings * machine = &m;
    const DirectSocketRemoteMachineSettings* directSocketMachine = dynamic_cast< const DirectSocketRemoteMachineSettings* >( machine );
    if( NULL == directSocketMachine ) {
        return false;
    }
    return host == directSocketMachine->getHost() && port == directSocketMachine->getPort();
}

QString DirectSocketRemoteMachineSettings::serialize() const
{
    return toString();
}

bool DirectSocketRemoteMachineSettings::deserialize( const QString & data )
{
    QStringList args = data.split( ":", QString::SkipEmptyParts );
    if( 2 != args.size() ) {
        return false;
    }
    host = args.at( 0 );
    bool ok = false;
    port = args.at( 1 ).toUInt( &ok );
    if( !ok ) {
        return false;
    }
    return true;
}

// DirectSocketRemoteMachineFactory

DirectSocketRemoteMachineFactory::DirectSocketRemoteMachineFactory()
{
}

DirectSocketRemoteMachineFactory::~DirectSocketRemoteMachineFactory()
{
}

RemoteMachine * DirectSocketRemoteMachineFactory::createInstance( const QString & serializedSettings ) const
{
    DirectSocketRemoteMachineSettings settings;
    if(settings.deserialize(serializedSettings))
    {
        return createInstance(&settings);
    }
    return NULL;
}

RemoteMachine * DirectSocketRemoteMachineFactory::createInstance( RemoteMachineSettings * settings ) const
{
    DirectSocketRemoteMachineSettings *castedSettings = dynamic_cast<DirectSocketRemoteMachineSettings *>(settings);
    if(NULL != castedSettings)
    {
        return new DirectSocketRemoteMachine(*castedSettings);
    }
    return NULL;
}

RemoteMachineSettings * DirectSocketRemoteMachineFactory::createSettings( const QString & serializedSettings ) const {
    DirectSocketRemoteMachineSettings * ret = new DirectSocketRemoteMachineSettings();
    if( ret->deserialize( serializedSettings ) ) {
        return ret;
    }
    delete ret;
    return NULL;
}

} // namespace GB2
