/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <core_api/AppContext.h>
#include <core_api/SWResultFilterRegistry.h>
#include <core_api/DocumentModel.h>
#include <core_api/L10n.h>
#include <core_api/SmithWatermanTaskFactoryRegistry.h>

#include <distributed_computing/SerializeUtils.h>

#include "SmithWatermanLocalTask.h"

namespace GB2 {

class DNATranslation;

/*************************************************
* SmithWatermanLocalTaskSettings
*************************************************/

SmithWatermanLocalTaskSettings::SmithWatermanLocalTaskSettings() : doCleanup( false ) {
}

SmithWatermanLocalTaskSettings::~SmithWatermanLocalTaskSettings() {
    if( doCleanup ) {
        cleanup();
    }
}

void SmithWatermanLocalTaskSettings::cleanup() {
    assert( doCleanup );
    settings.pSm = SMatrix();
    delete settings.resultListener;
    settings.resultListener = NULL;
}

SmithWatermanLocalTaskSettings::SmithWatermanLocalTaskSettings( const SmithWatermanSettings & s ) : settings( s ), doCleanup( false ) {
}

QVariant SmithWatermanLocalTaskSettings::serialize() const {
    QVariantList res;
    res << SerializeUtils::serializeValue( settings.ptrn );
    res << SerializeUtils::serializeValue( settings.sqnc );
    res << SerializeUtils::serializeValue( settings.globalRegion.startPos );
    res << SerializeUtils::serializeValue( settings.globalRegion.len );
    res << SerializeUtils::serializeValue( (int)settings.strand );
    res << SerializeUtils::serializeValue( settings.percentOfScore );
    res << SerializeUtils::serializeValue( settings.gapModel.scoreGapOpen );
    res << SerializeUtils::serializeValue( settings.gapModel.scoreGapExtd );
    res << settings.pSm.toQVariant();
    res << SerializeUtils::serializeValue( ( NULL == settings.resultFilter ? QString( "" ) : settings.resultFilter->getId() ) );
    res << SerializeUtils::serializeValue( ( NULL == settings.complTT ? QString( "" ) : settings.complTT->getTranslationId() ) );
    res << SerializeUtils::serializeValue( ( NULL == settings.aminoTT ? QString( "" ) : settings.aminoTT->getTranslationId() ) );
    return res;
}

bool SmithWatermanLocalTaskSettings::deserialize( const QVariant & data ) {
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    if( SERIALIZED_LIST_SZ != args.size() ) {
        return false;
    }
    
    if( !SerializeUtils::deserializeValue( args[0], &settings.ptrn ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[1], &settings.sqnc ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[2], &settings.globalRegion.startPos ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[3], &settings.globalRegion.len ) ) { return false; }
    {
        int tmp = 0;
        if( !SerializeUtils::deserializeValue( args[4], &tmp ) ) { return false; }
        settings.strand = (StrandOption)tmp;
    }
    if( !SerializeUtils::deserializeValue( args[5], &settings.percentOfScore ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[6], &settings.gapModel.scoreGapOpen ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[7], &settings.gapModel.scoreGapExtd ) ) { return false; }
    
    settings.pSm = SMatrix::fromQVariant( args[8] );
    
    QString resultFilterId;
    if( !SerializeUtils::deserializeValue( args[9], &resultFilterId ) ) { return false; }
    settings.resultFilter = AppContext::getSWResultFilterRegistry()->getFilter( resultFilterId );
    settings.resultListener = new SmithWatermanResultListener();
    
    QString translId;
    if( !SerializeUtils::deserializeValue( args[10], &translId ) ) { return false; }
    settings.complTT = AppContext::getDNATranslationRegistry()->lookupTranslation( translId );
    if( !SerializeUtils::deserializeValue( args[11], &translId ) ) { return false; }
    settings.aminoTT = AppContext::getDNATranslationRegistry()->lookupTranslation( translId );
    
    doCleanup = true;
    return true;
}

const SmithWatermanSettings & SmithWatermanLocalTaskSettings::getSWSettings() const {
    return settings;
}

/*************************************************
* SmithWatermanLocalTaskResult
*************************************************/

SmithWatermanLocalTaskResult::SmithWatermanLocalTaskResult() {
}

SmithWatermanLocalTaskResult::~SmithWatermanLocalTaskResult() {
}

QVariant SmithWatermanLocalTaskResult::serialize() const {
    QVariantList res;
    foreach( const SmithWatermanResult & swr, domains ) {
        QVariantList domainRes;
        domainRes << SerializeUtils::serializeValue( swr.complement );
        domainRes << SerializeUtils::serializeValue( swr.trans );
        domainRes << SerializeUtils::serializeValue( swr.score );
        domainRes << SerializeUtils::serializeValue( swr.region.startPos );
        domainRes << SerializeUtils::serializeValue( swr.region.len );
        res << QVariant( domainRes );
    }
    return res;
}

bool SmithWatermanLocalTaskResult::deserialize( const QVariant & data ) {
    assert( domains.isEmpty() );
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    
    foreach( const QVariant & arg, args ) {
        if( !arg.canConvert( QVariant::List ) ) {
            domains.clear();
            return false;
        }
        QVariantList argList = arg.toList();
        if( 5 != argList.size() ) {
            domains.clear();
            return false;
        }
        SmithWatermanResult swr;
        if( !SerializeUtils::deserializeValue( argList[0], &swr.complement ) ) { domains.clear(); return false; }
        if( !SerializeUtils::deserializeValue( argList[1], &swr.trans ) ) { domains.clear(); return false; }
        if( !SerializeUtils::deserializeValue( argList[2], &swr.score ) ) { domains.clear(); return false; }
        if( !SerializeUtils::deserializeValue( argList[3], &swr.region.startPos ) ) { domains.clear(); return false; }
        if( !SerializeUtils::deserializeValue( argList[4], &swr.region.len ) ) { domains.clear(); return false; }
        domains << swr;
    }
    return true;
}

QList< SmithWatermanResult > SmithWatermanLocalTaskResult::getResult() const {
    return domains;
}

void SmithWatermanLocalTaskResult::setResult( const QList< SmithWatermanResult > & d ) {
    domains = d;
}

/*************************************************
* SmithWatermanLocalTask
*************************************************/

SmithWatermanLocalTask::SmithWatermanLocalTask( SmithWatermanLocalTaskSettings * s ) 
: LocalTask( "", TaskFlags( TaskFlag_NoRun ) ), settings( s ), swTask( NULL ) {
    
    setTaskName( tr( "Smith-Waterman local task" ) );
    if( NULL == settings ) {
        setError( tr( "No settings given" ) );
        return;
    }
}

SmithWatermanLocalTask::~SmithWatermanLocalTask() {
    if( NULL != settings ) {
        delete settings;
        settings = NULL;
    }
}

void SmithWatermanLocalTask::prepare() {
    if( hasErrors() ) {
        return;
    }
    assert( NULL != settings );
    SmithWatermanTaskFactoryRegistry * swFactoryRegistry = AppContext::getSmithWatermanTaskFactoryRegistry();
    assert( NULL != swFactoryRegistry );
    QStringList swFactories = swFactoryRegistry->getListFactoryNames();
    if( swFactories.isEmpty() ) {
        setError( tr( "No Smith-Waterman realization found" ) );
        return;
    }
    SmithWatermanTaskFactory * swFactory = swFactoryRegistry->getFactory( swFactories.last() );
    assert( NULL != swFactory );
    
    swTask = swFactory->getTaskInstance( settings->getSWSettings(), tr( "Smith-Waterman local task" ) );
    addSubTask( swTask );
}

Task::ReportResult SmithWatermanLocalTask::report() {
    if( hasErrors() ) {
        return ReportResult_Finished;
    }
    if( swTask->hasErrors() ) {
        setError( swTask->getError() );
        return ReportResult_Finished;
    }
    result.setResult( settings->getSWSettings().resultListener->popResults() );
    return ReportResult_Finished;
}

const LocalTaskResult * SmithWatermanLocalTask::getResult() const {
    return &result;
}

/*************************************************
* SmithWatermanRemoteToAnnotationsTask
*************************************************/

SmithWatermanRemoteToAnnotationsTask::SmithWatermanRemoteToAnnotationsTask( RemoteMachineSettings * s, 
                                                                            const SmithWatermanSettings & c, 
                                                                            AnnotationTableObject * o, const QString & an, 
                                                                            const QString & ag ) 
: Task( tr( "Smith-Waterman task running on remote machine" ), TaskFlags_NR_FOSCOE 
       | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled ),
machineSettings( s ), config( c ), annotationObj( o ), aname( an ), agroup( ag ), machine( NULL ),
swTask( NULL ), reportCallback( NULL ) {
    
    checkArgs();
    if( hasErrors() ) {
        return;
    }
    machine = AppContext::getProtocolInfoRegistry()->getProtocolInfo( machineSettings->getProtocolId() )
        ->getRemoteMachineFactory()->createInstance( machineSettings );
    if( NULL == machine ) {
        setError( tr( "Cannot create remote machine from settings" ) );
        return;
    }
    SmithWatermanLocalTaskSettings localSettings( config );
    
    swTask = new RemoteTask( SmithWatermanLocalTaskFactory::ID, localSettings, machine );
    addSubTask( swTask );
}

void SmithWatermanRemoteToAnnotationsTask::checkArgs() {
    if( NULL == machineSettings ) {
        setError( L10N::badArgument( tr( "remote machine settings" ) ) );
        return;
    }
    if( NULL == annotationObj.data() ) {
        setError( L10N::badArgument( tr( "annotation object" ) ) );
        return;
    }
    if( aname.isEmpty() ) {
        setError( L10N::badArgument( tr( "annotation name" ) ) );
        return;
    }
    if( agroup.isEmpty() ) {
        setError( L10N::badArgument( tr( "annotation group" ) ) );
        return;
    }
}

SmithWatermanRemoteToAnnotationsTask::~SmithWatermanRemoteToAnnotationsTask() {
    if( NULL != machine ) {
        delete machine;
        machine = NULL;
    }
    if( NULL != reportCallback ) {
        delete reportCallback;
        reportCallback = NULL;
    }
}

Task::ReportResult SmithWatermanRemoteToAnnotationsTask::report() {
    propagateSubtaskError();
    if( hasErrors() || isCanceled() ) {
        return ReportResult_Finished;
    }
    if( annotationObj.isNull() ) {
        setError( tr( "Annotation object removed" ) );
        return ReportResult_Finished;
    }
    
    SmithWatermanLocalTaskResult * result = dynamic_cast< SmithWatermanLocalTaskResult* >( swTask->getResult() );
    if( NULL == result ) {
        setError( tr( "Remote task didn't produced result" ) );
        return ReportResult_Finished;
    }
    
    reportCallback = new SmithWatermanReportCallbackImpl( annotationObj.data(), aname, agroup );
    reportCallback->report( result->getResult() );
    
    return ReportResult_Finished;
}

QString SmithWatermanRemoteToAnnotationsTask::generateReport() const {
    QString res;
    res += "<table>";
    res+="<tr><td width=200><b>" + tr("Smith-Waterman task runned on remote machine.") + "</b></td><td>" + 
        machineSettings->toString() + "</td></tr>";

    if( hasErrors() || isCanceled() ) {
        res += "<tr><td width=200><b>" + tr("Task was not finished") + "</b></td><td></td></tr>";
        res += "</table>";
        return res;
    }
    
    res += "<tr><td><b>" + tr("Result annotation table") + "</b></td><td>" + annotationObj->getDocument()->getName() + "</td></tr>";
    res += "<tr><td><b>" + tr("Result annotation group") + "</b></td><td>" + agroup + "</td></tr>";
    res += "<tr><td><b>" + tr("Result annotation name") +  "</b></td><td>" + aname + "</td></tr>";
    
    SmithWatermanLocalTaskResult * result = dynamic_cast< SmithWatermanLocalTaskResult* >( swTask->getResult() );
    int nResults =  NULL == result? 0 : result->getResult().size();
    res += "<tr><td><b>" + tr("Results count") +  "</b></td><td>" + QString::number( nResults )+ "</td></tr>";
    res += "</table>";
    return res;
}

/*************************************************
* SmithWatermanLocalTaskFactory
*************************************************/

template<> const QString SmithWatermanLocalTaskFactory::ID = "Smith-Waterman search task";

} // GB2
