/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GraphicsRectangularBranchItem.h"
#include "GraphicsButtonItem.h"
#include "TreeViewerUtils.h"

#include <QtGui/QPainter>
#include <QtGui/QPen>
#include <QtCore/QStack>
#include <QtGui/QGraphicsScene>
#include <QtGui/QGraphicsView>
#include <gobjects/PhyTreeObject.h>
#include <core_api/AppContext.h>

namespace GB2 {

const qreal GraphicsRectangularBranchItem::DEFAULT_WIDTH = 25.0;
const qreal GraphicsRectangularBranchItem::MAXIMUM_WIDTH = 500.0;
const int GraphicsRectangularBranchItem::DEFAULT_HEIGHT = 25;

void GraphicsRectangularBranchItem::collapse() {
    collapsed = !collapsed;
    QList<QGraphicsItem*> items = childItems();
    if (collapsed) {
        int xmin = 0;
        qreal ymin = items[0]->pos().y(), ymax = 0;
        for (int i = 0, s = items.size(); i < s; ++i) {
            if (!dynamic_cast<GraphicsRectangularBranchItem*>(items[i]))
                continue;
            QPointF pos1 = items[i]->pos();
            if (pos1.x() < xmin || xmin == 0)
                xmin = pos1.x();
            if (pos1.y() < ymin)
                ymin = pos1.y();
            if (pos1.y() > ymax)
                ymax = pos1.y();
            if (items[i] != getDistanceText() && items[i] != getNameText()) {
                items[i]->hide();
            }
        }
        if (xmin >= 2 * GraphicsRectangularBranchItem::DEFAULT_WIDTH)
            xmin /= 2;
        if (xmin < GraphicsRectangularBranchItem::DEFAULT_WIDTH)
            xmin = GraphicsRectangularBranchItem::DEFAULT_WIDTH;

        QPen pen1(QColor(0, 0, 0));
        pen1.setWidth(SELECTED_PEN_WIDTH);
        pen1.setCosmetic(true);
        const int defHeight = qMin((int)(ymax - ymin) / 2, 30);
        QGraphicsRectItem *r = new QGraphicsRectItem(0, -defHeight / 2, xmin, defHeight, this);
        r->setPen(pen1);
    } else {
        for (int i = 0, s = items.size(); i < s; ++i) {
            if (dynamic_cast<QGraphicsRectItem*>(items[i])) {
                items[i]->setParentItem(NULL);
                scene()->removeItem(items[i]);
            } else {
                if (items[i] != getDistanceText() && items[i] != getNameText()) {
                    items[i]->show();
                }
            }
        }
    }
}

GraphicsRectangularBranchItem::GraphicsRectangularBranchItem(const QString& name, GraphicsRectangularBranchItem* pitem)
: GraphicsBranchItem(name), direction(GraphicsRectangularBranchItem::up) {
    setParentItem(pitem);
    setPos(0, 0);
    height = 0;
}

GraphicsRectangularBranchItem::GraphicsRectangularBranchItem(qreal x, qreal y, const QString& name)
: GraphicsBranchItem(false), direction(GraphicsRectangularBranchItem::up) {
    new GraphicsRectangularBranchItem(name, this);
    setPos(x, y);
}

GraphicsRectangularBranchItem::GraphicsRectangularBranchItem(qreal x, qreal y, const QString& name, qreal d)
: GraphicsBranchItem(d, false), direction(GraphicsRectangularBranchItem::up) {
    new GraphicsRectangularBranchItem(name, this);
    setPos(x, y);
}

GraphicsRectangularBranchItem::GraphicsRectangularBranchItem(qreal d)
: GraphicsBranchItem(d), direction(GraphicsRectangularBranchItem::up) {}

GraphicsRectangularBranchItem::GraphicsRectangularBranchItem()
: direction(GraphicsRectangularBranchItem::up) {}

void GraphicsRectangularBranchItem::setParentItem(QGraphicsItem *item) {
    prepareGeometryChange();
    height = direction == up ? pos().y() - item->pos().y() : item->pos().y() - pos().y();
    setPos(width, direction == up ? height : -height);
    QAbstractGraphicsShapeItem::setParentItem(item);
}

void GraphicsRectangularBranchItem::setDirection(Direction d) {
    prepareGeometryChange();
    direction = d;
}

QPainterPath GraphicsRectangularBranchItem::shape() const {
    QPainterPath path;
    return path;
}

QRectF GraphicsRectangularBranchItem::boundingRect() const {
    return QRectF(- width - 0.5, direction == up ? -height : -0.5, width + 0.5, height + 0.5);
}

void GraphicsRectangularBranchItem::paint(QPainter *painter, const QStyleOptionGraphicsItem*, QWidget*) {
    painter->setPen(pen());
    painter->drawLine(QPointF(0, 0), QPointF(-width, 0));
    painter->drawLine(QPointF(-width, 0), QPointF(-width, direction == up ? -height : height));
}

void GraphicsRectangularBranchItem::setHeight(qreal h) {
    if (height == h)
        return;

    if (direction == up)
        setPos(pos().x(), pos().y() - height + h);

    prepareGeometryChange();
    height = h;
}

}//namespace
