/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_ANNOTATIONS_TREE_VIEW_
#define _GB2_ANNOTATIONS_TREE_VIEW_

#include <core_api/core_api.h>

#include <QtCore/QFlags>
#include <QtCore/QTimer>
#include <QtGui/QTreeWidget>
#include <QtGui/QCloseEvent>
#include <QtGui/QLabel>


namespace GB2 {

class Annotation;
class AnnotationGroup;
class Qualifier;
class AnnotationTableObject;
class AVItem;
class AVGroupItem;
class AVAnnotationItem;
class AVQualifierItem;
class AnnotatedDNAView;
class AnnotationSelection;
class AnnotationGroupSelection;
class AnnotationModification;
class GObjectView;

enum ATVAnnUpdateFlag {
    ATVAnnUpdateFlag_BaseColumns = 0x1,
    ATVAnnUpdateFlag_QualColumns = 0x2
};

typedef QFlags<ATVAnnUpdateFlag> ATVAnnUpdateFlags;


class GB2_COREAPI_EXPORT AnnotationsTreeView : public QWidget {
    Q_OBJECT
public:
    AnnotationsTreeView(AnnotatedDNAView* ctx);

    void saveWidgetState();
    void restoreWidgetState();

    void adjustStaticMenu(QMenu *m) const {adjustMenu(m);}

    QTreeWidget* getTreeWidget() const {return tree;}
    
    QStringList getQualifierColumnNames() const {return qColumns;}
    void addQualifierColumn(const QString& q);
    void removeQualifierColumn(const QString& q);

    void saveState(QVariantMap& map) const;
    void updateState(const QVariantMap& map);

    void setSortingEnabled(bool v);

    AVItem* currentItem();

private slots:

    void sl_onAnnotationObjectAdded(AnnotationTableObject* obj);
    void sl_onAnnotationObjectRemoved(AnnotationTableObject* obj);

    void sl_onAnnotationsAdded(const QList<Annotation*>&);
    void sl_onAnnotationsRemoved(const QList<Annotation*>&);
    void sl_onAnnotationModified(const AnnotationModification& md);
    void sl_annotationObjectModifiedStateChanged();

    void sl_onGroupCreated(AnnotationGroup*);
    void sl_onGroupRemoved(AnnotationGroup* parent, AnnotationGroup* removed);
    void sl_onGroupRenamed(AnnotationGroup*, const QString& oldName);

    void sl_onAnnotationSettingsChanged(const QStringList& changedSettings);

    void sl_onAnnotationSelectionChanged(AnnotationSelection*, const QList<Annotation*>&, const QList<Annotation*>&);
    void sl_onAnnotationGroupSelectionChanged(AnnotationGroupSelection*, const QList<AnnotationGroup*>&, const QList<AnnotationGroup*>&);
    void sl_onItemSelectionChanged();
    void sl_onAddAnnotationObjectToView();
    void sl_removeObjectFromView();
    void sl_removeAnnsAndQs();
    void sl_onBuildPopupMenu(GObjectView* thiz, QMenu* menu);
    void sl_onCopyQualifierValue();
    void sl_onCopyQualifierURL();
    void sl_onToggleQualifierColumn();
    void sl_onRemoveColumnByHeaderClick();
    void sl_onCopyColumnText();
    void sl_onCopyColumnURL();
    
    void sl_rename();
    void sl_edit();
    void sl_addQualifier();
//    void sl_cutAnnotations();
//    void sl_copyAnnotations();
//    void sl_pasteAnnotations();

    void sl_itemEntered(QTreeWidgetItem * i, int column);
    void sl_itemClicked( QTreeWidgetItem * item, int column);
    void sl_itemDoubleClicked (QTreeWidgetItem * item, int column);
    void sl_itemExpanded(QTreeWidgetItem*);
    
    //TODO: deal with style()->styleHint(QStyle::SH_ItemView_ActivateItemOnSingleClick) correctly
    //    void sl_itemActivated(QTreeWidgetItem*, int) {sl_edit();}
    void sl_sortTree();

protected:
    bool eventFilter(QObject* o, QEvent* e);

private:
    void renameItem(AVItem* i);
    void editItem(AVItem* i);
    
    QString renameDialogHelper(AVItem* i, const QString& defText, const QString& title);
    bool editQualifierDialogHelper(AVQualifierItem* i, bool ro, Qualifier& res);
    void moveDialogToItem(QTreeWidgetItem* item, QDialog& d);
    
    void adjustMenu(QMenu* m_) const;
    AVGroupItem* buildGroupTree(AVGroupItem* parentGroup, AnnotationGroup* g);
    AVAnnotationItem* buildAnnotationTree(AVGroupItem* parentGroup, Annotation* a);
    void populateAnnotationQualifiers(AVAnnotationItem* ai);
    void updateAllAnnotations(ATVAnnUpdateFlags flags);

    AVGroupItem* findGroupItem(const AnnotationGroup* g) const;
    AVAnnotationItem* findAnnotationItem(const AnnotationGroup* g, const Annotation* a) const;
    AVAnnotationItem* findAnnotationItem(const AVGroupItem* gi, const Annotation* a) const;
    QList<AVAnnotationItem*> findAnnotationItems(const Annotation* a) const;

    void connectAnnotationSelection();
    void connectAnnotationGroupSelection();
    
    void updateState();
    void updateColumnContextActions(AVItem* item, int col);

    QTreeWidget* tree;

    AnnotatedDNAView*   ctx;
    QAction*            addAnnotationObjectAction;
    QAction*            removeObjectsFromViewAction;
    QAction*            removeAnnsAndQsAction;
    QAction*            copyQualifierAction;
    QAction*            copyQualifierURLAction;
    QAction*            toggleQualifierColumnAction;
    QAction*            removeColumnByHeaderClickAction;
    QAction*            copyColumnTextAction;
    QAction*            copyColumnURLAction;
//    QAction*            cutAnnotationsAction;
//    QAction*            copyAnnotationsAction;
//    QAction*            pasteAnnotationsAction;
    
    QAction*            renameAction;       // action to rename active group/qualifier/annotation only
    QAction*            editAction;         // action to edit active item -> only for non-readonly
    QAction*            viewAction;         // action to view active item -> could be used both for readonly and not readonly
    QAction*            addQualifierAction; // action to create qualifier. Editable annotation or editable qualifier must be selected
    
    Qt::MouseButton     lastMB;
    QStringList         headerLabels;
    QStringList         qColumns;
    int                 lastClickedColumn;
    QIcon               addColumnIcon;
    QIcon               removeColumnIcon;
    QTimer              sortTimer;
    QPoint              dragStartPos;
    bool                isDragging;

    static const QString annotationMimeType;
};

//////////////////////////////////////////////////////////////////////////
/// Tree model

//TODO: create qualifiers subtrees only when qualifier node is opened (usually qualifiers get ~ 90% of memory)

enum AVItemType {
    AVItemType_Group,
    AVItemType_Annotation,
    AVItemType_Qualifier
};

class AVItem : public QTreeWidgetItem {
public:
    AVItem(QTreeWidgetItem* parent, AVItemType _type) : QTreeWidgetItem(parent), type(_type) {}
    const AVItemType type;
    bool processLinks(const QString& qname, const QString& qval, int col);
    bool isColumnLinked(int col) const;
    QString buildLinkURL(int col) const;
    QString getFileUrl(int col) const;
    virtual bool isReadonly() const  {assert(parent()!=NULL); return (static_cast<AVItem*>(parent())->isReadonly());}

    virtual AnnotationsTreeView* getAnnotationTreeView() const {assert(parent()!=NULL); return (static_cast<AVItem*>(parent())->getAnnotationTreeView());}
    virtual AnnotationTableObject* getAnnotationTableObject() const {assert(parent()!=NULL); return (static_cast<AVItem*>(parent())->getAnnotationTableObject());}
    virtual AnnotationGroup* getAnnotationGroup() const {assert(parent()!=NULL); return (static_cast<AVItem*>(parent())->getAnnotationGroup());}

};


class AVGroupItem : public AVItem {
public:
	AVGroupItem(AnnotationsTreeView* atv, AVGroupItem* parent, AnnotationGroup* g);
    ~AVGroupItem();
	
    void updateVisual();
    void updateAnnotations(const QString& nameFilter, ATVAnnUpdateFlags flags);

    static const QIcon& getGroupIcon();
    static const QIcon& getDocumentIcon();
    virtual AnnotationsTreeView* getAnnotationTreeView() const  {return atv;}
    virtual bool isReadonly() const;
    virtual AnnotationTableObject* getAnnotationTableObject() const;
    virtual AnnotationGroup* getAnnotationGroup() const;

    AnnotationGroup* group;	
    AnnotationsTreeView* atv;
};

class AVAnnotationItem : public AVItem {
public:
    AVAnnotationItem(AVGroupItem* parent, Annotation* a);
    ~AVAnnotationItem();
    Annotation* annotation;
    mutable QString locationString;

    virtual QVariant data ( int column, int role ) const;
    void updateVisual(ATVAnnUpdateFlags flags);
    virtual bool operator< ( const QTreeWidgetItem & other ) const;
    bool isColumnNumeric(int col) const;
    double getNumericVal(int col) const;
    
    void removeQualifier(const Qualifier& q);
    void addQualifier(const Qualifier& q);
    AVQualifierItem* findQualifierItem(const QString& name, const QString& val) const;
    
    static QMap<QString, QIcon>& getIconsCache();
    bool hasNumericQColumns;
};

class AVQualifierItem: public AVItem {
public:
    AVQualifierItem(AVAnnotationItem* parent, const Qualifier& q);

    const QString qName;
    const QString qValue;
};


}//namespace


#endif
