/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_PROJECT_TREE_CONTROLLER_H_
#define _GB2_PROJECT_TREE_CONTROLLER_H_

#include <core_api/core_api.h>
#include <selection/DocumentSelection.h>
#include <selection/GObjectSelection.h>

#include <core_api/AppContext.h>

#include <core_api/GObjectReference.h>

#include <QtCore/QPointer>
#include <QtGui/QTreeWidget>
#include <QtGui/QIcon>


namespace GB2 {

class Task;
class GObjectConstraints;
class ProjViewItem;
class ProjViewDocumentItem;	
class ProjViewTypeItem;
class ProjViewObjectItem;

class GB2_COREAPI_EXPORT LoadDocumentTaskProvider {
public:
    virtual ~LoadDocumentTaskProvider(){};
    virtual Task* createLoadDocumentTask(Document* doc) const = 0;
};

enum ProjectTreeGroupMode {
    ProjectTreeGroupMode_Flat,         //objects are not grouped, only unloaded documents are shown
    ProjectTreeGroupMode_ByDocument,   //objects are grouped by document
    ProjectTreeGroupMode_ByType,       //objects are grouped by type
    
    ProjectTreeGroupMode_Min = ProjectTreeGroupMode_Flat,
    ProjectTreeGroupMode_Max = ProjectTreeGroupMode_ByType
};

//filtered objects and documents are not shown in project tree
class PTCObjectFilter : public QObject{
public:
    PTCObjectFilter(QObject* p = NULL) : QObject(p){}
    virtual bool filter(GObject* o) const = 0;
};

class PTCDocumentFilter : public QObject {
public:
    PTCDocumentFilter(QObject* p): QObject(p) {}
    virtual bool filter(Document* d) const = 0;
};


class PTCObjectRelationFilter : public PTCObjectFilter {
public:
    PTCObjectRelationFilter(const GObjectRelation& _rel, QObject* p = NULL) : PTCObjectFilter(p), rel(_rel){}
    bool filter(GObject* o) const;
    GObjectRelation rel;
};



class GB2_COREAPI_EXPORT ProjectTreeControllerModeSettings {
public:
    ProjectTreeControllerModeSettings() 
        : allowMultipleSelection(true), readOnlyFilter(TriState_Unknown), loadTaskProvider(NULL), 
        groupMode(ProjectTreeGroupMode_ByDocument), objectFilter(NULL), documentFilter(NULL){}
    
    QList<GObjectType>          objectTypesToShow;  // show only objects of specified type
    QList<GObjectConstraints*>  objectConstraints;  // show only objects that fits constraints
    QList<QPointer<GObject> >   excludeObjectList;  // do not show these objects
    QStringList                 tokensToShow;       // show documents/objects with all of tokens in a name
    bool                        allowMultipleSelection; //use multiple selection in tree
    TriState                    readOnlyFilter;     // unknown->all, true->filter(exclude) readonly, false -> keep only readonly
    LoadDocumentTaskProvider*   loadTaskProvider;   // use custom LoadDocumentTask factory instead of default
    ProjectTreeGroupMode        groupMode;          // group mode for objects

    //Note that objectFilter and documentFilter are called only on object add/remove ops!
    //WARN: object and document filters live-range is controlled by the side created these objects
    PTCObjectFilter*            objectFilter;       
    PTCDocumentFilter*          documentFilter;
    
    bool isDocumentShown(Document* doc) const;
    bool isTypeShown(GObjectType t) const;
    bool isObjectShown(GObject* o) const ;
};

class GB2_COREAPI_EXPORT ProjectTreeController : public QObject {
	Q_OBJECT
public:
	ProjectTreeController(QObject* parent, QTreeWidget* tree, const ProjectTreeControllerModeSettings& mode);
	
	const DocumentSelection* getDocumentSelection() const {return &documentSelection;}
	const GObjectSelection* getGObjectSelection() const {return &objectSelection;}

    void updateSettings(const ProjectTreeControllerModeSettings& mode);

    QAction* getGroupByDocumentAction() const {return groupByDocumentAction;}
    QAction* getGroupByTypeAction() const {return groupByTypeAction;}
    QAction* getGroupFlatAction() const {return groupFlatAction;}
    QAction* getLoadSeletectedDocumentsAction() { return loadSelectedDocumentsAction; }
    const ProjectTreeControllerModeSettings& getModeSettings() const {return mode;}

    void highlightItem(Document*);

signals:
	void si_onPopupMenuRequested(QMenu& popup);
	void si_doubleClicked(GObject*);

private slots:
	void sl_onTreeSelectionChanged();
	void sl_onContextMenuRequested(const QPoint & pos);
	void sl_onRemoveSelectedDocuments();
	void sl_onLoadSelectedDocuments();
    void sl_onUnloadSelectedDocuments();

	void sl_onDocumentAddedToProject(Document* d);
	void sl_onDocumentRemovedFromProject(Document* d);
	void sl_onDocumentModifiedStateChanged();
	void sl_onDocumentLoadedStateChanged();
    void sl_onDocumentURLorNameChanged();
    
	void sl_onObjectAdded(GObject* o);
	void sl_onObjectRemoved(GObject* o);
	void sl_onObjectModifiedStateChanged();

	void sl_onItemDoubleClicked(QTreeWidgetItem * item, int column);

    void sl_onResourceUserRegistered(const QString& res, Task* t);
    void sl_onResourceUserUnregistered(const QString& res, Task* t);
    void sl_onLoadingDocumentProgressChanged();
    void sl_onToggleReadonly();
    void sl_lockedStateChanged();
    
    void sl_onGroupByDocument();
    void sl_onGroupByType();
    void sl_onGroupFlat();

private:
	void updateActions();
	void updateSelection();
	void connectModel();
	void connectDocument(Document* d);
	void disconnectDocument(Document* d);
    void connectToResourceTracker();
	void connectGObject(GObject* d);
	void buildTree();
	void buildDocumentTree(Document* d);
    void flattenDocumentItem(ProjViewDocumentItem* docItem);
    void runLoadDocumentTask(Document* d);
    GObjectType getLoadedObjectType(GObject* obj) const;
    void filterItemsRecursive(ProjViewItem* pi);
    QSet<Document*>  getDocsInSelection(bool deriveFromObjects);
    void insertTreeItemSorted(ProjViewItem* p, ProjViewItem* item);
    void updateLoadingState(Document* d);


    ProjViewDocumentItem* findDocumentItem(Document* d) const;
    ProjViewDocumentItem* findDocumentItem(Document* d, bool create);
    ProjViewObjectItem* findGObjectItem(Document* d, GObject* o) const;
    ProjViewObjectItem* findGObjectItem(ProjViewItem* pi, GObject* o) const;
    ProjViewTypeItem*   findTypeItem(const GObjectType& t) const;
    ProjViewTypeItem*   findTypeItem(const GObjectType& t, bool create);


	QTreeWidget* tree;

	QAction* removeSelectedDocumentsAction;
	QAction* loadSelectedDocumentsAction;
    QAction* unloadSelectedDocumentsAction;
    QAction* addReadonlyFlagAction;
    QAction* removeReadonlyFlagAction;
    
    QAction* groupByDocumentAction;
    QAction* groupByTypeAction;
    QAction* groupFlatAction;

	GObjectSelection  objectSelection;
	DocumentSelection documentSelection;
    ProjectTreeControllerModeSettings mode;
    QSet<ProjViewItem*> itemsToUpdate;

public: 
	QIcon documentIcon;
    QIcon roDocumentIcon;
};

class GB2_COREAPI_EXPORT ProjViewItem : public QTreeWidgetItem {
public:
	ProjViewItem(ProjectTreeController* c) :  controller(c) {}
	bool isRootItem() {return parent() == NULL;}
	virtual bool isDocumentItem() const {return false;}
    virtual bool isTypeItem() const {return false;}
	virtual bool isObjectItem() const {return false;}
	virtual void updateVisual(bool recursive = false) = 0;
	ProjectTreeController* controller;
};

class GB2_COREAPI_EXPORT ProjViewTypeItem : public ProjViewItem {
public:
    ProjViewTypeItem(const GObjectType& t, ProjectTreeController* c);
    virtual bool isTypeItem() const {return true;}
    virtual bool operator< ( const QTreeWidgetItem & other ) const;
    virtual void updateVisual(bool recursive = false);
    GObjectType otype;
    QString     typePName;
};


class GB2_COREAPI_EXPORT ProjViewDocumentItem : public ProjViewItem {
public:
	ProjViewDocumentItem(Document* _doc, ProjectTreeController* c);
	virtual bool isDocumentItem() const {return true;}
	virtual void updateVisual(bool recursive = false);
    virtual bool operator< ( const QTreeWidgetItem & other ) const;
	Document* doc;
};

class GB2_COREAPI_EXPORT ProjViewObjectItem : public ProjViewItem {
public:
	ProjViewObjectItem(GObject* _obj, ProjectTreeController* c);
	virtual bool isObjectItem() const {return true;}
	virtual void updateVisual(bool recursive = false);
    virtual bool operator< ( const QTreeWidgetItem & other ) const;
	GObject* obj;
};


}//namespace
#endif
