/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <cassert>
#include <memory>

#include <QtGui/QMenu>
#include <QtGui/QCheckBox>

#include <core_services/AppContextImpl.h>
#include <core_api/MainWindow.h>
#include <core_api/NetworkConfiguration.h>
#include <core_api/AppSettings.h>
#include <core_api/Log.h>
#include <util_algorithm/SynchHttp.h>
#include <distributed_computing/SerializeUtils.h>

#include "DistributedComputingUtil.h"

#define RETRIEVE_MACHINE_INFO_LOG_CAT "Retrieve machine info"

namespace GB2 {

static LogCategory log( RETRIEVE_MACHINE_INFO_LOG_CAT );

/*******************************************
 * DistributedComputingUtil
 *******************************************/

DistributedComputingUtil::DistributedComputingUtil() {
    AppContextImpl * appContext = AppContextImpl::getApplicationContext();
    assert( NULL != appContext );
    
    ltfr = new LocalTaskFactoryRegistry();
    appContext->setLocalTaskFactoryRegistry( ltfr );
    ltm = new LocalTaskManager();
    appContext->setLocalTaskManager( ltm );
    pir = new ProtocolInfoRegistry();
    appContext->setProtocolInfoRegistry( pir );
    rmm = new RemoteMachineMonitor();
    appContext->setRemoteMachineMonitor( rmm );
    
    ltfr->registerLocalTaskFactory( &pingFactory );
    ltfr->registerLocalTaskFactory( &swFactory );
    
    if( NULL != AppContext::getMainWindow() ) { /* if not congene */
        QAction * showRemoteMachinesMonitor = new QAction( QIcon( ":core/images/remote_machine_monitor.png" ),
                                                           tr( "Remote machines monitor" ), this );
        connect( showRemoteMachinesMonitor, SIGNAL( triggered() ), SLOT( sl_showRemoteMachinesMonitor() ) );
        AppContext::getMainWindow()->getMenuManager()->getTopLevelMenu( MWMENU_SETTINGS )->addAction( showRemoteMachinesMonitor );
    }
}

DistributedComputingUtil::~DistributedComputingUtil() {
    delete ltfr;
    delete ltm;
    delete rmm;
    delete pir;
}

void DistributedComputingUtil::sl_showRemoteMachinesMonitor() {
    RemoteMachineMonitorDialogImpl dlg( QApplication::activeWindow(), rmm->getRemoteMachineMonitorItems() );
    int ret = dlg.exec();
    if( QDialog::Rejected == ret ) {
        return;
    }
    assert( QDialog::Accepted == ret );
    applyChangesForRemoteMachineMonitor( rmm, dlg.getModel() );
}

void DistributedComputingUtil::applyChangesForRemoteMachineMonitor( 
    RemoteMachineMonitor * rmm, const QList< RemoteMachineMonitorDialogItem > & model ) {
    assert( NULL != rmm );
    
    int sz = model.size();
    for( int i = 0 ; i < sz; ++i ) {
        RemoteMachineMonitorDialogItem item = model.at( i );
        switch( item.state ) {
        case MACHINE_OLD:
            rmm->setSelected( item.settings, item.cb->isChecked() );
            break;
        case MACHINE_NEW:
            if( !rmm->addMachine( item.settings, item.cb->isChecked() ) ) {
                // means that same machine exists in monitor
                delete item.settings;
            }
            break;
        case MACHINE_DELETED:
            rmm->removeMachine( item.settings );
            break;
        default:
            assert( false );
        }
    }
}

QStringList DistributedComputingUtil::filterRemoteMachineServices( const QStringList & services ) {
    QStringList res = services;
    res.removeAll( PingTaskFactory::ID );
    return res;
}

/*******************************************
* RetrieveRemoteMachineInfoTask
*******************************************/

RetrieveRemoteMachineInfoTask::RetrieveRemoteMachineInfoTask( RemoteMachine * m ) 
: Task( tr( "Retrieve remote machine info task" ), TaskFlags_FOSCOE ), doPing( false ), machine( m ) {
    if( NULL == machine ) {
        setError( tr( "Bad remote machine given" ) );
        return;
    }
    
    PingTaskSettings localSettings;
    pingTask = new RemoteTask( PingTaskFactory::ID, localSettings, machine );
    addSubTask( pingTask );
    
    log.message( LogLevel_INFO, tr( "Retrieve remote machine info task started" ) );
}

RetrieveRemoteMachineInfoTask::~RetrieveRemoteMachineInfoTask() {
    if( NULL != machine ) {
        delete machine;
        machine = NULL;
    }
}

void RetrieveRemoteMachineInfoTask::run() {
    if( hasErrors() ) {
        return;
    }
    assert( NULL != machine );
    RemoteTaskError error;
    
    if( isCanceled() ) { return; }
    error = machine->getUuid( &uuid );
    if( !error.getOk() ) {
        setError( error.getMsg() );
        return;
    }
    log.message( LogLevel_INFO, tr( "Retrieved uuid" ) );
    
    if( isCanceled() ) { return; }
    error = machine->getServicesList( &services );
    if( !error.getOk() ) {
        setError( error.getMsg() );
        return;
    }
    log.message( LogLevel_INFO, tr( "Retrieved services list" ) );
    
    if( isCanceled() ) { return; }
    error = machine->getHostName( &hostname );
    if( !error.getOk() ) {
        setError( error.getMsg() );
        return;
    }
    log.message( LogLevel_INFO, tr( "Retrieved hostname" ) );
}

Task::ReportResult RetrieveRemoteMachineInfoTask::report() {
    if( pingTask->isCanceled() ) {
        pingTask->setError( tr( "Ping task is canceled by user" ) );
    }
    if( pingTask->hasErrors() ) {
        setError( tr( "Ping task finished with error: " ) + pingTask->getError() );
        doPing = false;
        return ReportResult_Finished;
    }
    
    doPing = true;
    if( isCanceled() ) {
        setError( tr( "Task is canceled by user" ) );
        return ReportResult_Finished;
    }
    return ReportResult_Finished;
}

QStringList RetrieveRemoteMachineInfoTask::getServicesList() const {
    return services;
}

QUuid RetrieveRemoteMachineInfoTask::getUuid() const {
    return uuid;
}

QString RetrieveRemoteMachineInfoTask::getHostName() const {
    return hostname;
}

bool RetrieveRemoteMachineInfoTask::isPinging() const {
    return doPing;
}

RemoteMachine * RetrieveRemoteMachineInfoTask::takeMachine() {
    RemoteMachine * ret = machine;
    machine = NULL;
    return ret;
}

/*******************************************
* RetrievePublicMachinesTask
*******************************************/

const QString RetrievePublicMachinesTask::PUBLIC_MACHINES_KEEPER_SERVER = "http://ugene.unipro.ru";
const QString RetrievePublicMachinesTask::PUBLIC_MACHINES_KEEPER_PAGE   = "/public_machines.html";
const QString RetrievePublicMachinesTask::PUBLIC_MACHINES_STR_SEPARATOR = "<br>";

RetrievePublicMachinesTask::RetrievePublicMachinesTask() : Task( "Retrieve public remote machines", TaskFlag_None ) {
}

RetrievePublicMachinesTask::~RetrievePublicMachinesTask() {
    qDeleteAll( publicMachines );
}

void RetrievePublicMachinesTask::run() {
    SyncHTTP http( QUrl( PUBLIC_MACHINES_KEEPER_SERVER ).host() );
    NetworkConfiguration * nc = AppContext::getAppSettings()->getNetworkConfiguration();
    assert( NULL != nc );
    bool proxyUsed = nc->isProxyUsed( QNetworkProxy::HttpProxy );
    bool srvIsException = nc->getExceptionsList().contains( QUrl( PUBLIC_MACHINES_KEEPER_SERVER ).host() );
    
    if( proxyUsed && !srvIsException ) {
        http.setProxy( nc->getProxy( QNetworkProxy::HttpProxy ) );
    }
    processEncodedMachines( http.syncGet( PUBLIC_MACHINES_KEEPER_PAGE ) );
}

void RetrievePublicMachinesTask::processEncodedMachines( const QString & encodedMachinesStr ) {
    QStringList encodedMachines = encodedMachinesStr.split( PUBLIC_MACHINES_STR_SEPARATOR, QString::SkipEmptyParts  );
    foreach( const QString & encodedMachine, encodedMachines ) {
        RemoteMachineSettings * settings = NULL;
        if( !SerializeUtils::deserializeRemoteMachineSettings( encodedMachine.trimmed(), &settings ) ) {
            assert( NULL == settings );
            setError( tr( "Bad machine found on server" ) );
            continue;
        }
        publicMachines << settings;
    }
}

QList< RemoteMachineSettings* > RetrievePublicMachinesTask::getPublicMachines() const {
    return publicMachines;
}

QList< RemoteMachineSettings* > RetrievePublicMachinesTask::takePublicMachines() {
    QList< RemoteMachineSettings* > res = publicMachines;
    publicMachines.clear();
    return res;
}

/*******************************************
* SaveRemoteMachineSettings
*******************************************/
SaveRemoteMachineSettings::SaveRemoteMachineSettings(RemoteMachineSettings * machineSettings, const QString& file)
: Task(tr("Save remote machine settings task"), TaskFlag_None), filename(file) {
    if(filename.isEmpty()) {
        setError(tr("Output file not set"));
        return;
    }
    if( machineSettings == NULL ) {
        setError(tr("Nothing to write: empty remote machine settings"));
        return;
    }
    data = SerializeUtils::serializeRemoteMachineSettings(machineSettings).toAscii();
}

void SaveRemoteMachineSettings::run() {
    if(hasErrors() || isCanceled()) {
        return;
    }
    QFile out(filename);
    if( !out.open(QIODevice::WriteOnly) ) {
        setError(tr("Cannot open %1 file").arg(filename));
        return;
    }
    out.write(data);
    out.close();
}

} // GB2
