/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MusclePlugin.h"
#include "MuscleTask.h"
#include "MuscleWorker.h"
#include "MuscleAlignDialogController.h"

#include <core_api/AppContext.h>
#include <core_api/Task.h>
#include <core_api/Log.h>
#include <util_ov_msaedit/MSAEditor.h>
#include <util_ov_msaedit/MSAEditorFactory.h>
#include <gobjects/MAlignmentObject.h>
#include <gobjects/GObjectTypes.h>

#include <util_algorithm/GAutoDeleteList.h>
#include <util_gui/GUIUtils.h>
#include <util_gui/DialogUtils.h>

#include <test_framework/xmltest/XMLTestFormat.h>
#include <test_framework/GTest.h>
#include <test_framework/GTestFrameworkComponents.h>
#include <test_framework/xmltest/DocumentModelTests.h>

#include "umuscle_tests/umuscleTests.h"

#include <QtGui/QDialog>
#include <QtGui/QFileDialog>

#define ULOG_CAT_MUSCLE "Plugin: MUSCLE"

namespace GB2 {

extern "C" Q_DECL_EXPORT Plugin* GB2_PLUGIN_INIT_FUNC() {
    MusclePlugin * plug = new MusclePlugin();
    return plug;
}

LogCategory log(ULOG_CAT_MUSCLE);

    
MusclePlugin::MusclePlugin() 
: Plugin(tr("MUSCLE"), 
         tr("A port of MUSCLE package for multiple sequence alignment. Check http://www.drive5.com/muscle/ for the original version")),
         ctx(NULL)
{
    if (AppContext::getMainWindow()) {
        ctx = new MuscleMSAEditorContext(this);
        ctx->init();
    }
    LocalWorkflow::MuscleWorkerFactory::init();
    //uMUSCLE Test
    GTestFormatRegistry* tfr = AppContext::getTestFramework()->getTestFormatRegistry();
    XMLTestFormat *xmlTestFormat = qobject_cast<XMLTestFormat*>(tfr->findFormat("XML"));
    assert(xmlTestFormat!=NULL);

    GAutoDeleteList<XMLTestFactory>* l = new GAutoDeleteList<XMLTestFactory>(this);
    l->qlist = UMUSCLETests ::createTestFactories();

    foreach(XMLTestFactory* f, l->qlist) { 
        bool res = xmlTestFormat->registerTestFactory(f);
        Q_UNUSED(res);
        assert(res);
    }
}

MusclePlugin::~MusclePlugin() {
    //nothing to do
}

MSAEditor* MuscleAction::getMSAEditor() const {
    MSAEditor* e = qobject_cast<MSAEditor*>(getObjectView());
    assert(e!=NULL);
    return e;
}

void MuscleAction::sl_lockedStateChanged() {
    StateLockableItem* item = qobject_cast<StateLockableItem*>(sender());
    assert(item!=NULL);
    setEnabled(!item->isStateLocked());
}

MuscleMSAEditorContext::MuscleMSAEditorContext(QObject* p) : GObjectViewWindowContext(p, MSAEditorFactory::ID) {
}


void MuscleMSAEditorContext::initViewContext(GObjectView* view) {
    MSAEditor* msaed = qobject_cast<MSAEditor*>(view);
    assert(msaed!=NULL);

    bool objLocked = msaed->getMSAObject()->isStateLocked();
    MuscleAction* alignAction = new MuscleAction(this, view, tr("Align with MUSCLE"));
    alignAction->setIcon(QIcon(":umuscle/images/muscle_16.png"));
    alignAction->setEnabled(!objLocked);
    connect(alignAction, SIGNAL(triggered()), SLOT(sl_align()));
    connect(msaed->getMSAObject(), SIGNAL(si_lockedStateChanged()), alignAction, SLOT(sl_lockedStateChanged()));
    addViewAction(alignAction);

    MuscleAction* addSequencesAction = new MuscleAction(this, view, tr("Align sequences to profile with MUSCLE"));
    addSequencesAction->setIcon(QIcon(":umuscle/images/muscle_16.png"));
    addSequencesAction->setEnabled(!objLocked);
    connect(addSequencesAction, SIGNAL(triggered()), SLOT(sl_alignSequencesToProfile()));
    connect(msaed->getMSAObject(), SIGNAL(si_lockedStateChanged()), addSequencesAction, SLOT(sl_lockedStateChanged()));
    addViewAction(addSequencesAction);

    MuscleAction* alignProfilesAction = new MuscleAction(this, view, tr("Align profile to profile with MUSCLE"));
    alignProfilesAction->setIcon(QIcon(":umuscle/images/muscle_16.png"));
    alignProfilesAction->setEnabled(!objLocked);
    connect(alignProfilesAction, SIGNAL(triggered()), SLOT(sl_alignProfileToProfile()));
    connect(msaed->getMSAObject(), SIGNAL(si_lockedStateChanged()), alignProfilesAction, SLOT(sl_lockedStateChanged()));
    addViewAction(alignProfilesAction);
}

void MuscleMSAEditorContext::buildMenu(GObjectView* v, QMenu* m) {
    QList<GObjectViewAction *> actions = getViewActions(v);
    QMenu* alignMenu = GUIUtils::findSubMenu(m, MSAE_MENU_ALIGN);
    assert(alignMenu!=NULL);
    foreach(GObjectViewAction* a, actions) {
        alignMenu->addAction(a);
    }    
}

void MuscleMSAEditorContext::sl_align() {
    MuscleAction* action = qobject_cast<MuscleAction*>(sender());
    assert(action!=NULL);
    MSAEditor* ed = action->getMSAEditor();
    MAlignmentObject* obj = ed->getMSAObject(); 
    assert(!obj->isStateLocked());

    MuscleTaskSettings s;
    MuscleAlignDialogController dlg(ed->getWidget(), obj->getMAlignment(), s);
    
    int rc = dlg.exec();
    if (rc != QDialog::Accepted) {
        return;
    }

    MuscleGObjectTask* t = new MuscleGObjectTask(obj, s);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void MuscleMSAEditorContext::sl_alignSequencesToProfile() {
    MuscleAction* action = qobject_cast<MuscleAction*>(sender());
    assert(action!=NULL);
    MSAEditor* ed = action->getMSAEditor();
    MAlignmentObject* obj = ed->getMSAObject(); 
    assert(!obj->isStateLocked());

    LastOpenDirHelper lod;
    lod.url = QFileDialog::getOpenFileName(NULL, tr("Select file with sequences"), lod);
    if (lod.url.isEmpty()) {
        return;
    }
    AppContext::getTaskScheduler()->registerTopLevelTask(new MuscleAddSequencesToProfileTask(obj, lod.url, MuscleAddSequencesToProfileTask::Sequences2Profile));
}

void MuscleMSAEditorContext::sl_alignProfileToProfile() {
    MuscleAction* action = qobject_cast<MuscleAction*>(sender());
    assert(action!=NULL);
    MSAEditor* ed = action->getMSAEditor();
    MAlignmentObject* obj = ed->getMSAObject(); 
    assert(!obj->isStateLocked());

    LastOpenDirHelper lod;
    lod.url = QFileDialog::getOpenFileName(NULL, tr("Select file with alignment"), lod,
        DialogUtils::prepareDocumentsFileFilterByObjType(GObjectTypes::MULTIPLE_ALIGNMENT, true));

    if (lod.url.isEmpty()) {
        return;
    }
    AppContext::getTaskScheduler()->registerTopLevelTask(new MuscleAddSequencesToProfileTask(obj, lod.url, MuscleAddSequencesToProfileTask::Profile2Profile));
}

}//namespace

