/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/Counter.h>
#include "SiteconSearchTask.h"

namespace GB2 {

SiteconSearchTask::SiteconSearchTask(const SiteconModel& m, const char* s, int l, const SiteconSearchCfg& cfg, int ro) 
: Task(tr("sitecon_search"), TaskFlags_NR_FOSCOE), model(m), cfg(cfg), resultsOffset(ro)
{
    GCOUNTER( cvar, tvar, "SiteconSearchTask" );
    model.checkState(true);
    model.matrix = SiteconAlgorithm::normalize(model.matrix, model.settings);
    SequenceWalkerConfig c;
    c.seq = s;
    c.seqSize = l;
    c.complTrans  = cfg.complTT;
    c.aminoTrans = NULL;

    c.chunkSize = l;
    c.overlapSize = 0;

    SequenceWalkerTask* t = new SequenceWalkerTask(c, this, tr("sitecon_search_parallel"));
    addSubTask(t);
}

void SiteconSearchTask::onRegion(SequenceWalkerSubtask* t, TaskStateInfo& ti) {
//TODO: process border case as if there are 'N' chars before 0 and after seqlen
    if (cfg.complOnly && !t->isDNAComplemented()) {
        return;
    }
    LRegion globalRegion = t->getGlobalRegion();
    int seqLen = globalRegion.len;
    const char* seq = t->getGlobalConfig().seq + globalRegion.startPos;;
    int modelSize = model.settings.windowSize;
    ti.progress =0;
    int lenPerPercent = seqLen / 100;
    int pLeft = lenPerPercent;
    DNATranslation* complTT = t->isDNAComplemented() ? t->getGlobalConfig().complTrans : NULL;
    for (int i = 0, n = seqLen - modelSize; i < n && !ti.cancelFlag; i++, --pLeft) {
        float psum = SiteconAlgorithm::calculatePSum(seq+i, modelSize, model.matrix, model.settings, model.deviationThresh, complTT);
        if (psum < 0 || psum >=1) {
            ti.setError(  tr("internal_error_invalid_psum:%1").arg(psum) );
            return;
        }
        SiteconSearchResult r;
        r.psum = 100*psum;
        r.err1 = model.err1[r.psum];
        r.err2 = model.err2[r.psum];
        if (r.psum >= cfg.minPSUM && r.err1 >= cfg.minE1 && r.err2 <= cfg.maxE2) {//report result
            
            r.modelInfo = model.modelName;
            r.complement = t->isDNAComplemented();
            r.region.startPos = globalRegion.startPos +  i + resultsOffset;
            r.region.len = modelSize;
            addResult(r);
        }
        if (pLeft == 0) {
            ti.progress++;
            pLeft = lenPerPercent;
        }
    }
}


void SiteconSearchTask::addResult(const SiteconSearchResult& r) {
    lock.lock();
    results.append(r);
    lock.unlock();
}

QList<SiteconSearchResult> SiteconSearchTask::takeResults() {
    lock.lock();
    QList<SiteconSearchResult> res = results;
    results.clear();
    lock.unlock();
    return res;
}

}//namespace
