/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/Log.h>
#include <core_api/AppContext.h>

#include <datatype/AnnotationData.h>
#include <datatype/DNASequence.h>

#include <util_tasks/TaskSignalMapper.h>
#include <util_tasks/FailTask.h>
#include <util_tasks/MultiTask.h>

#include <util_text/TextUtils.h>

#include <workflow/ConfigurationEditor.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow/TypeSet.h>

#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_library/BioDatatypes.h>

#include "SequenceSplitWorker.h"
#include "GenericReadActor.h"

namespace GB2 {
namespace LocalWorkflow {

static LogCategory log(ULOG_CAT_WD);

const QString SequenceSplitWorkerFactory::ACTOR("sequencesplit");

const static QString REGIONED_SEQ_TYPE("regioned.sequence");

const static QString IN_PORT("in");
const static QString OUT_PORT("out");

const static QString ACCEPTED_NAMES_ATTR( "1accepted_names" );
const static QString FILTERED_NAMES_ATTR( "2filtered_names" );
const static QString WHICH_FILTER_ATTR( "3preferred_filter" );
const static QString TRANSLATE_ATTR( "4translate" );
const static QString COMPLEMENT_ATTR( "5complement" );
const static QString EXTEND_LEFT_ATTR( "6extleft" );
const static QString EXTEND_RIGHT_ATTR( "7extright" );
const static QString GAP_LENGTH_ATTR( "9gaplength" );

QString SequenceSplitPromter::composeRichDoc() {
    BusPort * input = qobject_cast<BusPort *> ( target->getPort(IN_PORT) );
    Actor * seqProducer = input->getProducer( IN_PORT );
    QString seqProducerText = seqProducer ? tr("from <u>%1</u>,").arg(seqProducer->getLabel()) : QString();

    //extract names. use max 2 names from the list
    bool useAccepted = getParameter( WHICH_FILTER_ATTR ).toBool();
    QString filterText;
    QString names;
    if( useAccepted ) {
        filterText += tr( " which name is " );
        names = getParameter( ACCEPTED_NAMES_ATTR ).toString();
    } else {
        filterText += tr( " which name is not " );
        names = getParameter( FILTERED_NAMES_ATTR ).toString();
    }
    QStringList annNamesLit = names.split( QRegExp("\\s+"), QString::SkipEmptyParts );
    const int maxNamesMention = 2;
    for( int i = 0, end = qMin(annNamesLit.size(), maxNamesMention); i < end; ++i ) {
        filterText += tr( "\"<u>%1</u>\"" ).arg(annNamesLit.at(i));
        filterText += ", ";
    }
    if( annNamesLit.size() > maxNamesMention ) {
        filterText.remove( filterText.size()-1, 2 ); //remove ", "
        filterText += tr(" or others, ");
    } else if( annNamesLit.empty() ) {
        filterText = ",";
    }

    //translate or not?
    bool translate = getParameter( TRANSLATE_ATTR ).toBool();
    QString translateText = translate ? tr( "<u>translate</u> it if annotation marks translated subsequence, " ) : "";

    //complement or not?
    bool complement = getParameter( COMPLEMENT_ATTR ).toBool();
    QString complementText = complement ? tr( "make it <u>reverse-complement</u> if annotation is located on complement strand, " ) : "";

    //expand
    QString expandText;
    int expandLeft = getParameter( EXTEND_LEFT_ATTR ).toInt();
    int expandRight = getParameter( EXTEND_RIGHT_ATTR ).toInt();
    if( expandLeft ) {
        expandText += tr("expand it to left with <u>%1</u>, ").arg(expandLeft);
    }
    if( expandRight ) {
        expandText += tr("expand it to right with <u>%1</u>").arg(expandRight);
    }
    if( !expandRight && expandLeft ){
        expandText.remove( expandText.size()-1, 2);
    }

    //merge result
    QString doc = tr("Extract each annotated sequence region%1 %2 %3%4")
        .arg(filterText)
        .arg(complementText) 
        .arg(translateText)
        .arg(expandText);
    doc.remove( QRegExp("[\\,\\s]*$") ); //remove all commas and spaces from the end;
    doc.append(".");
    return doc;

}

void SequenceSplitWorker::init() {
    seqPort = ports.value(IN_PORT);
    outPort = ports.value(OUT_PORT);

    QString namesStr = actor->getParameter( ACCEPTED_NAMES_ATTR )->value.toString();
    acceptedNames = namesStr.split( QRegExp("\\s+"), QString::SkipEmptyParts ); //split by whitespace

    QString filteredStr = actor->getParameter( FILTERED_NAMES_ATTR )->value.toString();
    filteredNames = filteredStr.split( QRegExp("\\s+"), QString::SkipEmptyParts ); //split by whitespace

    useAcceptedOrFiltered = actor->getParameter( WHICH_FILTER_ATTR )->value.toBool();
    cfg.translate = actor->getParameter( TRANSLATE_ATTR )->value.toBool();
    cfg.complement = actor->getParameter( COMPLEMENT_ATTR )->value.toBool();

    cfg.extLeft = actor->getParameter( EXTEND_LEFT_ATTR )->value.toInt();
    cfg.extRight = actor->getParameter( EXTEND_RIGHT_ATTR )->value.toInt();

    cfg.gapLength = actor->getParameter( GAP_LENGTH_ATTR )->value.toInt();
    cfg.gapSym = '-'; //FIXME
}

bool SequenceSplitWorker::isReady() {
    return seqPort->hasMessage();
}

Task * SequenceSplitWorker::tick() {
    QVariantMap qm = seqPort->get().getData().toMap();
    DNASequence inputSeq = qm.value( BioActorLibrary::SEQ_SLOT_ID ).value<DNASequence>();
    inputAnns = qVariantValue<QList<SharedAnnotationData> >( qm.value(BioActorLibrary::FEATURE_TABLE_SLOT_ID) );
    
    if( inputSeq.isNull() || inputAnns.isEmpty() ) {
        if( failFast ) {
            return new FailTask( tr("Null input data provided to SequenceSplitWorker") );
        } 

        outPort->put( Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), QVariant()) );
        if( seqPort->isEnded() ) {
            outPort->setEnded();
        }
        return 0;
    } 

    ssTasks.clear();

    foreach( SharedAnnotationData ann, inputAnns ) {
        bool ok = false;
        if( useAcceptedOrFiltered ) {
            ok = acceptedNames.contains( ann->name, Qt::CaseInsensitive );
        } else {
            ok = !filteredNames.contains( ann->name, Qt::CaseInsensitive );
        }
        
        if(ok) {
            Task * t = new ExtractAnnotatedRegionTask( inputSeq, ann, cfg );
            ssTasks.push_back(t);
        }
    }
    Task * t = new MultiTask( "Sequence split tasks", ssTasks );
    connect( new TaskSignalMapper(t), SIGNAL(si_taskFinished(Task*)), SLOT(sl_onTaskFinished(Task*)) );
    return t;
}

void SequenceSplitWorker::cleanup() {
}

bool SequenceSplitWorker::isDone() {
    return seqPort->isEnded();
}

void SequenceSplitWorker::sl_onTaskFinished( Task * t ) {
    foreach( Task * t, ssTasks ) {
        ExtractAnnotatedRegionTask * ssT = qobject_cast<ExtractAnnotatedRegionTask *>(t);
        assert( ssT );

        DNASequence resSeq = ssT->getResultedSequence();
        QVariant vSeq = qVariantFromValue<DNASequence>(resSeq);

        SharedAnnotationData resAnn = ssT->createResultedAnnotation();
        QList<SharedAnnotationData> annToPut;
        annToPut.push_back( resAnn );
        QVariant vAnns = qVariantFromValue<QList<SharedAnnotationData> >(annToPut);

        QVariantMap messageData;
        messageData[ BioActorLibrary::SEQ_SLOT_ID ] = vSeq;
        messageData[ BioActorLibrary::FEATURE_TABLE_SLOT_ID ] = vAnns;

        DataTypePtr messageType = WorkflowEnv::getDataTypeRegistry()->getById( REGIONED_SEQ_TYPE );
        if( outPort ) {
            outPort->put( Message(messageType, messageData) );
        }
    }
    if( seqPort->isEnded() ) {
        outPort->setEnded();
    }
}

void SequenceSplitWorkerFactory::init() {
    QList<PortDescriptor*> portDescs; 
    QList<Attribute*> attribs;

    //accept sequence and annotated regions as input
    QMap<Descriptor, DataTypePtr> inputMap;
    inputMap[ BioActorLibrary::SEQ_SLOT() ] = BioDataTypes::DNA_SEQUENCE_TYPE();
    inputMap[ BioActorLibrary::FEATURE_TABLE_SLOT() ] = BioDataTypes::ANNOTATION_TABLE_TYPE();

    DataTypePtr inSet( new DataTypeSet(Descriptor(REGIONED_SEQ_TYPE), inputMap) );
    DataTypeRegistry * dr = WorkflowEnv::getDataTypeRegistry();
    assert(dr);
    dr->registerEntry( inSet );

    { //Create input port descriptors 
        Descriptor seqDesc( IN_PORT, SequenceSplitWorker::tr("Input sequence"), SequenceSplitWorker::tr("A sequence which will be split into annotated regions.") );
        Descriptor outDesc( OUT_PORT, SequenceSplitWorker::tr("Annotated regions"), SequenceSplitWorker::tr("Resulted subsequences, translated and complemented according to corresponding annotations.") );

        portDescs << new PortDescriptor( seqDesc, inSet, /*input*/ true );
        portDescs << new PortDescriptor( outDesc, inSet, /*input*/false, /*multi*/true );
    }

    { //Create attributes descriptors    
        Descriptor acceptedNamesDesc( ACCEPTED_NAMES_ATTR, 
                                      SequenceSplitWorker::tr("Accepted names"), 
                                      SequenceSplitWorker::tr("Accepted annotations names. Another annotations will be filtered. Use space as the separator.") );
        Descriptor filteredNamesDesc( FILTERED_NAMES_ATTR, 
                                      SequenceSplitWorker::tr("Filtered names"), 
                                      SequenceSplitWorker::tr("Annotations with these names will be filtered out. Use space as the separator.") );
        Descriptor whichFilterDesc( WHICH_FILTER_ATTR,
                                    SequenceSplitWorker::tr("Prefer accepted names"),
                                    SequenceSplitWorker::tr("Selects the name filter: positive value means filtering by accepted names, negative - by filtered names.") );
        Descriptor translateDesc( TRANSLATE_ATTR,
                                  SequenceSplitWorker::tr("Translate"),
                                  SequenceSplitWorker::tr("Translate the annotated regions if the corresponding annotation marks protein subsequence.") );
        Descriptor complementDesc( COMPLEMENT_ATTR,
                                   SequenceSplitWorker::tr("Complement"),
                                   SequenceSplitWorker::tr("Complement the annotated regions if the corresponding annotation is located on complement strand.") );
        Descriptor extendLeftDesc( EXTEND_LEFT_ATTR,
                                   SequenceSplitWorker::tr("Extend left"),
                                   SequenceSplitWorker::tr("Extend the resulted regions to left") );
        Descriptor extendRightDesc( EXTEND_RIGHT_ATTR,
                                    SequenceSplitWorker::tr("Extend right"),
                                    SequenceSplitWorker::tr("Extend the resulted regions to right") );
        Descriptor gapLengthDesc( GAP_LENGTH_ATTR, 
                                  SequenceSplitWorker::tr("Gap length"),
                                  SequenceSplitWorker::tr("Insert gap of specified length between merged locations of annotation.") );

        attribs << new Attribute( acceptedNamesDesc, CoreDataTypes::STRING_TYPE(), /*required*/false );
        attribs << new Attribute( filteredNamesDesc, CoreDataTypes::STRING_TYPE(), /*required*/false, QVariant("source") );
        attribs << new Attribute( whichFilterDesc, CoreDataTypes::BOOL_TYPE(), /*required*/ false, QVariant(false) );
        attribs << new Attribute( translateDesc, CoreDataTypes::BOOL_TYPE(), /*required*/ false, QVariant(true) );
        attribs << new Attribute( complementDesc, CoreDataTypes::BOOL_TYPE(), /*required*/ false, QVariant(true) );
        attribs << new Attribute( extendLeftDesc, CoreDataTypes::NUM_TYPE(), /*required*/ false, QVariant(0) );
        attribs << new Attribute( extendRightDesc, CoreDataTypes::NUM_TYPE(), /*required*/ false, QVariant(0) );
        attribs << new Attribute( gapLengthDesc, CoreDataTypes::NUM_TYPE(), false, QVariant(1) );
    }

    Descriptor desc( SequenceSplitWorkerFactory::ACTOR, 
                     SequenceSplitWorker::tr("Sequence splitter"), 
                     SequenceSplitWorker::tr("Split input sequences into annotated regions.") );
    ActorPrototype * proto = new BusActorPrototype( desc, portDescs, attribs );

    //create delegates for attribute editing
    QMap<QString, PropertyDelegate *> delegates;   
    {
        QVariantMap eMap; eMap["minimum"] = (0); eMap["maximum"] = (INT_MAX);
        delegates[EXTEND_LEFT_ATTR] = new SpinBoxDelegate( eMap );
        delegates[EXTEND_RIGHT_ATTR] = new SpinBoxDelegate( eMap );
        delegates[GAP_LENGTH_ATTR] = new SpinBoxDelegate( eMap );
    }


    proto->setEditor( new DelegateEditor(delegates) );

//    proto->setIconPath( "" );
    proto->setPrompter( new SequenceSplitPromter() );
    WorkflowEnv::getProtoRegistry()->registerProto( BioActorLibrary::CATEGORY_BASIC(), proto );

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById( LocalDomainFactory::ID );
    localDomain->registerEntry( new SequenceSplitWorkerFactory() );
}

Worker * SequenceSplitWorkerFactory::createWorker( Actor * a ) {
    return new SequenceSplitWorker(a);
}

} //ns LocalWorkflow
} //ns GB2
