/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <memory>
#include <core_api/IOAdapter.h>
#include <core_api/AppContext.h>
#include <util_gui/GUIUtils.h>
#include <util_text/TextUtils.h>
#include "DNAExportToCSVTask.h"

namespace GB2 {

DNAExportToCSVTask::DNAExportToCSVTask(QList<Annotation> annotations, QByteArray sequence, DNATranslation *complementTranslation, bool exportSequence, QString url):
    Task("DNAExportToCSV", TaskFlag_None),
    annotations(annotations),
    sequence(sequence),
    complementTranslation(complementTranslation),
    exportSequence(exportSequence),
    url(url)
{
}

namespace
{
    template<class Container>
    void writeCSVLine(Container container, IOAdapter *ioAdapter) throw(int)
    {
        bool first = true;
        foreach(QString value, container)
        {
            if(!first)
            {
                if(0 == ioAdapter->writeBlock(","))
                {
                    throw 0;
                }
            }
            if(0 == ioAdapter->writeBlock(("\"" + value.replace("\"","\"\"") + "\"").toLocal8Bit()))
            {
                throw 0;
            }
            first = false;
        }
        if(0 == ioAdapter->writeBlock("\n"))
        {
            throw 0;
        }
    }
}

void DNAExportToCSVTask::run()
{
    try
    {
        std::auto_ptr<IOAdapter> ioAdapter;
        {
            IOAdapterId ioAdapterId = BaseIOAdapters::url2io(url);
            IOAdapterFactory *ioAdapterFactory = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(ioAdapterId);
            if(NULL == ioAdapterFactory)
            {
                stateInfo.setError(tr("No IO adapter found for URL: %1").arg(url));
                return;
            }
            ioAdapter.reset(ioAdapterFactory->createIOAdapter());
        }
        if(!ioAdapter->open(url, IOAdapterMode_Write))
        {
            stateInfo.setError(Translations::errorOpeningFileWrite(url));
            return;
        }
        QHash<QString, int> columnIndices;
        QStringList columnNames;
        {
            columnNames.append(tr("Group"));
            columnNames.append(tr("Name"));
            columnNames.append(tr("Start"));
            columnNames.append(tr("End"));
            columnNames.append(tr("Length"));
            columnNames.append(tr("Complement"));
            if(exportSequence)
            {
                columnNames.append(tr("Sequence"));
            }
            foreach(Annotation annotation, annotations)
            {
                foreach(Qualifier qualifier, annotation.getQualifiers())
                {
                    if(!columnIndices.contains(qualifier.getQualifierName()))
                    {
                        columnIndices.insert(qualifier.getQualifierName(), columnNames.size());
                        columnNames.append(qualifier.getQualifierName());
                    }
                }
            }
            writeCSVLine(columnNames, ioAdapter.get());
        }
        foreach(Annotation annotation, annotations)
        {
            foreach(LRegion region, annotation.getLocation())
            {
                QVector<QString> values;
                values.append(annotation.getGroups().last()->getGroupPath());
                values.append(annotation.getAnnotationName());
                if(!annotation.isOnComplementStrand())
                {
                    values.append(QString::number(region.startPos + 1));
                    values.append(QString::number(region.startPos + region.len));
                }
                else
                {
                    values.append(QString::number(region.startPos + region.len));
                    values.append(QString::number(region.startPos + 1));
                }
                values.append(QString::number(region.len));
                if(annotation.isOnComplementStrand())
                {
                    values.append(tr("yes"));
                }
                else
                {
                    values.append(tr("no"));
                }
                if(exportSequence)
                {
                    QByteArray sequencePart = sequence.mid(region.startPos, region.len);
                    if(annotation.isOnComplementStrand())
                    {
                        complementTranslation->translate(sequencePart.data(), sequencePart.size());
                        TextUtils::reverse(sequencePart.data(), sequencePart.size());
                    }
                    values.append(sequencePart);
                }
                values.resize(columnNames.size());
                foreach(Qualifier qualifier, annotation.getQualifiers())
                {
                    values[columnIndices[qualifier.getQualifierName()]] = qualifier.getQualifierValue();
                }
                writeCSVLine(values, ioAdapter.get());
            }
        }
    }
    catch(int)
    {
        stateInfo.setError(Translations::errorWritingFile(url));
        return;
    }
}

} // namespace GB2
