/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/AppContext.h>
#include <core_api/DNAAlphabet.h>
#include <util_text/TextUtils.h>
#include <gobjects/AnnotationSettings.h>

#include "ExtractAnnotatedRegionTask.h"

namespace GB2{ 

ExtractAnnotatedRegionTask::ExtractAnnotatedRegionTask( const DNASequence & sequence_, SharedAnnotationData sd_, const ExtractAnnotatedRegionTaskSettings & cfg_ ) :
Task( tr("Sequence split"), TaskFlag_None ), inputSeq(sequence_), inputAnn(sd_), cfg(cfg_), complT(0), aminoT(0)
{
}

void ExtractAnnotatedRegionTask::prepare() {
    prepareTranslations();
    resultedSeq.alphabet = aminoT ? aminoT->getDstAlphabet() : complT ? complT->getDstAlphabet() : inputSeq.alphabet;
    resultedSeq.info = inputSeq.info;
}

void ExtractAnnotatedRegionTask::prepareTranslations() {
    //TODO move these logic somewhere upstairs
    bool aminoSeq = inputSeq.alphabet->isAmino();
    if( !aminoSeq ) {
        if( cfg.complement && inputAnn->complement ) {
            QList<DNATranslation*> compTTs = AppContext::getDNATranslationRegistry()->
                lookupTranslation( inputSeq.alphabet, DNATranslationType_NUCL_2_COMPLNUCL );
            if (!compTTs.isEmpty()) {
                complT = compTTs.first(); 
            }
        }

        AnnotationSettingsRegistry * asr = AppContext::getAnnotationsSettingsRegistry();
        AnnotationSettings * as = asr->getAnnotationSettings( inputAnn->name );
        bool amino = (TriState_Yes == inputAnn->aminoStrand) || (as && as->amino);

        if( cfg.translate && amino ) {
            DNATranslationType dnaTranslType = (inputSeq.alphabet->getType() == DNAAlphabet_NUCL) ? DNATranslationType_NUCL_2_AMINO : DNATranslationType_RAW_2_AMINO;
            QList<DNATranslation*> aminoTTs = AppContext::getDNATranslationRegistry()->lookupTranslation( inputSeq.alphabet, dnaTranslType );
            if( !aminoTTs.isEmpty() ) {
                aminoT = aminoTTs.first();
            } 
        }
    }
}

void ExtractAnnotatedRegionTask::run() {
    createResultedSequence(); //fills extended regions list
}

SharedAnnotationData ExtractAnnotatedRegionTask::createResultedAnnotation() {
    AnnotationData * newAnn = new AnnotationData();
    *newAnn = *inputAnn;

    bool translated = bool(aminoT);
    bool complemented = bool(complT);
    unsigned normalizer = translated ? 3 : 1;

    if( complemented ) {
        newAnn->complement = false; //sequence was complemented, so complement annotation is now located on the direct strand
    }

    //create regions in output sequence
    newAnn->location.clear();

    //create 'local' regions
    //localRegion.startPos - offset in the extended region (non-normalized)
    //localRegion.len - length of the resulted region (the same as lenght of the input region)
    QList<LRegion> localRegions;
    for( int i = 0, end = extendedRegions.size(); i < end; ++i ) {
        const LRegion & extReg = extendedRegions.at(i);
        const LRegion & inpReg = inputAnn->location.at(i);

        LRegion localReg;
        localReg.startPos = (inpReg.startPos - extReg.startPos) ;
        localReg.len = inpReg.len ;

        if( complemented ) {
            localReg.startPos = extReg.len - localReg.startPos - localReg.len;
        }
        localRegions.push_back( localReg );
    }

    //create resulted regions
    for( int i = 0, offs = 0, end = extendedRegions.size(); i < end; ++i ) {
        LRegion reg;
        reg.startPos = offs + localRegions.at(i).startPos / normalizer;
        reg.len = localRegions.at(i).len / normalizer;

        newAnn->location.push_back(reg);
        offs += extendedRegions.at(i).len / normalizer + cfg.gapLength;
    }

    return SharedAnnotationData(newAnn);
}

void ExtractAnnotatedRegionTask::createResultedSequence()
{
    unsigned extLeftActual = cfg.extLeft;
    unsigned extRightActual = cfg.extRight;

    if( aminoT ) {
        extLeftActual *= 3;
        extRightActual *= 3;
    }

    const QByteArray & sequence = inputSeq.seq;
    const QList<LRegion> & location = inputAnn->location;
    QByteArray & res = resultedSeq.seq;

    LRegion sequenceRange(0, sequence.size());
    //extend regions
    QListIterator<LRegion> i(location);
    while( i.hasNext() ) {
        LRegion reg = i.next();
        reg.startPos -= extLeftActual;
        reg.len += extRightActual + extLeftActual;
        reg = reg.intersect( sequenceRange );

        extendedRegions.push_back( reg );
    }

    for( int i = 0, end = extendedRegions.size(); i < end; ++i ) {
        if( 0 != i ) {
            res.append( QByteArray(cfg.gapLength, cfg.gapSym) );
        }
        LRegion reg = extendedRegions.at(i);
        QByteArray partSeq(sequence.constData() + reg.startPos, reg.len);
        if( complT ) {
            TextUtils::translate(complT->getOne2OneMapper(), partSeq.data(), partSeq.size());
            TextUtils::reverse(partSeq.data(), partSeq.size());
        }
        if( aminoT ) {
            aminoT->translate(partSeq.data(), reg.len, partSeq.data(), reg.len);
            partSeq.resize(partSeq.size()/3);
        }
        res.append(partSeq);
    }

}

DNASequence ExtractAnnotatedRegionTask::getResultedSequence() const {
    return resultedSeq;
}


} //ns GB2
