/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <datatype/DNASequence.h>

#include <gobjects/DNASequenceObject.h>
#include <gobjects/GObjectTypes.h>

#include <core_api/DocumentModel.h>
#include <core_api/DocumentFormats.h>
#include <core_api/GObject.h>
#include <core_api/SmithWatermanTaskFactoryRegistry.h>
#include <core_api/AppContext.h>
#include <core_api/Log.h>

#include <qbytearray.h>

#include <util_smith_waterman/SubstMatrixFactory.h>
#include <util_smith_waterman/SmithWatermanSettings.h>
 
#include <util_smith_waterman/SubstMatrix.h>

#include <document_format/SubstMatrixFile.h>

#include "util_smith_waterman/SmithWatermanTests.h"


#define FILE_SUBSTITUTION_MATRIX_ATTR "subst_f"
#define FILE_FASTA_CONTAIN_SEQUENCE_ATTR "seq_f"
#define FILE_FASTA_CONTAIN_PATTERN_ATTR "pattern_f"
#define GAP_OPEN_ATTR "g_o"
#define GAP_EXT_ATTR "g_e"
#define PERCENT_OF_SCORE_ATTR "percent_of_score"
#define EXPECTED_RESULT_ATTR "expected_res"
#define ENV_IMPL_ATTR "IMPL"

#include "iostream"

using namespace std;
namespace GB2 {

	static LogCategory log("Smith Waterman plugin");

	void GTest_SmithWatermnan::sortByScore(QList<SmithWatermanResult> & resultsForSort) {
		log.info("Sorting SmithWatermanResult");
		for (int i = 0; i < resultsForSort.size(); i++) {
			for (int j = i + 1; j < resultsForSort.size(); j++) {
				if (resultsForSort.at(i).score < resultsForSort.at(j).score) {
					SmithWatermanResult buf = resultsForSort.at(i);
					resultsForSort[i] = resultsForSort.at(j);
					resultsForSort[j] = buf;
				}
				if (resultsForSort.at(i).score == resultsForSort.at(j).score &&
					resultsForSort.at(i).region.startPos > resultsForSort.at(j).region.startPos) {
						SmithWatermanResult buf = resultsForSort.at(i);
						resultsForSort[i] = resultsForSort.at(j);
						resultsForSort[j] = buf;
				}
				if (resultsForSort.at(i).score == resultsForSort.at(j).score &&
					resultsForSort.at(i).region.startPos == resultsForSort.at(j).region.startPos &&
					resultsForSort.at(i).region.len > resultsForSort.at(j).region.len) {
						SmithWatermanResult buf = resultsForSort.at(i);
						resultsForSort[i] = resultsForSort.at(j);
						resultsForSort[j] = buf;
				}
			}
		}
		log.info("Finish Sorting SmithWatermanResult");
	}


	void GTest_SmithWatermnan::init(XMLTestFormat *tf, const QDomElement& el) {		
		Q_UNUSED(tf);

		log.info("RUN GTest_SmithWatermnan::init(XMLTestFormat *tf, const QDomElement& el)");
		
		searchSeqDocName = el.attribute(FILE_FASTA_CONTAIN_SEQUENCE_ATTR);
		if (searchSeqDocName.isEmpty()) {
			log.error("FailMissingValue: seq_f");
			failMissingValue(FILE_FASTA_CONTAIN_SEQUENCE_ATTR);
			return;
		} 

		patternSeqDocName = el.attribute(FILE_FASTA_CONTAIN_PATTERN_ATTR);
		if (patternSeqDocName.isEmpty()) {
			log.error("Fail Missing Value: pat_f");
			failMissingValue(FILE_FASTA_CONTAIN_PATTERN_ATTR);
			return;
		} 

		pathToSubst = el.attribute(FILE_SUBSTITUTION_MATRIX_ATTR);
		if (pathToSubst.isEmpty()) {
			log.error("Fail Missing Value: subst_f");
			failMissingValue(FILE_SUBSTITUTION_MATRIX_ATTR);
			return;
		} 		

		QString buffer = el.attribute(GAP_OPEN_ATTR);
		bool ok = false;

		if (!buffer.isEmpty()) {
			ok=false;
			gapOpen = buffer.toInt(&ok);
			if(!ok) {
				log.error("Missing Value: g_o");
				failMissingValue(GAP_OPEN_ATTR);
				return;
			}
		}

		buffer = el.attribute(GAP_EXT_ATTR);
		if (!buffer.isEmpty()) {
			ok=false;
			gapExtension = buffer.toInt(&ok);
			if(!ok) {
				log.error("Missing Value: g_e");
				failMissingValue(GAP_EXT_ATTR);
				return;
			}
		}

		buffer = el.attribute(PERCENT_OF_SCORE_ATTR);
		if (!buffer.isEmpty()) {
			ok=false;
			percentOfScore = buffer.toFloat(&ok);
			if(!ok) {
				log.error("Missing Value: percent_of_score");
				failMissingValue(PERCENT_OF_SCORE_ATTR);
				return;
			}
		}

		expected_res = el.attribute(EXPECTED_RESULT_ATTR);		
		if (expected_res.isEmpty()) {
			log.error("Missing Value: expected_res");
			failMissingValue(EXPECTED_RESULT_ATTR);
			return;
		}

 		if (!parseExpected_res()) {
			log.error("value not correct: expected_res");
 			stateInfo.setError(QString("value not correct %1").arg(EXPECTED_RESULT_ATTR));
			return;
 		}

		impl = env->getVar(ENV_IMPL_ATTR);		
		if (impl.isEmpty()) {
			log.error("Missing Value: IMPL");
			failMissingValue(ENV_IMPL_ATTR);
			return;
		}

		log.info("FINISH GTest_SmithWatermnan::init(XMLTestFormat *tf, const QDomElement& el)");
	}


	void GTest_SmithWatermnan::prepare() {

		log.info("RUN GTest_SmithWatermnan::prepare()");
		//get search sequence		
		DNASequenceObject * searchSeqObj = getContext<DNASequenceObject>(this, searchSeqDocName);
		if(searchSeqObj==NULL){
			stateInfo.setError(QString("error can't cast to sequence from GObject"));
			return;
		}
		searchSeq = searchSeqObj->getDNASequence().seq;

		//get pattern sequence
		DNASequenceObject * patternSeqObj = getContext<DNASequenceObject>(this, patternSeqDocName);
		if(patternSeqObj==NULL){
			stateInfo.setError(QString("error can't cast to sequence from GObject"));
			return;
		}
		patternSeq = patternSeqObj->getDNASequence().seq;

		//set subst matrix

		QString pathToCommonData = getEnv()->getVar("COMMON_DATA_DIR");		
		if(patternSeqObj==NULL){
			stateInfo.setError(QString("error can't get path to common_data dir"));
			return;
		}
		QString fullPathToSubst = pathToCommonData + "/" + pathToSubst;		

		SubstMatrix * mtx = SubstMatrixFile::read(fullPathToSubst); 
		if (0 == mtx) {
			stateInfo.setError(QString("value not set %1").arg(FILE_SUBSTITUTION_MATRIX_ATTR));
			return;
		}		

		SmithWatermanSettings s;
		s.pSm = mtx;
		s.sqnc = searchSeq;
		s.ptrn = patternSeq;
		s.globalRegion.startPos = 0;
		s.globalRegion.len = searchSeq.length();
		s.gapModel.scoreGapOpen = gapOpen;
		s.gapModel.scoreGapExtd = gapExtension;
		s.percentOfScore = percentOfScore;				
		s.aminoTT = NULL;		
		s.complTT = NULL;
		s.strand = SmithWatermanStrand_Direct;
		
		rcb = new SmithWatermanReportCallbackImpl(0,
			QString(""),
			QString(""));
		
		s.resultCallback = rcb;
		s.resultListener = rcb->getListener();		
		s.resultFilter = 0;		

		if (AppContext::getSmithWatermanTaskFactoryRegistry()->getAlgorithm(impl) != NULL) {
			swAlgorithmTask = (Task *) AppContext::getSmithWatermanTaskFactoryRegistry()->getAlgorithm(impl)->getTaskInstance(s, "tests SmithWaterman");		
		} else {
			stateInfo.setError(QString("Not known impl of Smith-Waterman: %1").arg(impl));
			return;
		}

		rcb->setParent(swAlgorithmTask);
		addSubTask(swAlgorithmTask);

		log.info("FINISH GTest_SmithWatermnan::prepare()");
	}

	bool GTest_SmithWatermnan::parseExpected_res() {
		log.info("RUN GTest_SmithWatermnan::parseExpected_res()");
		
		SWresult swRes;			
		QStringList expectedList = expected_res.split("**");	

		foreach(QString res, expectedList) {		
			QStringList resValues = res.split(",");
			if (resValues.size() != 2) {
				stateInfo.setError(QString("wrong count values in expected result: %1").arg(resValues.size()));
				return false;
			}

			//////// first enterval
			QStringList bounds = resValues.at(1).split("..");			
			if (bounds.size() != 2) {
				stateInfo.setError(  QString("wrong region in expected result %1").arg(resValues.at(1)) );
				return false;
			}
			bool startOk, finishOk;
			int start = bounds.first().toInt(&startOk);
			int finish = bounds.last().toInt(&finishOk);
			if (startOk && finishOk != true) {
				stateInfo.setError(  QString("wrong region in expected result %1").arg(resValues.at(1)) );
				return false;
			}
			swRes.sInterval.startPos = start;
			swRes.sInterval.len = finish - start;
			
			start = resValues.at(0).toInt(&startOk);
			if (startOk != true) {
				stateInfo.setError(  QString("wrong scorein expected result %1").arg(resValues.at(0)) );
				return false;
			}
			swRes.score= start;			

			expectedRes.append(swRes);			
		}
		// print expected results
// 		foreach(SWresult exRes, expectedRes) {
// 			cout <<"score: " <<exRes.score 
// 				<<" interval: " <<exRes.sInterval.startPos <<".." <<exRes.sInterval.endPos() <<endl;
// 
//  		}
		log.info("FINISH GTest_SmithWatermnan::parseExpected_res()");
		return true;
	}

	bool GTest_SmithWatermnan::toInt(QString & str, int & num) {
		bool ok = false;		
		if (!str.isEmpty()) {			
			num = str.toInt(&ok);
		} 
		return ok;		
	}	


	Task::ReportResult GTest_SmithWatermnan::report() {
		log.info("RUN GTest_SmithWatermnan::report()");			
		
		QList<SmithWatermanResult> resultList = rcb->getListener()->popAllResults();

		sortByScore(resultList);

		//Print result		
// 		cout <<"result size " << resultList.size() <<endl;
// 		for (int qq = 0; qq < resultList.size(); qq++) {
// 			cout <<"res " <<qq <<": " <<resultList.at(qq).score <<"|" <<resultList.at(qq).region.startPos 
// 		 		<<"|" <<resultList.at(qq).region.endPos() <<endl;
// 		 	}
// 
// 		for (int qq = 0; qq < resultList.size(); qq++) {
// 			cout <<"**" <<resultList.at(qq).score <<"," <<resultList.at(qq).region.startPos <<".."
// 				<<resultList.at(qq).region.endPos();
// 		}
//		cout <<endl;

		if (expectedRes.size() != resultList.size()) {
			stateInfo.setError(QString("Not expected result: count result not coincide"));
 			return ReportResult_Finished;
		}

		for (int i = 0; i < resultList.size(); i++) {
			if (expectedRes.at(i).score != resultList.at(i).score ||
				expectedRes.at(i).sInterval != resultList.at(i).region) {
					stateInfo.setError(QString("Not expected result"));
					return ReportResult_Finished;
			}
		}
		
		log.info("FINISH GTest_SmithWatermnan::report()");
		return ReportResult_Finished;
	}

}
