/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditor.h"
#include "MSAEditorTasks.h"
#include "MSAEditorFactory.h"
#include "MSAEditorState.h"
#include "MSAEditorConsensusArea.h"
#include "MSAEditorOffsetsView.h"
#include "MSAEditorSequenceArea.h"
#include "MSAEditorNameList.h"
#include "MSAEditorStatusBar.h"
#include "MSAEditorUndoFramework.h"

#include <gobjects/MAlignmentObject.h>
#include <util_gui/GUIUtils.h>
#include <util_gui/DialogUtils.h>

#include <QtCore/QEvent>

#include <QtGui/QLabel>
#include <QtGui/QPainter>
#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QGridLayout>
#include <QtGui/QResizeEvent>
#include <QtGui/QToolBar>
#include <QtGui/QFileDialog>

namespace GB2 {

#define IMAGE_DIR   "image"
/* TRANSLATOR GB2::MSAEditor */ 

MSAEditor::MSAEditor(const QString& viewName, MAlignmentObject* obj) : GObjectView(MSAEditorFactory::ID, viewName),  msaObject(obj), ui(NULL) 
{
    objects.append(msaObject);
    requiredObjects.append(msaObject);
}

MSAEditor::~MSAEditor() {

}

void MSAEditor::buildStaticToolbar(QToolBar* tb) {
    GObjectView::buildStaticToolbar(tb);
}

void MSAEditor::buildStaticMenu(QMenu* m) {
    addCopyMenu(m);
    addEditMenu(m);
    addAlignMenu(m);
    addStatisticsMenu(m);
    addViewMenu(m);
    addAdvancedMenu(m);

    GObjectView::buildStaticMenu(m);

    GUIUtils::disableEmptySubmenus(m);
}


void MSAEditor::addCopyMenu(QMenu* m) {
    QMenu* cm = m->addMenu(tr("copy_menu"));
    cm->menuAction()->setObjectName(MSAE_MENU_COPY);
}

void MSAEditor::addEditMenu(QMenu* m) {
    QMenu* em = m->addMenu(tr("edit_menu"));
    em->menuAction()->setObjectName(MSAE_MENU_EDIT);
}

void MSAEditor::addViewMenu(QMenu* m) {
    QMenu* em = m->addMenu(tr("View"));
    em->menuAction()->setObjectName(MSAE_MENU_VIEW);
    em->addAction(ui->offsetsView->getToggleColumnsViewAction());
}

void MSAEditor::addAlignMenu(QMenu* m) {
    QMenu* em = m->addMenu(tr("align_menu"));
    em->setIcon(QIcon(":core/images/align.png"));
    em->menuAction()->setObjectName(MSAE_MENU_ALIGN);
}

void MSAEditor::addAdvancedMenu(QMenu* m) {
    QMenu* em = m->addMenu(tr("advanced_menu"));
    em->menuAction()->setObjectName(MSAE_MENU_ADVANCED);
}

void MSAEditor::addStatisticsMenu(QMenu* m) {
    QMenu* em = m->addMenu(tr("Statistics"));
    em->setIcon(QIcon(":core/images/chart_bar.png"));
    em->menuAction()->setObjectName(MSAE_MENU_STATISTICS);
}

Task* MSAEditor::updateViewTask(const QString& stateName, const QVariantMap& stateData) {
    return new UpdateMSAEditorTask(this, stateName, stateData);
}

QVariantMap MSAEditor::saveState() {
    return MSAEditorState::saveState(this);
}

QWidget* MSAEditor::createWidget() {
    assert(ui == NULL);
    ui = new MSAEditorUI(this);
    connect(ui , SIGNAL(customContextMenuRequested(const QPoint &)), SLOT(sl_onContextMenuRequested(const QPoint &)));
    return ui;
}

int MSAEditor::getAlignmentLen() const {
    return msaObject->getMAlignment().getLength();
}

int MSAEditor::getNumSequences() const {
    return msaObject->getMAlignment().getNumSequences();
}

void MSAEditor::sl_onContextMenuRequested(const QPoint & pos) {
    Q_UNUSED(pos);

    QMenu m;

    addCopyMenu(&m);
    addEditMenu(&m);
    addAlignMenu(&m);
    addStatisticsMenu(&m);
    addViewMenu(&m);
    addAdvancedMenu(&m);

    emit si_buildPopupMenu(this, &m);

    GUIUtils::disableEmptySubmenus(&m);

    m.exec(QCursor::pos());
}

//////////////////////////////////////////////////////////////////////////
MSAEditorUI::MSAEditorUI(MSAEditor* _editor) : editor(_editor) {
    undoFWK = new MSAEditorUndoFramework(this, editor->getMSAObject());

    setContextMenuPolicy(Qt::CustomContextMenu);
    setMinimumSize(300, 200);

    QScrollBar* shBar = new QScrollBar(Qt::Horizontal);
    QScrollBar* svBar = new QScrollBar(Qt::Vertical);
    QScrollBar* nhBar = new QScrollBar(Qt::Horizontal);
    seqArea = new MSAEditorSequenceArea(this, shBar, svBar);
    nameList = new MSAEditorNameList(this, nhBar);
    consArea = new MSAEditorConsensusArea(this);
    offsetsView = new MSAEditorOffsetsViewController(this, editor->getMSAObject(), seqArea);
    statusWidget = new MSAEditorStatusWidget(editor->getMSAObject(), seqArea);
    
    seqArea->installEventFilter(this);
    nameList->installEventFilter(this);
    offsetsView->getLeftWidget()->installEventFilter(this);
    offsetsView->getRightWidget()->installEventFilter(this);

    QGridLayout* seqAreaLayout = new QGridLayout();
    seqAreaLayout->setMargin(0);
    seqAreaLayout->setSpacing(0);
    seqAreaLayout->addWidget(offsetsView->getLeftWidget(), 0, 0);
    seqAreaLayout->addWidget(seqArea, 0, 1);
    seqAreaLayout->addWidget(offsetsView->getRightWidget(), 0, 2);
    seqAreaLayout->addWidget(svBar, 0, 3);
    seqAreaLayout->addWidget(shBar, 1, 0, 1, 3);
    QWidget* seqAreaContainer = new QWidget();
    seqAreaContainer->setLayout(seqAreaLayout);
    

    QVBoxLayout* nameAreaLayout = new QVBoxLayout();
    nameAreaLayout->setMargin(0);
    nameAreaLayout->setSpacing(0);
    nameAreaLayout->addWidget(nameList);
    nameAreaLayout->addWidget(nhBar);
    QWidget* nameAreaContainer = new QWidget();
    nameAreaContainer->setLayout(nameAreaLayout);

    splitter = new QSplitter(Qt::Horizontal);
    splitter->addWidget(nameAreaContainer);
    splitter->addWidget(seqAreaContainer);
    splitter->setStretchFactor(0, 0);
    splitter->setStretchFactor(1, 1);
    splitter->setSizes(QList<int>()<<220<<220); //todo: make initial namelist size depends on font
    splitter->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);

//    QToolBar* tb = new QToolBar();
//    tb->addWidget(statusWidget);

    QVBoxLayout* mainLayout = new QVBoxLayout();
    mainLayout->setMargin(0);
    mainLayout->setSpacing(0);
    mainLayout->addWidget(splitter);
    mainLayout->addWidget(statusWidget);
    setLayout(mainLayout);
    
    addLine(0, createLabelWidget(tr("Consensus")), consArea);
}


void MSAEditorUI::addLine(int row, QWidget* naw, QWidget* saw) {
    QHBoxLayout* l = new QHBoxLayout();
    l->setSpacing(0);
    l->setMargin(0);
    
    l->addWidget(naw);
    nameAreaWidgets.append(naw);
    
    QWidget* lw1 = createLabelWidget();
    lw1Widgets.append(lw1);
    l->addWidget(lw1);

    l->addWidget(saw);
    seqAreaWidgets.append(saw);
    
    QWidget* lw2 = createLabelWidget();
    lw2Widgets.append(lw2);
    l->addWidget(lw2);

    QWidget* w = new QWidget();
    w->setLayout(l);
    
    ((QVBoxLayout*)layout())->insertWidget(row, w);

    updateMSALayout();
}


QWidget* MSAEditorUI::createLabelWidget(const QString& text, Qt::Alignment ali) const {
    return new MSALabelWidget(this, text, ali);
}

bool MSAEditorUI::eventFilter(QObject*, QEvent* e) {
    if (e->type() == QEvent::Resize) {
        updateMSALayout();
    }
    return false;
}

QAction* MSAEditorUI::getUndoAction() const {
    return undoFWK->getUndoAction();
}

QAction* MSAEditorUI::getRedoAction() const {
    return undoFWK->getRedoAction();
}

void MSAEditorUI::updateMSALayout() {
    int naw = nameList->width();
    foreach(QWidget* w, nameAreaWidgets) {
        w->setFixedWidth(naw);
    }
    bool offsetsShown = offsetsView->getToggleColumnsViewAction()->isChecked();
    int lw1w = splitter->handleWidth() + (offsetsShown ? offsetsView->getLeftWidget()->width() : 0);
    foreach(QWidget* lw1, lw1Widgets) {
        lw1->setFixedWidth(lw1w);
    }
    
    /* sequence area widget size is set automatically by QT -> less paint events
    int saw = seqArea->width();
    foreach(QWidget* w, seqAreaWidgets) {
        w->setMaximumWidth(saw);
    }*/

    int lw2w = seqArea->getVBar()->width() + (offsetsShown ? offsetsView->getRightWidget()->width() : 0);
    foreach(QWidget* lw2, lw2Widgets) {
        lw2->setFixedWidth(lw2w);
    }
}

void MSAEditorUI::sl_saveScreenshot() {
    LastOpenDirHelper lod(IMAGE_DIR);
    QRect screenRect = nameList->geometry();
    screenRect.setBottomRight(seqArea->parentWidget()->geometry().bottomRight());
    QPixmap curPixMap = QPixmap::grabWidget(this, screenRect);
    QStringList formats = (QStringList() << "png" << "jpg" << "tiff");
    QString initialPath = lod.dir + "/untitled";//QDir::currentPath() + tr("/untitled");
    QString fileName = QFileDialog::getSaveFileName(this, tr("Save As"),
        initialPath,
        QString("%1 Files (*.%2);;%3 Files (*.%4);;%5 Files (*.%6);;All Files (*)")
        .arg(formats.at(0).toUpper())
        .arg(formats.at(0))
        .arg(formats.at(1).toUpper())
        .arg(formats.at(1))
        .arg(formats.at(2).toUpper())
        .arg(formats.at(2)));
    if (!fileName.isEmpty())
        curPixMap.save(fileName);
}

MSALabelWidget::MSALabelWidget(const MSAEditorUI* _ui, const QString & _t, Qt::Alignment _a) 
: ui(_ui), text(_t), ali(_a)
{
    connect(ui->getSequenceWidget(), SIGNAL(si_scaleChanged()), SLOT(sl_scaleChanged()));
}

void MSALabelWidget::paintEvent(QPaintEvent *) {
    QPainter p(this);
    p.fillRect(rect(), Qt::white);
    if (!text.isEmpty()) {
        p.setFont(ui->getSequenceWidget()->getFont());
        p.drawText(rect(), text, ali);
    }
}

}//namespace

