/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAColorScheme.h"
#include "MSAEditorFactory.h"

#include <gobjects/MAlignmentObject.h>
#include <util_gui/GUIUtils.h>
#include <util_algorithm/MSAUtils.h>

namespace GB2 {

//////////////////////////////////////////////////////////////////////////
// factories

MSAColorSchemeFactory::MSAColorSchemeFactory(QObject* p, const QString& _id, const QString& _name, DNAAlphabetType _at)
: QObject(p), id(_id), name(_name), aType(_at)
{
}

MSAColorSchemeStaticFactory::MSAColorSchemeStaticFactory(QObject* p, const QString& _id, const QString& _name, 
                                                         DNAAlphabetType _atype, const QVector<QColor>& _colorsPerChar) 
: MSAColorSchemeFactory(p, _id, _name, _atype), colorsPerChar(_colorsPerChar)
{
}

MSAColorScheme* MSAColorSchemeStaticFactory::create(QObject* p, MAlignmentObject* o) {
    return new MSAColorSchemeStatic(p, this, o, colorsPerChar);
}


MSAColorSchemePercIdentFactory::MSAColorSchemePercIdentFactory(QObject* p, const QString& _id, const QString& _name, DNAAlphabetType _atype) 
: MSAColorSchemeFactory(p, _id, _name, _atype)
{
}

MSAColorScheme* MSAColorSchemePercIdentFactory::create(QObject* p, MAlignmentObject* o) {
    return new MSAColorSchemePercIdent(p, this, o);
}


MSAColorSchemeClustalXFactory::MSAColorSchemeClustalXFactory(QObject* p, const QString& _id, const QString& _name, DNAAlphabetType _atype) 
: MSAColorSchemeFactory(p, _id, _name, _atype)
{
}

MSAColorScheme* MSAColorSchemeClustalXFactory::create(QObject* p, MAlignmentObject* o) {
    return new MSAColorSchemeClustalX(p, this, o);
}


//////////////////////////////////////////////////////////////////////////
// schemes

MSAColorScheme::MSAColorScheme(QObject* p, MSAColorSchemeFactory* f, MAlignmentObject* o) : QObject(p), factory(f), maObj(o) {
}


MSAColorSchemeStatic::MSAColorSchemeStatic(QObject* p, MSAColorSchemeFactory* f, MAlignmentObject* o, const QVector<QColor>& _cp)
: MSAColorScheme(p, f, o), colorsPerChar(_cp)
{
}

QColor MSAColorSchemeStatic::getColor(int seq, int pos) {
    char c = maObj->getMAlignment().charAt(seq, pos);
    return getColor(c);
}


/// PERCENT
MSAColorSchemePercIdent::MSAColorSchemePercIdent(QObject* p, MSAColorSchemeFactory* f, MAlignmentObject* o) 
: MSAColorScheme(p, f, o)
{
    cacheVersion = 0;
    objVersion = 1;
    mask4[0]=81;
    mask4[1]=61;
    mask4[2]=41;
    mask4[3]=25;

    colorsByRange[0] = QColor("#6464FF");
    colorsByRange[1] = QColor("#9999FF");
    colorsByRange[2] = QColor("#CCCCFF");
    colorsByRange[3] = QColor();

    connect(maObj, SIGNAL(si_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)), 
                  SLOT(sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)));
}

QColor MSAColorSchemePercIdent::getColor(int seq, int pos) {
    updateCache();
    char c = maObj->getMAlignment().charAt(seq, pos);
    if (c == MAlignment_GapChar) {
        return QColor();
    }
    quint32 packedVal = indentCache[pos];
    MSAUtils::unpackConsensusCharsFromInt(packedVal, tmpChars, tmpRanges);
    for (int i=0; i < 4; i++) {
        if (c == tmpChars[i]) {
            int range = tmpRanges[i];
            return colorsByRange[range];
        }
    }
    return QColor();
}

void MSAColorSchemePercIdent::updateCache() {
    if (cacheVersion == objVersion) {
        return;
    }
    const MAlignment& ma = maObj->getMAlignment();
    int aliLen = ma.getLength();
    indentCache.resize(aliLen);
    for (int i=0; i < aliLen; i++) {
        indentCache[i] = MSAUtils::packConsensusCharsToInt(ma, i, mask4, true);
    }
    cacheVersion = objVersion;
}

/// CLUSTAL
MSAColorSchemeClustalX::MSAColorSchemeClustalX(QObject* p, MSAColorSchemeFactory* f, MAlignmentObject* maObj) 
: MSAColorScheme(p, f, maObj)
{
    objVersion = 1;
    cacheVersion = 0;
    aliLen = maObj->getMAlignment().getLength();

    colorByIdx[ClustalColor_BLUE]    = "#80a0f0";
    colorByIdx[ClustalColor_RED]     = "#f01505";
    colorByIdx[ClustalColor_GREEN]   = "#15c015";
    colorByIdx[ClustalColor_PINK]    = "#f08080";
    colorByIdx[ClustalColor_MAGENTA] = "#c048c0";
    colorByIdx[ClustalColor_ORANGE]  = "#f09048";
    colorByIdx[ClustalColor_CYAN]    = "#15a4a4";
    colorByIdx[ClustalColor_YELLOW]  = "#c0c000";

    connect(maObj, SIGNAL(si_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)), 
        SLOT(sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)));
}

QColor MSAColorSchemeClustalX::getColor(int seq, int pos) {
    if (cacheVersion!=objVersion) {
        updateCache();
    }
    int idx = getColorIdx(seq, pos);
    assert(idx >=0 && idx < ClustalColor_NUM_COLORS);
    return colorByIdx[idx];
}

int MSAColorSchemeClustalX::getColorIdx(int seq, int pos) {
    bool low = false;
    int cacheIdx = getCacheIdx(seq, pos, low);
    quint8 val = colorsCache[cacheIdx];
    int colorIdx = low ? val & 0x0F : (val & 0xF0) >> 4;
    assert(colorIdx >=0 && colorIdx < ClustalColor_NUM_COLORS);
    return colorIdx;
}

void MSAColorSchemeClustalX::setColorIdx(int seq, int pos, int colorIdx) {
    assert(colorIdx >=0 && colorIdx < ClustalColor_NUM_COLORS);
    bool low = false;
    int cacheIdx = getCacheIdx(seq, pos, low);
    quint8 val = colorsCache[cacheIdx];
    if (low) {
        val = (val & 0xF0) | colorIdx;
    } else {
        val = (val & 0x0F) | (colorIdx << 4);
    }
    colorsCache[cacheIdx] = val; 
}

static int basesContent(const int* freqs, const char* str, int len) {
    int res = 0;
    for (int i=0; i < len; i++) {
        uchar c = str[i];
        res +=freqs[c];
    }
    return res;
}

void MSAColorSchemeClustalX::updateCache() {
    if (cacheVersion == objVersion) {
        return;
    }
    // compute colors for whole ali
    // use 4 bits per color
    const MAlignment& ma = maObj->getMAlignment();
    int nSeq = ma.getNumSequences();
    aliLen = maObj->getMAlignment().getLength();
    cacheVersion = objVersion;

    bool stub = false;
    int cacheSize = getCacheIdx(nSeq, aliLen, stub) + 1;
    colorsCache.resize(cacheSize);

    /*  source: http://ekhidna.biocenter.helsinki.fi/pfam2/clustal_colours

        BLUE
            (W,L,V,I,M,F):  {50%, P}{60%, WLVIMAFCYHP}
            (A):            {50%, P}{60%, WLVIMAFCYHP}{85%, T,S,G}
            (C):            {50%, P}{60%, WLVIMAFCYHP}{85%, S}
        RED
            (K,R):          {60%, KR}{85%, Q}
        GREEN
            (T):            {50%, TS}{60%, WLVIMAFCYHP}
            (S):            {50%, TS}{80%, WLVIMAFCYHP}
            (N):            {50%, N}{85%, D}
            (Q):            {50%, QE}{60%, KR}
        PINK
            (C):            {85%, C}
        MAGENTA
            (D):            {50%, DE,N}
            (E):            {50%, DE,QE}
        ORANGE
            (G):            {ALWAYS}
        CYAN
            (H,Y):          {50%, P}{60%, WLVIMAFCYHP}
        YELLOW
            (P):            {ALWAYS}
    
        WARN: do not count gaps in percents!
    */


    QVector<int> freqsByChar(256);
    const int* freqs = freqsByChar.data();
    
    for (int pos = 0; pos < aliLen; pos++) {
        int nonGapChars = 0;
        MSAUtils::getColumnFreqs(ma, pos, freqsByChar, nonGapChars);
        int content50 = int(nonGapChars * 50.0 / 100);
        int content60 = int(nonGapChars * 60.0 / 100);
        int content80 = int(nonGapChars * 80.0 / 100);
        int content85 = int(nonGapChars * 85.0 / 100);

        for (int seq = 0; seq < nSeq; seq++) {
            char c = ma.charAt(seq, pos);
            int colorIdx = ClustalColor_NO_COLOR;
            switch(c) {
                case 'W': //(W,L,V,I,M,F): {50%, P}{60%, WLVIMAFCYHP} -> BLUE
                case 'L':
                case 'V':
                case 'I':
                case 'M':
                case 'F':
                    if (freqs['P'] > content50 || basesContent(freqs, "WLVIMAFCYHP", 11) > content60) {
                        colorIdx = ClustalColor_BLUE;
                    }
                    break;
                case 'A': // {50%, P}{60%, WLVIMAFCYHP}{85%, T,S,G} -> BLUE
                    if (freqs['P'] > content50 || basesContent(freqs, "WLVIMAFCYHP", 11) > content60) {
                        colorIdx = ClustalColor_BLUE;
                    } else if (freqs['T'] > content85 || freqs['S'] > content85 || freqs['G']>85) {
                        colorIdx = ClustalColor_BLUE;
                    }
                    break;

                case 'K': //{60%, KR}{85%, Q} -> RED
                case 'R': 
                    if ((freqs['K'] + freqs['R'] > content60) || freqs['Q'] > content85) {
                        colorIdx = ClustalColor_RED;
                    }
                    break;
                
                case 'T': // {50%, TS}{60%, WLVIMAFCYHP} -> GREEN
                    if ((freqs['T'] + freqs['S'] > content50) || basesContent(freqs, "WLVIMAFCYHP", 11) > content60) {
                        colorIdx = ClustalColor_GREEN;
                    }
                    break;
                
                case 'S': // {50%, TS}{80%, WLVIMAFCYHP} -> GREEN
                    if ((freqs['T'] + freqs['S'] > content50) || basesContent(freqs, "WLVIMAFCYHP", 11) > content80) {
                        colorIdx = ClustalColor_GREEN;
                    }
                    break;

                case 'N': // {50%, N}{85%, D} -> GREEN
                    if (freqs['N'] > content50 || freqs['D'] > content85) {
                        colorIdx = ClustalColor_GREEN;
                    }
                    break;

                case 'Q': // {50%, QE}{60%, KR} -> GREEN
                    if ((freqs['Q'] + freqs['E']) > content50 || (freqs['K'] + freqs['R']) > content60) {
                        colorIdx = ClustalColor_GREEN;
                    }
                    break;

                case 'C': //{85%, C} -> PINK
                          //{50%, P}{60%, WLVIMAFCYHP}{85%, S} -> BLUE
                    if (freqs['C'] > content85) {
                        colorIdx = ClustalColor_PINK;
                    } else if (freqs['P'] > content50 || basesContent(freqs, "WLVIMAFCYHP", 11) > content60 || freqs['S'] > content85) {
                        colorIdx = ClustalColor_BLUE;
                    }
                    break;                          

                case 'D': //{50%, DE,N} -> MAGENTA
                    if ((freqs['D'] + freqs['E']) > content50 || freqs['N'] > content50) {
                        colorIdx = ClustalColor_MAGENTA;
                    }
                    break;
                case 'E': //{50%, DE,QE} -> MAGENTA
                    if ((freqs['D'] + freqs['E']) > content50 || (freqs['Q'] + freqs['E']) > content50) {
                        colorIdx = ClustalColor_MAGENTA;
                    }
                    break;
                case 'G': //{ALWAYS} -> ORANGE
                    colorIdx = ClustalColor_ORANGE;
                    break;

                case 'H': // {50%, P}{60%, WLVIMAFCYHP} -> CYAN 
                case 'Y':
                    if (freqs['P'] > content50 || basesContent(freqs, "WLVIMAFCYHP", 11) > content60) {
                        colorIdx = ClustalColor_CYAN;
                    }
                    break;

                case 'P': //{ALWAYS} -> YELLOW
                    colorIdx = ClustalColor_YELLOW;
                    break;
                default: 
                    break;

            }
            setColorIdx(seq, pos, colorIdx);
        }
    }
}


//////////////////////////////////////////////////////////////////////////
// registry
MSAColorSchemeRegistry::MSAColorSchemeRegistry() {
    initBuiltInSchemes();
}

QList<MSAColorSchemeFactory*> MSAColorSchemeRegistry::getMSAColorSchemes(DNAAlphabetType atype) const {
    QList<MSAColorSchemeFactory*> res;
    foreach(MSAColorSchemeFactory* f, colorers) {
        if (f->getAlphabetType() == atype) {
            res.append(f);
        }
    }
    return res;
}

MSAColorSchemeFactory* MSAColorSchemeRegistry::getMSAColorSchemeFactoryById(const QString& id) const {
    foreach(MSAColorSchemeFactory* csf, colorers) {
        if(csf->getId() == id) {
            return csf;
        }
    }
    return NULL;
}

static bool compareNames(const MSAColorSchemeFactory* a1, const MSAColorSchemeFactory* a2) {
    if (a1->getId() == MSAColorScheme::EMPTY_NUCL) {
        return true;
    }
    if (a2->getId() == MSAColorScheme::EMPTY_NUCL) {
        return false;
    }
    if (a1->getId() == MSAColorScheme::EMPTY_AMINO) {
        return true;
    }
    if (a2->getId() == MSAColorScheme::EMPTY_AMINO) {
        return false;
    }
    return a1->getName() < a2->getName();
}

void MSAColorSchemeRegistry::addMSAColorSchemeFactory(MSAColorSchemeFactory* csf) {
    assert(getMSAColorSchemeFactoryById(csf->getId()) == NULL);
    colorers.append(csf);
    qStableSort(colorers.begin(), colorers.end(), compareNames);
}

static void fillEmptyCS(QVector<QColor>& colorsPerChar) {
    colorsPerChar.fill(QColor(), 256);
}

static void fillLightColorsCS(QVector<QColor>& colorsPerChar) {
    for (int i = 0; i < 256; i++) {
        colorsPerChar[i] = GUIUtils::genLightColor(QString((char)i));
    }
    colorsPerChar[MAlignment_GapChar] = QColor(); //invalid color -> no color at all
}

#define SET_C(ch, cl) colorsPerChar[ch]=colorsPerChar[ch+('a'-'A')]=cl

static void addUGENEAmino(QVector<QColor>& colorsPerChar) {
    //amino groups: "KRH", "GPST", "FWY", "ILM"
    QColor krh("#FFEE00");
    SET_C('K', krh);
    SET_C('R', krh.darker(120));
    SET_C('H', krh.lighter(120));

    QColor gpst("#FF5082");
    SET_C('G', gpst);
    SET_C('P', gpst.darker(120));
    SET_C('S', gpst.lighter(120));
    SET_C('T', gpst.lighter(150));

    QColor fwy("#3DF490");
    SET_C('F', fwy);
    SET_C('W', fwy.darker(120));
    SET_C('Y', fwy.lighter(120));

    QColor ilm("#00ABED");
    SET_C('I', ilm);
    SET_C('L', ilm.darker(120));
    SET_C('M', ilm.lighter(120));

    //fix some color overlaps:
    //e looks like q by default
    SET_C('E', "#C0BDBB"); //gray
    SET_C('X', "#FCFCFC");
}


static void addUGENENucl(QVector<QColor>& colorsPerChar) {
    SET_C('A', "#FCFF92"); // yellow
    SET_C('C', "#70F970"); // green
    SET_C('T', "#FF99B1"); // light red
    SET_C('G', "#4EADE1"); // light blue
    SET_C('U', colorsPerChar['T'].lighter(120));
    SET_C('N', "#FCFCFC");
}

//TODO: check extended AMINO chars!!! (O/B/U?)

static void addZappoAmino(QVector<QColor>& colorsPerChar) {
    //Aliphatic/hydrophobic:    ILVAM       #ffafaf
    SET_C('I', "#ffafaf"); 
    SET_C('L', "#ffafaf"); 
    SET_C('V', "#ffafaf"); 
    SET_C('A', "#ffafaf"); 
    SET_C('M', "#ffafaf"); 

    //Aromatic:  FWY         #ffc800
    SET_C('F', "#ffc800"); 
    SET_C('W', "#ffc800"); 
    SET_C('Y', "#ffc800"); 

    //Positive   KRH         #6464ff
    SET_C('K', "#6464ff"); 
    SET_C('R', "#6464ff"); 
    SET_C('H', "#6464ff"); 

    //Negative   DE          #ff0000
    SET_C('D', "#ff0000"); 
    SET_C('E', "#ff0000"); 
    
    //Hydrophil  STNQ        #00ff00
    SET_C('S', "#00ff00"); 
    SET_C('T', "#00ff00"); 
    SET_C('N', "#00ff00"); 
    SET_C('Q', "#00ff00"); 
    
    //conformat  PG          #ff00ff
    SET_C('P', "#ff00ff"); 
    SET_C('G', "#ff00ff"); 

    //Cysteine   C           #ffff00
    SET_C('C', "#ffff00"); 
}

static void addTailorAmino(QVector<QColor>& colorsPerChar) {
    SET_C('A', "#ccff00"); 
    SET_C('V', "#99ff00"); 
    SET_C('I', "#66ff00"); 
    SET_C('L', "#33ff00"); 
    SET_C('M', "#00ff00"); 
    SET_C('F', "#00ff66"); 
    SET_C('Y', "#00ffcc"); 
    SET_C('W', "#00ccff"); 
    SET_C('H', "#0066ff"); 
    SET_C('R', "#0000ff"); 
    SET_C('K', "#6600ff"); 
    SET_C('N', "#cc00ff"); 
    SET_C('Q', "#ff00cc"); 
    SET_C('E', "#ff0066"); 
    SET_C('D', "#ff0000"); 
    SET_C('S', "#ff3300"); 
    SET_C('T', "#ff6600"); 
    SET_C('G', "#ff9900"); 
    SET_C('P', "#ffcc00"); 
    SET_C('C', "#ffff00"); 
}

static void addHydroAmino(QVector<QColor>& colorsPerChar) {
//The most hydrophobic residues according to this table are colored red and the most hydrophilic ones are colored blue.
    SET_C('I', "#ff0000"); 
    SET_C('V', "#f60009"); 
    SET_C('L', "#ea0015"); 
    SET_C('F', "#cb0034"); 
    SET_C('C', "#c2003d"); 
    SET_C('M', "#b0004f"); 
    SET_C('A', "#ad0052"); 
    SET_C('G', "#6a0095"); 
    SET_C('X', "#680097"); 
    SET_C('T', "#61009e"); 
    SET_C('S', "#5e00a1"); 
    SET_C('W', "#5b00a4"); 
    SET_C('Y', "#4f00b0"); 
    SET_C('P', "#4600b9"); 
    SET_C('H', "#1500ea"); 
    SET_C('E', "#0c00f3"); 
    SET_C('Z', "#0c00f3"); 
    SET_C('Q', "#0c00f3"); 
    SET_C('D', "#0c00f3"); 
    SET_C('B', "#0c00f3"); 
    SET_C('N', "#0c00f3"); 
    SET_C('K', "#0000ff"); 
    SET_C('R', "#0000ff"); 
}

static void addHelixAmino(QVector<QColor>& colorsPerChar) {
    SET_C('E', "#ff00ff"); 
    SET_C('M', "#ef10ef"); 
    SET_C('A', "#e718e7"); 
    SET_C('Z', "#c936c9"); 
    SET_C('L', "#ae51ae"); 
    SET_C('K', "#a05fa0"); 
    SET_C('F', "#986798"); 
    SET_C('Q', "#926d92"); 
    SET_C('I', "#8a758a"); 
    SET_C('W', "#8a758a"); 
    SET_C('V', "#857a85"); 
    SET_C('D', "#778877"); 
    SET_C('X', "#758a75"); 
    SET_C('H', "#758a75"); 
    SET_C('R', "#6f906f"); 
    SET_C('B', "#49b649"); 
    SET_C('T', "#47b847"); 
    SET_C('S', "#36c936"); 
    SET_C('C', "#23dc23"); 
    SET_C('Y', "#21de21"); 
    SET_C('N', "#1be41b"); 
    SET_C('G', "#00ff00"); 
    SET_C('P', "#00ff00"); 
}

static void addStrandAmino(QVector<QColor>& colorsPerChar) {
    SET_C('V', "#ffff00"); 
    SET_C('I', "#ecec13"); 
    SET_C('Y', "#d3d32c"); 
    SET_C('F', "#c2c23d"); 
    SET_C('W', "#c0c03f"); 
    SET_C('L', "#b2b24d"); 
    SET_C('T', "#9d9d62"); 
    SET_C('C', "#9d9d62"); 
    SET_C('Q', "#8c8c73"); 
    SET_C('M', "#82827d"); 
    SET_C('X', "#797986"); 
    SET_C('R', "#6b6b94"); 
    SET_C('N', "#64649b"); 
    SET_C('H', "#60609f"); 
    SET_C('A', "#5858a7"); 
    SET_C('S', "#4949b6"); 
    SET_C('G', "#4949b6"); 
    SET_C('Z', "#4747b8"); 
    SET_C('K', "#4747b8"); 
    SET_C('B', "#4343bc"); 
    SET_C('P', "#2323dc"); 
    SET_C('D', "#2121de"); 
    SET_C('E', "#0000ff"); 
}

static void addTurnAmino(QVector<QColor>& colorsPerChar) {
    SET_C('N', "#ff0000"); 
    SET_C('G', "#ff0000"); 
    SET_C('P', "#f60909"); 
    SET_C('B', "#f30c0c"); 
    SET_C('D', "#e81717"); 
    SET_C('S', "#e11e1e"); 
    SET_C('C', "#a85757"); 
    SET_C('Y', "#9d6262"); 
    SET_C('K', "#7e8181"); 
    SET_C('X', "#7c8383"); 
    SET_C('Q', "#778888"); 
    SET_C('W', "#738c8c"); 
    SET_C('T', "#738c8c"); 
    SET_C('R', "#708f8f"); 
    SET_C('H', "#708f8f"); 
    SET_C('Z', "#5ba4a4"); 
    SET_C('E', "#3fc0c0"); 
    SET_C('A', "#2cd3d3"); 
    SET_C('F', "#1ee1e1"); 
    SET_C('M', "#1ee1e1"); 
    SET_C('L', "#1ce3e3"); 
    SET_C('V', "#07f8f8"); 
    SET_C('I', "#00ffff"); 
}

static void addBuriedAmino(QVector<QColor>& colorsPerChar) {
    SET_C('C', "#0000ff"); 
    SET_C('I', "#0054ab"); 
    SET_C('V', "#005fa0"); 
    SET_C('L', "#007b84"); 
    SET_C('F', "#008778"); 
    SET_C('M', "#009768"); 
    SET_C('G', "#009d62"); 
    SET_C('A', "#00a35c"); 
    SET_C('W', "#00a857"); 
    SET_C('X', "#00b649"); 
    SET_C('S', "#00d52a"); 
    SET_C('H', "#00d52a"); 
    SET_C('T', "#00db24"); 
    SET_C('P', "#00e01f"); 
    SET_C('Y', "#00e619"); 
    SET_C('N', "#00eb14"); 
    SET_C('B', "#00eb14"); 
    SET_C('D', "#00eb14"); 
    SET_C('Q', "#00f10e"); 
    SET_C('Z', "#00f10e"); 
    SET_C('E', "#00f10e"); 
    SET_C('R', "#00fc03"); 
    SET_C('K', "#00ff00"); 
}

static void addJalviewNucl(QVector<QColor>& colorsPerChar) {
    SET_C('A', "#64F73F"); 
    SET_C('C', "#FFB340"); 
    SET_C('G', "#EB413C"); 
    SET_C('T', "#3C88EE"); 
    SET_C('U', colorsPerChar['T'].lighter(105)); 
}

//SET_C('', "#"); 

QString MSAColorScheme::EMPTY_NUCL      = "COLOR_SCHEME_EMPTY_NUCL";
QString MSAColorScheme::UGENE_NUCL      = "COLOR_SCHEME_UGENE_NUCL";
QString MSAColorScheme::JALVIEW_NUCL    = "COLOR_SCHEME_JALVIEW_NUCL";
QString MSAColorScheme::IDENTPERC_NUCL  = "COLOR_SCHEME_IDENTPERC_NUCL";

QString MSAColorScheme::EMPTY_AMINO     = "COLOR_SCHEME_EMPTY_AMINO";
QString MSAColorScheme::UGENE_AMINO     = "COLOR_SCHEME_UGENE_AMINO";
QString MSAColorScheme::ZAPPO_AMINO     = "COLOR_SCHEME_ZAPPO_AMINO";
QString MSAColorScheme::TAILOR_AMINO    = "COLOR_SCHEME_TAILOR_AMINO";
QString MSAColorScheme::HYDRO_AMINO     = "COLOR_SCHEME_HYDRO_AMINO";
QString MSAColorScheme::HELIX_AMINO     = "COLOR_SCHEME_HELIX_AMINO";
QString MSAColorScheme::STRAND_AMINO    = "COLOR_SCHEME_STRAND_AMINO";
QString MSAColorScheme::TURN_AMINO      = "COLOR_SCHEME_TURN_AMINO";
QString MSAColorScheme::BURIED_AMINO    = "COLOR_SCHEME_BURIED_AMINO";
QString MSAColorScheme::IDENTPERC_AMINO = "COLOR_SCHEME_IDENTPERC_AMINO";
QString MSAColorScheme::CLUSTALX_AMINO  = "COLOR_SCHEME_CLUSTALX_AMINO";

void MSAColorSchemeRegistry::initBuiltInSchemes() {
    QVector<QColor> colorsPerChar;

    //nucleic
    fillEmptyCS(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::EMPTY_NUCL, tr("No colors"), DNAAlphabet_NUCL, colorsPerChar));

    fillLightColorsCS(colorsPerChar);
    addUGENENucl(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::UGENE_NUCL, tr("UGENE"), DNAAlphabet_NUCL, colorsPerChar));

    fillEmptyCS(colorsPerChar);
    addJalviewNucl(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::JALVIEW_NUCL, tr("Jalview"), DNAAlphabet_NUCL, colorsPerChar));

    addMSAColorSchemeFactory(new MSAColorSchemePercIdentFactory(this, MSAColorScheme::IDENTPERC_NUCL,  tr("Percentage Identity"), DNAAlphabet_NUCL));

    //amino
    fillEmptyCS(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::EMPTY_AMINO, tr("No colors"), DNAAlphabet_AMINO, colorsPerChar));

    fillLightColorsCS(colorsPerChar);
    addUGENEAmino(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::UGENE_AMINO, tr("UGENE"), DNAAlphabet_AMINO, colorsPerChar));
    
    fillEmptyCS(colorsPerChar);
    addZappoAmino(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::ZAPPO_AMINO, tr("Zappo"), DNAAlphabet_AMINO, colorsPerChar));

    fillEmptyCS(colorsPerChar);
    addTailorAmino(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::TAILOR_AMINO, tr("Tailor"), DNAAlphabet_AMINO, colorsPerChar));

    fillEmptyCS(colorsPerChar);
    addHydroAmino(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::HYDRO_AMINO, tr("Hydrophobicity"), DNAAlphabet_AMINO, colorsPerChar));

    fillEmptyCS(colorsPerChar);
    addHelixAmino(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::HELIX_AMINO, tr("Helix propensity"), DNAAlphabet_AMINO, colorsPerChar));

    fillEmptyCS(colorsPerChar);
    addStrandAmino(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::STRAND_AMINO, tr("Strand propensity"), DNAAlphabet_AMINO, colorsPerChar));

    fillEmptyCS(colorsPerChar);
    addTurnAmino(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::TURN_AMINO, tr("Turn propensity"), DNAAlphabet_AMINO, colorsPerChar));

    fillEmptyCS(colorsPerChar);
    addBuriedAmino(colorsPerChar);
    addMSAColorSchemeFactory(new MSAColorSchemeStaticFactory(this, MSAColorScheme::BURIED_AMINO, tr("Buried index"), DNAAlphabet_AMINO, colorsPerChar));

    addMSAColorSchemeFactory(new MSAColorSchemePercIdentFactory(this, MSAColorScheme::IDENTPERC_AMINO, tr("Percentage Identity"), DNAAlphabet_AMINO));
    
    addMSAColorSchemeFactory(new MSAColorSchemeClustalXFactory(this, MSAColorScheme::CLUSTALX_AMINO,  tr("Clustal X"), DNAAlphabet_AMINO));
}

}//namespace
