/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RepeatWorker.h"
#include "FindRepeatsDialog.h"

#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>

#include <datatype/DNASequence.h>
#include <core_api/DNATranslation.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <core_api/GObjectReference.h>

#include <QtGui/QApplication>

/* TRANSLATOR GB2::LocalWorkflow::RepeatWorker */

namespace GB2 {
namespace LocalWorkflow {

static const QString IN_PORT("in");
static const QString OUT_PORT("out");

static const QString NAME_ATTR("1name");
static const QString LEN_ATTR("2minlen");
static const QString IDENTITY_ATTR("3identity");
static const QString MIN_DIST_ATTR("4mindist");
static const QString MAX_DIST_ATTR("5maxdist");
static const QString INVERT_ATTR("6invert");
static const QString NESTED_ATTR("7nested");
static const QString ALGO_ATTR("8algo");
static const QString THREADS_ATTR("9threads");

const QString RepeatWorkerFactory::ACTOR_ID("repeat.finder");

static LogCategory log(ULOG_CAT_WD);

void RepeatWorkerFactory::init() {
    QList<PortDescriptor*> p; QList<Attribute*> a;

    {
        Descriptor id(IN_PORT, RepeatWorker::tr("Input sequences"), RepeatWorker::tr("A nucleotide sequence to search repeats in."));
        Descriptor od(OUT_PORT, RepeatWorker::tr("Repeat annotations"), RepeatWorker::tr("A set of annotations marking repeats found in the sequence."));
        p << new PortDescriptor(id, BioDataTypes::DNA_SEQUENCE_TYPE(), true /*input*/);
        p << new PortDescriptor(od, BioDataTypes::ANNOTATION_TABLE_TYPE(), false /*input*/, true /*multi*/);
    }
    {
        Descriptor nd(NAME_ATTR, RepeatWorker::tr("Annotate as"), RepeatWorker::tr("Name of the result annotations marking found repeats."));
        Descriptor idd(IDENTITY_ATTR, RepeatWorker::tr("Identity"), RepeatWorker::tr("Repeats identity."));
        Descriptor ld(LEN_ATTR, RepeatWorker::tr("Min length"), RepeatWorker::tr("Minimum length of repeats."));
        Descriptor mid(MIN_DIST_ATTR, RepeatWorker::tr("Min distance"), RepeatWorker::tr("Minimum distance between repeats."));
        Descriptor mad(MAX_DIST_ATTR, RepeatWorker::tr("Max distance"), RepeatWorker::tr("Maximum distance between repeats."));
        Descriptor ind(INVERT_ATTR, RepeatWorker::tr("Inverted"), RepeatWorker::tr("Search for inverted repeats."));
        Descriptor nsd(NESTED_ATTR, RepeatWorker::tr("Filter nested"), RepeatWorker::tr("Filter nested repeats."));
        Descriptor ald(ALGO_ATTR, RepeatWorker::tr("Algorithm"), RepeatWorker::tr("Control over variations of algorithm."));
        Descriptor thd(THREADS_ATTR, RepeatWorker::tr("Parallel threads"), RepeatWorker::tr("Number of parallel threads used for the task."));

        FindRepeatsTaskSettings cfg = FindRepeatsDialog::defaultSettings();
        a << new Attribute(nd, CoreDataTypes::STRING_TYPE(), true, "repeat_unit");
        a << new Attribute(ld, CoreDataTypes::NUM_TYPE(), false, cfg.minLen);
        a << new Attribute(idd, CoreDataTypes::NUM_TYPE(), false, cfg.getIdentity());
        a << new Attribute(mid, CoreDataTypes::NUM_TYPE(), false, cfg.minDist);
        a << new Attribute(mad, CoreDataTypes::NUM_TYPE(), false, cfg.maxDist);
        a << new Attribute(ind, CoreDataTypes::BOOL_TYPE(), false, cfg.inverted);
        a << new Attribute(nsd, CoreDataTypes::BOOL_TYPE(), false, cfg.filterNested);
        a << new Attribute(ald, CoreDataTypes::NUM_TYPE(), false, cfg.algo);
        a << new Attribute(thd, CoreDataTypes::NUM_TYPE(), false, cfg.nThreads);
    }

    Descriptor desc(ACTOR_ID, RepeatWorker::tr("Find repeats"), 
        RepeatWorker::tr("Finds repeats in each supplied sequence, stores found regions as annotations.")
        );
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;    
    {
        QVariantMap m; m["minimum"] = 0; m["maximum"] = INT_MAX; m["suffix"] = RepeatWorker::tr(" bp");
        delegates[MIN_DIST_ATTR] = new SpinBoxDelegate(m);
        m["specialValueText"] = RepeatWorker::tr("Any");
        delegates[MAX_DIST_ATTR] = new SpinBoxDelegate(m);
        m["minimum"] = 2;
        delegates[LEN_ATTR] = new SpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = 50; m["maximum"] = 100; m["suffix"] = "%";
        delegates[IDENTITY_ATTR] = new SpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["specialValueText"] = "Auto";
        delegates[THREADS_ATTR] = new SpinBoxDelegate(m);
    }
    {
        QVariantMap m; 
        m["Auto"] = RFAlgorithm_Auto;
        m["Diagonals"] = RFAlgorithm_Diagonal;
        m["Suffix index"] = RFAlgorithm_Suffix;
        delegates[ALGO_ATTR] = new ComboBoxDelegate(m);
    }

    proto->setPrompter(new RepeatPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":repeat_finder/images/repeats.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_BASIC(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new RepeatWorkerFactory());
}

QString RepeatPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(IN_PORT));
    Actor* producer = input->getProducer(IN_PORT);
    QString producerName = producer ? tr(" from %1").arg(producer->getLabel()) : "";

    // TODO extend ?
    QString resultName = getRequiredParam(NAME_ATTR);
    QString inverted = getParameter(INVERT_ATTR).toBool() ? tr("inverted") : tr("direct");

    QString doc = tr("For each sequence<u>%1</u>, find <u>%2</u> repeats."
        "<br>Detect <u>%3% identical</u> repeats <u>not shorter than %4 bps</u>."
        "<br>Output the list of found regions annotated as <u>%5</u>.")
        .arg(producerName) //sequence from Read Fasta 1
        .arg(inverted) 
        .arg(getParameter(IDENTITY_ATTR).toInt())
        .arg(getParameter(LEN_ATTR).toInt())
        .arg(resultName);
    
    return doc;
}

void RepeatWorker::init() {
    input = ports.value(IN_PORT);
    output = ports.value(OUT_PORT);
    cfg.algo = RFAlgorithm(actor->getParameter(ALGO_ATTR)->value.toInt());
    cfg.minLen = actor->getParameter(LEN_ATTR)->value.toInt();
    cfg.minDist = actor->getParameter(MIN_DIST_ATTR)->value.toInt();
    cfg.maxDist = actor->getParameter(MAX_DIST_ATTR)->value.toInt();
    cfg.setIdentity(actor->getParameter(IDENTITY_ATTR)->value.toInt());
    cfg.nThreads = actor->getParameter(THREADS_ATTR)->value.toInt();
    cfg.inverted = actor->getParameter(INVERT_ATTR)->value.toBool();
    cfg.filterNested = actor->getParameter(NESTED_ATTR)->value.toBool();
    resultName = actor->getParameter(NAME_ATTR)->value.toString();
}

bool RepeatWorker::isReady() {
    return (input && input->hasMessage());
}

Task* RepeatWorker::tick() {
    DNASequence seq = input->get().getData().value<DNASequence>();
    Task* t = new FindRepeatsToAnnotationsTask(cfg, seq, resultName, QString(), GObjectReference());
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void RepeatWorker::sl_taskFinished() {
    FindRepeatsToAnnotationsTask* t = qobject_cast<FindRepeatsToAnnotationsTask*>(sender());
    if (t->getState() != Task::State_Finished || t->hasErrors()) return;
    if (output) {
        const QList<SharedAnnotationData>& res = t->importAnnotations();
        QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(res);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
        if (input->isEnded()) {
            output->setEnded();
        }
        log.info(tr("Found %1 repeats").arg(res.size()));
    }
}

bool RepeatWorker::isDone() {
    return !input || input->isEnded();
}

} //namespace LocalWorkflow
} //namespace GB2
