/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GScriptRunner.h"

#include <core_api/Settings.h>
#include <core_api/AppContext.h>
#include <core_api/Task.h>
#include <core_api/Log.h>
#include <core_api/ProjectModel.h>
#include <core_api/MainWindow.h>
#include "GScriptModuleRegistry.h"
#include <QThread>
#include <QtGui/QMainWindow>
#include <QtGui/QApplication>
#include <QTimer>

#include <QtScript>

namespace GB2 {

GScriptRunner::GScriptRunner(QString _scriptFileName)
: scriptFileName(_scriptFileName)
{
//TODO: complete redesign required!
    connect(AppContext::getPluginSupport(),SIGNAL(si_allStartUpPluginsLoaded()),this,SLOT(sl_runScript()));
    script=parseScriptFile(scriptFileName);
    checkStateTimer=new QTimer(this);
    connect(checkStateTimer, SIGNAL(timeout()), this, SLOT(sl_checkStateScriptEngine()));
}

void GScriptRunner::sl_runScript(){
    runScript(scriptFileName);
}
void GScriptRunner::sl_runSerialScript()
{
    runEngine=new QScriptEngine(this);
    foreach(QString element, script->bindingsList){
        if(!GScriptRunner::setupBindings(runEngine,element))
            qDebug() <<(QString("Binding %1 not found").arg(element));
    }
    //////////////////////////////////////////////////////////////////////////
    //for refactoring...
    //main_window
    QMainWindow* mw = AppContext::getMainWindow()->getQMainWindow();
    assert(mw!=NULL);

    QScriptValue scriptMainWindow = runEngine->newQObject((QObject*)mw);
    runEngine->globalObject().setProperty("main_window", scriptMainWindow);

    //jsTestHelperLib...
    QFile* jsTestHelperLib=new QFile();
    jsTestHelperLib->setFileName(QApplication::applicationDirPath()+"/scripts/libTestHelper.js");
    if(!jsTestHelperLib->exists()) {
        jsTestHelperLib->setFileName(QApplication::applicationDirPath()+"/../../scripts/libTestHelper.js");
    }
    assert(jsTestHelperLib->exists());

    jsTestHelperLib->open(QIODevice::ReadOnly | QIODevice::Text);
    runEngine->evaluate(jsTestHelperLib->readAll());
    jsTestHelperLib->close();
    //end of refactoring...
    //////////////////////////////////////////////////////////////////////////
    runEngine->setProcessEventsInterval(100);
    checkStateTimer->start(100); 
    runEngine->evaluate(script->text);
}
void GScriptRunner::sl_checkStateScriptEngine(){
    if(runEngine!=NULL){
        if((runEngine->hasUncaughtException())&&(script->exitOnException)){
            //runEngine have uncaught exception
            runEngine->clearExceptions();
            exit(1);
        }
        if((!runEngine->isEvaluating())&&(script->exitOnEnding)){
            //runEngine end evaluating
            exit(0);
        }
        if((runEngine->hasUncaughtException())&&(!script->exitOnException)){
            //runEngine have uncaught exception
            runEngine->clearExceptions();
            delete runEngine;
            checkStateTimer->stop(); 
        }
        if((!runEngine->isEvaluating())&&(!script->exitOnEnding)){
            //runEngine end evaluating
            delete runEngine;
            checkStateTimer->stop(); 
        }

    }
}

void GScriptRunner::runScript(QString scriptFileName){

    if(script!=NULL){
    Task* cst=createScriptTask(script);
    AppContext::getTaskScheduler()->registerTopLevelTask(cst);
    }
}

Task* GScriptRunner::createScriptTask(GScript* script){
    this->script=script;
    return new GScriptTask(script);
}

bool GScriptRunner::setupBindings(QScriptEngine *engine,QString bindingName){
    if (bindingName.contains("qt")){
        engine->importExtension(bindingName);
        if(engine->hasUncaughtException())
            return false;
        else
            return true;
    }else {
        AppContext::getScriptModuleRegistry()->getModuleByName(bindingName)->setup(engine);
        return true;
    }
}
GScript* GScriptRunner::parseScriptFile(QString scriptFileName){
    if (scriptFileName.size()==0){
        return NULL;
    }
    QFileInfo* scriptFileInfo=new QFileInfo(scriptFileName);
    QFile* scriptFile=new QFile(scriptFileInfo->absoluteFilePath());
    if (!scriptFile->exists()){
        return NULL;
    }

    GScript* script=new GScript();
    bool setSeparateThreads=true;
    bool setExitOnException=true;
    bool setExitOnEnding=true;
    scriptFile->open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream in(scriptFile);
    while (!in.atEnd()) {
        QString line = in.readLine();
        if (line.size()!=0){
            if (line.indexOf("//#")==0){//Example: //#include "qt.core"
                if(line.indexOf("//#include")==0){
                    QString including_str=line.split(' ').value(1).split('"').value(1);
                    if (including_str.size()!=0){
                        script->bindingsList.append(including_str);
                    }
                }else if(line.indexOf("//#separateThreads")==0){
                    if(line.split('=').value(1).contains("true")){
                        script->separateThreads=true;
                    }else{
                        script->separateThreads=false;
                    }
                    setSeparateThreads=false;
                }else if(line.indexOf("//#exitOnException")==0){
                    if(line.split('=').value(1).contains("true")){
                        script->exitOnException=true;
                    }else{
                        script->exitOnException=false;
                    }
                    setExitOnException=false;
                }else if(line.indexOf("//#exitOnEnding")==0){
                    if(line.split('=').value(1).contains("true")){
                        script->exitOnEnding=true;
                    }else{
                        script->exitOnEnding=false;
                    }
                    setExitOnEnding=false;
                }
            }else{
                script->text.append(line);
                script->text.append(QString("\n"));
            }
        }
    }
    scriptFile->close();
    if(setSeparateThreads){
        script->separateThreads=false;
    }
    if(setExitOnException){
        script->exitOnException=false;
    }
    if(setExitOnEnding){
        script->exitOnEnding=false;
    }
    return script;
}

//////////////////////////////////////////////////////////////////////////
// Script Task

GScriptTask::GScriptTask(GScript* _script)
: Task("ScriptTask", TaskFlag_None), script(_script)
{
}
void GScriptTask::prepare(){
    runEngine=new QScriptEngine(this);
    if(script->separateThreads){
    foreach(QString element, script->bindingsList){
        if(!GScriptRunner::setupBindings(runEngine,element))
            setError(tr("Binding %1 not found").arg(element));
    }
//////////////////////////////////////////////////////////////////////////
//for refactoring...
//main_window
    QMainWindow* mw = AppContext::getMainWindow()->getQMainWindow();
    assert(mw!=NULL);

    QScriptValue scriptMainWindow = runEngine->newQObject((QObject*)mw);
    runEngine->globalObject().setProperty("main_window", scriptMainWindow);

//jsTestHelperLib...
    QFile* jsTestHelperLib=new QFile();
    jsTestHelperLib->setFileName(QApplication::applicationDirPath()+"/scripts/libTestHelper.js");
    if(!jsTestHelperLib->exists()) {
        jsTestHelperLib->setFileName(QApplication::applicationDirPath()+"/../../scripts/libTestHelper.js");
    }
    assert(jsTestHelperLib->exists());

    jsTestHelperLib->open(QIODevice::ReadOnly | QIODevice::Text);
    runEngine->evaluate(jsTestHelperLib->readAll());
    jsTestHelperLib->close();
//end of refactoring...
//////////////////////////////////////////////////////////////////////////
    runEngine->setProcessEventsInterval(100);
    }
}
void GScriptTask::run(){
    if(script->separateThreads){
        runEngine->evaluate(script->text);
    }
}
Task::ReportResult GScriptTask::report()
{
    if(!script->separateThreads){
        QTimer::singleShot(1000,AppContext::getScriptRunner(), SLOT(sl_runSerialScript()));
    }
    if((runEngine!=NULL)&&(script->separateThreads)){
        if((runEngine->hasUncaughtException())&&(script->exitOnException)){
            //runEngine have uncaught exception
            runEngine->clearExceptions();
            exit(1);
        }
        if(script->exitOnEnding){
            //runEngine end evaluating
            exit(0);
        }
    }
    return ReportResult_Finished;
}
}//namespace
