/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "HMMBuildWorker.h"
#include "HMMIOWorker.h"
#include "HMMBuildDialogController.h"
#include "u_calibrate/HMMCalibrateDialogController.h"

#include <workflow/TypeSet.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_support/DelegateEditors.h>

#include <datatype/MAlignment.h>
#include <core_api/AppContext.h>
#include <core_api/Log.h>

/* TRANSLATOR GB2::LocalWorkflow::HMMBuildWorker */

namespace GB2 {
namespace LocalWorkflow {


const QString HMMBuildWorkerFactory::ACTOR("uhmmer.build");

static LogCategory log(ULOG_CAT_WD);

static const QString IN_PORT("in");
static const QString OUT_PORT("out");


static const QString MODE_ATTR("1mode");
static const QString NAME_ATTR("2name");
// int     nsample;      // number of random seqs to sample
// int     seed;         // random number seed             
// int     fixedlen;     // fixed length, or 0 if unused
// float   lenmean;      // mean of length distribution
// float   lensd;        // std dev of length distribution 

static const QString CALIBRATE_ATTR("3calibrate");
static const QString THREADS_ATTR("4threads");
static const QString FIXEDLEN_ATTR("5fixedlen");
static const QString LENMEAN_ATTR("6lenmean");
static const QString NUM_ATTR("7nsample");
static const QString LENDEV_ATTR("8stddev");
static const QString SEED_ATTR("9seed");


void HMMBuildWorkerFactory::init()
{
    QList<PortDescriptor*> p; QList<Attribute*> a;
    {
        Descriptor id(IN_PORT, HMMBuildWorker::tr("Input MSA"), HMMBuildWorker::tr("Input multiple sequence alignment for building statistical model."));
        Descriptor od(OUT_PORT, HMMBuildWorker::tr("HMM profile"), HMMBuildWorker::tr("Produced HMM profile"));
        p << new PortDescriptor(id, BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), true /*input*/);
        p << new PortDescriptor(od, HMMLib::HMM_PROFILE_TYPE(), false /*input*/, true /*multi*/);
    }

    Descriptor mod(MODE_ATTR, HMMBuildWorker::tr("HMM strategy"), HMMBuildWorker::tr("Specifies kind of alignments you want to allow"));
    Descriptor nad(NAME_ATTR, HMMBuildWorker::tr("Profile name"), HMMBuildWorker::tr("Descriptive name of the HMM profile"));
    // int     nsample;      // number of random seqs to sample
    // int     seed;         // random number seed             
    // int     fixedlen;     // fixed length, or 0 if unused
    // float   lenmean;      // mean of length distribution
    // float   lensd;        // std dev of length distribution 

    Descriptor cad(CALIBRATE_ATTR, HMMBuildWorker::tr("Calibrate profile"), HMMBuildWorker::tr("Enables/disables optional profile calibration." 
        "<p>An empirical HMM calibration costs time but it only has to be done once per model, and can greatly increase the sensitivity of a database search."));
    Descriptor td(THREADS_ATTR, HMMBuildWorker::tr("Parallel calibration"), 
        HMMBuildWorker::tr("Number of parallel threads that the calibration will run in."));
    Descriptor fid(FIXEDLEN_ATTR, HMMBuildWorker::tr("Fixed length of samples"), 
        QApplication::translate("HMMCalibrateDialog", "fixed_tip", 0, QApplication::UnicodeUTF8));
    Descriptor lmd(LENMEAN_ATTR, HMMBuildWorker::tr("Mean length of samples"), 
        QApplication::translate("HMMCalibrateDialog", "mean_tip", 0, QApplication::UnicodeUTF8));
    Descriptor nud(NUM_ATTR, HMMBuildWorker::tr("Number of samples"), 
        QApplication::translate("HMMCalibrateDialog", "num_tip", 0, QApplication::UnicodeUTF8));
    Descriptor ldd(LENDEV_ATTR, HMMBuildWorker::tr("Standard deviation"), 
        QApplication::translate("HMMCalibrateDialog", "sd_tip", 0, QApplication::UnicodeUTF8));
    Descriptor sed(SEED_ATTR, HMMBuildWorker::tr("Random seed"),
        QApplication::translate("HMMCalibrateDialog", "seed_tip", 0, QApplication::UnicodeUTF8));

//     nsample      = 5000;
//     fixedlen     = 0;
//     lenmean      = 325.;
//     lensd        = 200.;
//     seed         = (int) time ((time_t *) NULL);

    a << new Attribute(nad, CoreDataTypes::STRING_TYPE());
    a << new Attribute(cad, CoreDataTypes::BOOL_TYPE(), false, QVariant(true));
    a << new Attribute(mod, CoreDataTypes::NUM_TYPE(), false, QVariant(int(P7_LS_CONFIG)));
    a << new Attribute(nud, CoreDataTypes::NUM_TYPE(), false, QVariant(5000));
    a << new Attribute(sed, CoreDataTypes::NUM_TYPE(), false, QVariant(0));
    a << new Attribute(fid, CoreDataTypes::NUM_TYPE(), false, QVariant(0));
    a << new Attribute(lmd, CoreDataTypes::NUM_TYPE(), false, QVariant(325));
    a << new Attribute(ldd, CoreDataTypes::NUM_TYPE(), false, QVariant(double(200)));
    a << new Attribute(td, CoreDataTypes::NUM_TYPE(), false, QVariant(1));

    Descriptor desc(HMMBuildWorkerFactory::ACTOR, HMMBuildWorker::tr("HMM build"), HMMBuildWorker::tr("Builds a HMM profile from a multiple sequence alignment."
        "<p>The HMM profile is a statistical model which captures position-specific information"
        " about how conserved each column of the alignment is, and which residues are likely."));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;    
    
    {
        QVariantMap lenMap; lenMap["minimum"] = 0; lenMap["maximum"] = INT_MAX;
        delegates[FIXEDLEN_ATTR] = new SpinBoxDelegate(lenMap);
    }
    {
        QVariantMap numMap; numMap["minimum"] = 1; numMap["maximum"] = INT_MAX;
        delegates[NUM_ATTR] = new SpinBoxDelegate(numMap);
    }
    {
        QVariantMap m; m["minimum"] = 0; m["maximum"] = INT_MAX;
        delegates[SEED_ATTR] = new SpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = 1; m["maximum"] = INT_MAX;
        delegates[LENMEAN_ATTR] = new SpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = double(.01); m["maximum"] = double(1000000.00); m["decimals"] = 2;
        delegates[LENDEV_ATTR] = new DoubleSpinBoxDelegate(m);
    }
    {
        QVariantMap m; m["minimum"] = 1; m["maximum"] = 100;
        delegates[THREADS_ATTR] = new SpinBoxDelegate(m);
    }

    QVariantMap modeMap; 
    modeMap["hmms"] = QVariant(P7_BASE_CONFIG);
    modeMap["hmmfs"] = QVariant(P7_FS_CONFIG);
    modeMap[QString("hmmls (%1)").arg(HMMBuildWorker::tr("Default"))] = QVariant(P7_LS_CONFIG);
    modeMap["hmmsw"] = QVariant(P7_SW_CONFIG);
    delegates[MODE_ATTR] = new ComboBoxDelegate(modeMap);

    proto->setEditor(new DelegateEditor(delegates));
    proto->icon = QIcon(":/uhmmer/images/hmmer_16.png");
    proto->setPrompter(new HMMBuildPrompter());
    WorkflowEnv::getProtoRegistry()->registerProto(HMMLib::HMM_CATEGORY(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new HMMBuildWorkerFactory());
}

static bool isDefaultCfg(PrompterBaseImpl* actor) {
    return int(P7_LS_CONFIG) == actor->getParameter(MODE_ATTR).toInt()
        && 5000 == actor->getParameter(NUM_ATTR).toInt()
        && 0 == actor->getParameter(SEED_ATTR).toInt()
        && 0 == actor->getParameter(FIXEDLEN_ATTR).toInt()
        && 325 == actor->getParameter(LENMEAN_ATTR).toInt()
        && double(200) == actor->getParameter(LENDEV_ATTR).toDouble();
}

QString HMMBuildPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(IN_PORT));
    Actor* msaProducer = input->getProducer(IN_PORT);

    QString msaName = msaProducer ? tr("For each MSA from <u>%1</u>,").arg(msaProducer->getLabel()) : "";

    QString calibrate;
    if (getParameter(CALIBRATE_ATTR).toBool()) {
        calibrate = tr(" and calibrate");
    }
    QString cfg = isDefaultCfg(this) ? tr("default") : tr("custom");

    QString doc = tr("%1 build%2 HMM profile using <u>%3</u> settings.")
        .arg(msaName)
        .arg(calibrate)
        .arg(cfg);

    return doc;
}

void HMMBuildWorkerFactory::cleanup() {
    delete WorkflowEnv::getProtoRegistry()->unregisterProto(ACTOR);
    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    delete localDomain->unregisterEntry(ACTOR);
}

void HMMBuildWorker::init() {
    input = ports.value(IN_PORT);
    output = ports.value(OUT_PORT);
    cfg.name = actor->getParameter(NAME_ATTR)->value.toString();
    cfg.strategy = HMMBuildStrategy(actor->getParameter(MODE_ATTR)->value.toInt());

    calSettings.fixedlen = actor->getParameter(FIXEDLEN_ATTR)->value.toInt();
    calSettings.lenmean = actor->getParameter(LENMEAN_ATTR)->value.toInt();
    calSettings.nsample = actor->getParameter(NUM_ATTR)->value.toInt();
    calSettings.lensd = (float)actor->getParameter(LENDEV_ATTR)->value.toDouble();
    calSettings.seed = actor->getParameter(SEED_ATTR)->value.toInt();
    calSettings.nThreads = actor->getParameter(THREADS_ATTR)->value.toInt();
    calibrate = actor->getParameter(CALIBRATE_ATTR)->value.toBool();
}

bool HMMBuildWorker::isReady() {
    return nextTick || input->hasMessage();
}

Task* HMMBuildWorker::tick() {
    if (nextTick) {
        Task* t = nextTick;
        nextTick = NULL;
        connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
        return t;
    }
    const MAlignment& ma = input->get().getData().value<MAlignment>();
    Task* t = new HMMBuildTask(cfg, ma);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void HMMBuildWorker::sl_taskFinished() {
    Task* t = qobject_cast<Task*>(sender());
    if (t->getState() != Task::State_Finished) return;

    HMMBuildTask* build = qobject_cast<HMMBuildTask*>(sender());
    plan7_s* hmm = NULL;
    if (build) {
        assert(!nextTick);
        hmm = build->getHMM();
        if (calibrate) {
            if (calSettings.nThreads == 1) {
                nextTick = new HMMCalibrateTask(hmm, calSettings);
            } else {
                nextTick = new HMMCalibrateParallelTask(hmm, calSettings);
            }
        }
        log.info(tr("Built HMM profile"));
    } else {
        HMMCalibrateAbstractTask* calibrate = qobject_cast<HMMCalibrateAbstractTask*>(sender());
        assert(calibrate);
        hmm = calibrate->getHMM();
        log.info(tr("Calibrated HMM profile"));
    }
    output->put(Message(HMMLib::HMM_PROFILE_TYPE(), qVariantFromValue<plan7_s*>(hmm)));
    if (input->isEnded()) {
        output->setEnded();
    }
}

bool HMMBuildWorker::isDone() {
    return !input || input->isEnded();
}

} //namespace LocalWorkflow
} //namespace GB2
