/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include <core_api/ScriptRegistry.h>
#include <core_api/ServiceTypes.h>

#include <time.h>

#include "SendSelectionDialogImpl.h"
#include "ScriptEngineContext.h"

#include <QtCore/QDebug>
#include <QtCore/QFile>

#include <QtScript/QScriptEngine>
#include <QtScript/QScriptValueIterator>
#include <QtGui/QMessageBox>

namespace GB2
{
const static char * SETTINGS_SCRIPT_MAX_RESULT_LEN = "http_annotator/max_res_len";
const static char * SETTINGS_SCRIPT_MIN_RESULT_LEN = "http_annotator/min_res_len";

SendSelectionDialogImpl::SendSelectionDialogImpl( QList<Script *> _scripts, const GObjectReference& seqRef, bool _isAminoSeq ) :
scripts(_scripts), isAminoSeq(_isAminoSeq), extImported(false)
{
    ca_m.data->name = "misc_feature";//overrided by script
    ca_m.hideAnnotationName = true;
    ca_m.hideLocation = true;
    ca_m.sequenceObjectRef = seqRef;
    ca_c = new CreateAnnotationWidgetController( &ca_m, this );
    setupUi( this );
    int idx = 1;
    vertLayout->insertWidget( idx, ca_c->getWidget() );

    setupScriptsList();
    setMinimumSize( layout()->totalSizeHint() );
    strandBothRadio->setChecked( true );

    maxResultLen = AppContext::getSettings()->getValue( SETTINGS_SCRIPT_MAX_RESULT_LEN ).toInt();
    if( !maxResultLen ) {
        maxResultLen = 1024;
    }
    minResultLen = AppContext::getSettings()->getValue( SETTINGS_SCRIPT_MIN_RESULT_LEN ).toInt();
    

    maxResultLenSpin->setValue( maxResultLen );
    minResultLenSpin->setValue( minResultLen );
    minResultLenSpin->setMaximum( maxResultLen );
    maxResultLenSpin->setMinimum( minResultLen );

    customSettingsButton->setDisabled( true );

    connect( chooseScriptCombo, SIGNAL(currentIndexChanged(int)), SLOT(sl_scriptSelected(int)) );
    connect( minResultLenSpin, SIGNAL(valueChanged(int)), SLOT(sl_minResLenChanged(int)) );
    connect( maxResultLenSpin, SIGNAL(valueChanged(int)), SLOT(sl_maxResLenChanged(int)) );
    connect( okButton, SIGNAL(clicked()), SLOT(sl_OK()) );
    connect( customSettingsButton, SIGNAL(clicked()), SLOT(sl_customSettings()) );
    sl_scriptSelected( 0 );
}

void SendSelectionDialogImpl::setupScriptsList() {
    //cannot analyze amino sequences using nucleotide databases
    if( isAminoSeq ) {
        for( int i = 0; i < scripts.size(); ++i ) {
            QString alpha_str;
            Script * s = scripts.at(i);
            s->init_engine( &engine );
            HttpAnnotatorAlphabet al = ScriptHttpAnnotatorContext::getAlphabet(&engine, &alpha_str);
            if( HttpAnnotatorAlphabet_Nucleo == al) {
                scripts.removeAt(i);
            }
        }
    }
    foreach( Script * scr, scripts ) {
        chooseScriptCombo->addItem( scr->getName() );
    }
}

void SendSelectionDialogImpl::setupAlphabet() {
    QString alpha_str;
    HttpAnnotatorAlphabet al = ScriptHttpAnnotatorContext::getAlphabet( &engine, &alpha_str );
    scriptAlpha->setText( alpha_str );

    if( HttpAnnotatorAlphabet_Amino == al ) {
        alphaAminoRadio->setChecked( true );
        alphabetBox->setDisabled( true );
    } else if( HttpAnnotatorAlphabet_Nucleo == al ){
        alphaNucleoRadio->setChecked( true );
        alphabetBox->setDisabled( true );
    } else {
        alphaAnyRadio->setChecked( true );
        alphabetBox->setDisabled( false );
    }
}

void SendSelectionDialogImpl::setupStrand() {
    if( isAminoSeq ) {
        strandDirectRadio->setChecked( true );
        strandBox->setDisabled( true );
        return;
    }
    HttpAnnotatorStrand st = ScriptHttpAnnotatorContext::getStrand( &engine );
    if( HttpAnnotatorStrand_Both == st ) {
        strandDirectRadio->setChecked( true );
        strandBox->setDisabled( true );
    } else {
        strandBothRadio->setChecked( true );
        strandBox->setDisabled( false );
    }
}

void SendSelectionDialogImpl::sl_scriptSelected( int index ) {
    if( scripts.isEmpty() ) {
        return;
    }
    Script * scr = scripts.at( index );

    scr->init_engine( &engine );
    maxQueryLen = ScriptHttpAnnotatorContext::getMaxQueryLen( &engine );        
    bool cs = ScriptHttpAnnotatorContext::hasCustomSettings( &engine );

    scriptMaxInput->setText( QString::number(maxQueryLen) );
    customSettingsButton->setDisabled( !cs );

    setupAlphabet();
    setupStrand();
}

void SendSelectionDialogImpl::sl_maxResLenChanged( int i ) {
    minResultLenSpin->setMaximum( i );
}
void SendSelectionDialogImpl::sl_minResLenChanged( int i ) {
    maxResultLenSpin->setMinimum( i );
}

void SendSelectionDialogImpl::sl_OK() {
    QString error = ca_c->validate();
    if (!error.isEmpty()) {
        QMessageBox::critical(NULL, tr("Error"), error);
        return;
    }
    ca_c->prepareAnnotationObject();
    
    maxResultLen = maxResultLenSpin->value();
    minResultLen = minResultLenSpin->value();

    strand = ( strandBothRadio->isChecked() ? SendSelectionStrand_Both :
        (strandDirectRadio->isChecked() ? SendSelectionStrand_Direct : SendSelectionStrand_Complement) );
    alpha = ( alphaAnyRadio->isChecked() ? SendSelectionAlphabet_Any : 
        (alphaNucleoRadio->isChecked() ? SendSelectionAlphabet_Nucleo : SendSelectionAlphabet_Amino) );
    chosen = scripts.at( chooseScriptCombo->currentIndex() );

    AppContext::getSettings()->setValue( SETTINGS_SCRIPT_MAX_RESULT_LEN, maxResultLen );
    AppContext::getSettings()->setValue( SETTINGS_SCRIPT_MIN_RESULT_LEN, minResultLen );

    accept();
}

void SendSelectionDialogImpl::sl_customSettings() {
    Script * scr = scripts.at( chooseScriptCombo->currentIndex() );
    QScriptEngine engine;

    if( !extImported ) {
        Script::importExtensions( &engine );
        extImported = true;
    }
    scr->init_engine(&engine);
    scr->callSetup(&engine);
    scr->callConfigure(&engine);
}

} //namespace
