/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ADVClipboard.h"

#include "AnnotatedDNAView.h"
#include "ADVSequenceObjectContext.h"
#include "ADVConstants.h"

#include <core_api/DNATranslation.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/AnnotationTableObject.h>
#include <selection/DNASequenceSelection.h>
#include <selection/AnnotationSelection.h>
#include <selection/SelectionUtils.h>
#include <util_text/TextUtils.h>
#include <util_gui/GUIUtils.h>

#include <QtCore/QTextStream>
#include <QtGui/QClipboard>
#include <QtGui/QApplication>
#include <QtGui/QMenu>
#include <QtGui/QMessageBox>

namespace GB2 {

ADVClipboard::ADVClipboard(AnnotatedDNAView* c) : QObject(c) {
    ctx = c;
    //TODO: listen seqadded/seqremoved!!
    
    connect(ctx, SIGNAL(si_focusChanged(ADVSequenceWidget*, ADVSequenceWidget*)), 
                 SLOT(sl_onFocusedSequenceWidgetChanged(ADVSequenceWidget*, ADVSequenceWidget*)));

    foreach(ADVSequenceObjectContext* sCtx, ctx->getSequenceContexts()) {
        connectSequence(sCtx);
    }
    

    copySequenceAction = new QAction(QIcon(":/core/images/copy_sequence.png"), tr("copy_sequence"), this);
    copySequenceAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_C));
    
    copyTranslationAction = new QAction(QIcon(":/core/images/copy_translation.png"), tr("copy_translation"), this);
    copyTranslationAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_T));
    
    copyComplementSequenceAction = new QAction(QIcon(":/core/images/copy_complement_sequence.png"), tr("copy_complement_sequence"), this);
    copyComplementSequenceAction->setShortcut(QKeySequence(Qt::CTRL | Qt::SHIFT | Qt::Key_C));
    
    copyComplementTranslationAction = new QAction(QIcon(":/core/images/copy_complement_translation.png"), tr("copy_complement_translation"), this);
    copyComplementTranslationAction->setShortcut(QKeySequence(Qt::CTRL | Qt::SHIFT | Qt::Key_T));

    copyAnnotationSequenceAction = new QAction(QIcon(":/core/images/copy_annotation_sequence.png"), tr("copy_annotation_sequence"), this);
    copyAnnotationSequenceTranslationAction = new QAction(QIcon(":/core/images/copy_annotation_translation.png"), tr("copy_annotation_sequence_translation"), this);

    connect(copySequenceAction, SIGNAL(triggered()), SLOT(sl_copySequence()));
    connect(copyTranslationAction, SIGNAL(triggered()), SLOT(sl_copyTranslation()));
    connect(copyComplementSequenceAction, SIGNAL(triggered()), SLOT(sl_copyComplementSequence()));
    connect(copyComplementTranslationAction, SIGNAL(triggered()), SLOT(sl_copyComplementTranslation()));
    connect(copyAnnotationSequenceAction, SIGNAL(triggered()), SLOT(sl_copyAnnotationSequence()));
    connect(copyAnnotationSequenceTranslationAction, SIGNAL(triggered()), SLOT(sl_copyAnnotationSequenceTranslation()));

    updateActions();
}

void ADVClipboard::connectSequence(ADVSequenceObjectContext* sCtx) {
    connect(sCtx->getSequenceSelection(), 
      SIGNAL(si_selectionChanged(LRegionsSelection*, const QList<LRegion>&, const QList<LRegion>&)), 
      SLOT(sl_onDNASelectionChanged(LRegionsSelection*, const QList<LRegion>& , const QList<LRegion>&)));

    connect(sCtx->getAnnotatedDNAView()->getAnnotationsSelection(), 
      SIGNAL(si_selectionChanged(AnnotationSelection* , const QList<Annotation*>&, const QList<Annotation*>&)),
      SLOT(sl_onAnnotationSelectionChanged(AnnotationSelection* , const QList<Annotation*>&, const QList<Annotation*>&)));
}

void ADVClipboard::sl_onDNASelectionChanged(LRegionsSelection* thiz, const QList<LRegion>& added, const QList<LRegion>& removed) {
    Q_UNUSED(thiz); Q_UNUSED(added); Q_UNUSED(removed);
    updateActions();
}

void ADVClipboard::sl_onAnnotationSelectionChanged(AnnotationSelection* thiz, const QList<Annotation*>& added, const QList<Annotation*>& removed) {
    Q_UNUSED(thiz); Q_UNUSED(added); Q_UNUSED(removed);
    updateActions();
}

void ADVClipboard::sl_copySequence() {
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    if (seqCtx == NULL) {
        QMessageBox::critical(NULL, tr("error"), "no_sequence_in_focus");
        return;
    }

    QString res;
    QTextStream stream(&res);
    const QByteArray& sequence = seqCtx->getSequenceObject()->getSequence();
    const QList<LRegion>& list = seqCtx->getSequenceSelection()->getSelectedRegions();
    for (int i = 0; i < list.size(); i++) {
        const LRegion& reg = list.at(i);
        QByteArray text(sequence.constData() + reg.startPos, reg.len);
        stream << text;
        if (i < list.size()-1) {
            stream<<endl;
        }

    }
    stream.flush();
    QApplication::clipboard()->setText(res);
}

void ADVClipboard::sl_copyTranslation() {
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    if (seqCtx == NULL) {
        QMessageBox::critical(NULL, tr("error"), "no_sequence_in_focus");
        return;
    }

    QString res;
    QTextStream stream(&res);
    
    const QByteArray& sequence = seqCtx->getSequenceObject()->getSequence();
    const QList<LRegion>& list = seqCtx->getSequenceSelection()->getSelectedRegions();
    DNATranslation* aminoTT = seqCtx->getAminoTT();
    assert(aminoTT!=NULL);
    for (int i = 0; i < list.size(); i++) {
        const LRegion& creg = list.at(i);
        LRegion reg = SelectionUtils::normalizeRegionBy3(creg, sequence.length(), true);
        QByteArray text(reg.len/3, 0);
        aminoTT->translate(sequence.constData() + reg.startPos, reg.len, text.data(), text.size());
        
        stream << text;
        if (i < list.size()-1) {
            stream<<endl;
        }
    }
    stream.flush();
    QApplication::clipboard()->setText(res);
}

void ADVClipboard::sl_copyComplementSequence() {
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    if (seqCtx == NULL) {
        QMessageBox::critical(NULL, tr("error"), "no_sequence_in_focus");
        return;
    }

    QString res;
    QTextStream stream(&res);
    
    const QByteArray& sequence = seqCtx->getSequenceObject()->getSequence();
    const QList<LRegion>& list = seqCtx->getSequenceSelection()->getSelectedRegions();
    DNATranslation* complTT = seqCtx->getComplementTT();
    assert(complTT!=NULL);
    for (int i = 0; i < list.size(); i++) {
        const LRegion& reg = list.at(i);
        QByteArray text(sequence.constData() + reg.startPos, reg.len);
        TextUtils::translate(complTT->getOne2OneMapper(), text.data(), text.size());
        TextUtils::reverse(text.data(), text.size());
        
        stream << text;
        if (i < list.size()-1) {
            stream<<endl;
        }
    }
    stream.flush();
    QApplication::clipboard()->setText(res);
}

void ADVClipboard::sl_copyComplementTranslation() {
    //TODO: reuse AnnotationSelection utils
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    if (seqCtx == NULL) {
        QMessageBox::critical(NULL, tr("error"), tr("no_sequence_in_focus"));
        return;
    }
    
    QString res;
    QTextStream stream(&res);
    const QByteArray& sequence = seqCtx->getSequenceObject()->getSequence();
    const QList<LRegion>& list = seqCtx->getSequenceSelection()->getSelectedRegions();
    DNATranslation* aminoTT = seqCtx->getAminoTT();
    DNATranslation* complTT = seqCtx->getComplementTT();
    assert(aminoTT != NULL && aminoTT != NULL);
    for (int i = 0; i < list.size(); i++) {
        const LRegion& creg = list.at(i);
        LRegion reg = SelectionUtils::normalizeRegionBy3(creg, sequence.length(), false);
        QByteArray complNucl(sequence.constData() + reg.startPos, reg.len);
        TextUtils::translate(complTT->getOne2OneMapper(), complNucl.data(), complNucl.size());
        TextUtils::reverse(complNucl.data(), complNucl.size());
        
        QByteArray text(reg.len/3, 0);
        aminoTT->translate(complNucl.constData(), reg.len, text.data(), text.size());

        stream << text;
        if (i < list.size()-1) {
            stream<<endl;
        }
    }
    stream.flush();
    QApplication::clipboard()->setText(res);
}


void ADVClipboard::sl_copyAnnotationSequence() {
    QByteArray res;
    const QList<AnnotationSelectionData>& as = ctx->getAnnotationsSelection()->getSelection();
    
    //BUG528: add alphabet symbol role: insertion mark
    char gapSym = '-';
    for (int i=0, n = as.size(); i < n; i++) {
        const AnnotationSelectionData& sd = as.at(i);
        if (i!=0) {
            res.append('\n');
        }
        ADVSequenceObjectContext* seqCtx = ctx->getSequenceContext(sd.annotation->getGObject());
        if (seqCtx == NULL) {
            res.append(gapSym);//?? generate sequence with len == region-len using default sym?
            continue;
        }
        const QByteArray& sequence = seqCtx->getSequenceData();
        DNATranslation* complTT = seqCtx->getComplementTT();
        AnnotationSelection::getAnnotationSequence(res, sd, gapSym, sequence, complTT, NULL);
    }
    QApplication::clipboard()->setText(res);
}


void ADVClipboard::sl_copyAnnotationSequenceTranslation() {
    QByteArray res;
    const QList<AnnotationSelectionData>& as = ctx->getAnnotationsSelection()->getSelection();

    //BUG528: add alphabet symbol role: insertion mark
    //TODO: reuse AnnotationSelection utils
    char gapSym = '-';
    for (int i=0, n = as.size(); i < n; i++) {
        const AnnotationSelectionData& sd = as.at(i);
        if (i!=0) {
            res.append('\n');
        }
        ADVSequenceObjectContext* seqCtx = ctx->getSequenceContext(sd.annotation->getGObject());
        if (seqCtx == NULL) {
            res.append(gapSym);//?? generate sequence with len == region-len using default sym?
            continue;
        }
        const QByteArray& sequence = seqCtx->getSequenceData();
        DNATranslation* complTT = seqCtx->getComplementTT();
        DNATranslation* aminoTT = seqCtx->getComplementTT();
        LRegion sequenceRange(0, sequence.size());

        int start = sd.locationIdx == -1 ? 0 : sd.locationIdx;
        int nLocations = sd.locationIdx == -1 ? sd.annotation->getLocation().size() : 1;
        for (int j = start, last = start + nLocations; j < last; j++) {
            if ( j!=start ) {
                res.append(gapSym);
            }
            LRegion creg = sd.annotation->getLocation().at(j).intersect(sequenceRange);
            if (sd.annotation->isOnComplementStrand()) {
                assert(complTT!=NULL);
                LRegion reg = SelectionUtils::normalizeRegionBy3(creg, sequence.length(), false);
                QByteArray complNucl(sequence.constData() + reg.startPos, reg.len);
                TextUtils::translate(complTT->getOne2OneMapper(), complNucl.data(), complNucl.size());
                TextUtils::reverse(complNucl.data(), complNucl.size());

                QByteArray text(reg.len/3, 0);
                aminoTT->translate(complNucl.constData(), reg.len, text.data(), text.size());

                res.append(text);
            } else {
                LRegion reg = SelectionUtils::normalizeRegionBy3(creg, sequence.length(), true);
                QByteArray text(reg.len/3, 0);
                aminoTT->translate(sequence.constData() + reg.startPos, reg.len, text.data(), text.size());

                res.append(text);
            }
        }
    }
    QApplication::clipboard()->setText(res);
}



void ADVClipboard::updateActions() {
    ADVSequenceObjectContext* seqCtx = getSequenceContext();
    DNASequenceSelection* sel = seqCtx == NULL ? NULL : seqCtx->getSequenceSelection();
    bool hasSequence = sel != NULL;
    bool hasComplement = hasSequence && seqCtx->getComplementTT()!=NULL;
    bool hasTranslation = hasSequence && seqCtx->getAminoTT()!=NULL;
    bool selectionIsNotEmpty = hasSequence && !sel->getSelectedRegions().isEmpty();
    
    copySequenceAction->setEnabled(selectionIsNotEmpty);
    copyTranslationAction->setEnabled(selectionIsNotEmpty && hasTranslation);
    copyComplementSequenceAction->setEnabled(selectionIsNotEmpty && hasComplement);
    copyComplementTranslationAction->setEnabled(selectionIsNotEmpty && hasComplement && hasTranslation);
    
    bool hasAnnotationSelection = !ctx->getAnnotationsSelection()->isEmpty();
    bool hasSequenceForAnnotations = false;
    bool hasTranslationForAnnotations = false;
    if (hasAnnotationSelection) {
        const QList<AnnotationSelectionData>& as = ctx->getAnnotationsSelection()->getSelection();
        foreach(const AnnotationSelectionData& sd, as) {
            ADVSequenceObjectContext* asCtx = ctx->getSequenceContext(sd.annotation->getGObject());
            if (asCtx == NULL) {
                continue;
            }
            hasSequenceForAnnotations = true;
            hasTranslationForAnnotations = hasTranslationForAnnotations || asCtx->getAminoTT()!=NULL;
            if (hasTranslationForAnnotations) {
                break;
            }
        }
    }
    copyAnnotationSequenceAction->setEnabled(hasAnnotationSelection && hasSequenceForAnnotations);
    copyAnnotationSequenceTranslationAction->setEnabled(hasAnnotationSelection && hasTranslationForAnnotations);
}


void ADVClipboard::addCopyMenu(QMenu* m) {
    QMenu* copyMenu = new QMenu(tr("copy_menu"), m);
    copyMenu->menuAction()->setObjectName(ADV_MENU_COPY);
    
    copyMenu->addAction(copySequenceAction);
    copyMenu->addAction(copyComplementSequenceAction);
    copyMenu->addAction(copyTranslationAction);
    copyMenu->addAction(copyComplementTranslationAction);
    copyMenu->addAction(copyAnnotationSequenceAction);
    copyMenu->addAction(copyAnnotationSequenceTranslationAction);

    m->addMenu(copyMenu);
}


ADVSequenceObjectContext* ADVClipboard::getSequenceContext() const {
    return ctx->getSequenceInFocus();
}

void ADVClipboard::sl_onFocusedSequenceWidgetChanged(ADVSequenceWidget* oldW, ADVSequenceWidget* newW) {
    Q_UNUSED(oldW);
    Q_UNUSED(newW);
    updateActions();
}
}//namespace
