/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_PROJECT_TREE_CONTROLLER_H_
#define _GB2_PROJECT_TREE_CONTROLLER_H_

#include <core_api/core_api.h>
#include <selection/DocumentSelection.h>
#include <selection/GObjectSelection.h>

#include <core_api/AppContext.h>

#include <core_api/GObjectReference.h>

#include <QtCore/QPointer>
#include <QtGui/QTreeWidget>
#include <QtGui/QIcon>


namespace GB2 {

class ProjViewDocumentTreeItem;	
class ProjViewObjectTreeItem;
class Task;
class GObjectConstraints;

class GB2_COREAPI_EXPORT LoadDocumentTaskProvider {
public:
    virtual ~LoadDocumentTaskProvider(){};
    virtual Task* createLoadDocumentTask(Document* doc) const = 0;
};

class GB2_COREAPI_EXPORT ProjectTreeControllerModeSettings {
public:
    ProjectTreeControllerModeSettings() 
        : allowMultipleSelection(true), sorted(false), readOnlyFilter(TriState_Unknown), loadTaskProvider(NULL){}

    QList<GObjectType>          objectTypesToShow;
    QList<GObjectConstraints*>  objectConstraints;
    QList<QPointer<GObject> >   excludeObjectList;
    QStringList                 tokensToShow;
    bool                        allowMultipleSelection;
    bool                        sorted;
    GObjectRelation             objectRelation;
    TriState                    readOnlyFilter; //unknown -> all, true -> filter(exclude) readonly, false -> include only readonly
    LoadDocumentTaskProvider*   loadTaskProvider;
    
    bool isDocumentShown(Document* doc) const;
    bool isObjectShown(GObjectType t) const;
    bool isObjectShown(GObject* o) const ;
};

class GB2_COREAPI_EXPORT ProjectTreeController : public QObject {
	Q_OBJECT
public:
	ProjectTreeController(QObject* parent, QTreeWidget* tree, const ProjectTreeControllerModeSettings& mode);
	
	const DocumentSelection* getDocumentSelection() const {return &documentSelection;}
	const GObjectSelection* getGObjectSelection() const {return &objectSelection;}

signals:
	void si_onPopupMenuRequested(QMenu& popup);
	void si_doubleClicked(GObject*);

private slots:
	void sl_onTreeSelectionChanged();
	void sl_onContextMenuRequested(const QPoint & pos);
	void sl_onRemoveSelectedDocuments();
	void sl_onLoadSelectedDocuments();
    void sl_onUnloadSelectedDocuments();

	void sl_onDocumentAddedToProject(Document* d);
	void sl_onDocumentRemovedFromProject(Document* d);
	void sl_onDocumentModifiedStateChanged();
	void sl_onDocumentLoadedStateChanged();

	void sl_onObjectAdded(GObject* o);
	void sl_onObjectRemoved(GObject* o);
	void sl_onObjectModifiedStateChanged();

	void sl_onItemDoubleClicked(QTreeWidgetItem * item, int column);

    void sl_onResourceUserRegistered(const QString& res, Task* t);
    void sl_onResourceUserUnregistered(const QString& res, Task* t);
    void sl_onLoadingDocumentProgressChanged();
    void sl_onToggleReadonly();
    void sl_lockedStateChanged();


private:
	void updateActions();
	void updateSelection();
	void connectModel();
	void connectDocument(Document* d);
	void disconnectDocument(Document* d);
    void connectToResourceTracker();
	void connectGObject(GObject* d);
	void buildTree();
	void buildDocumentTree(Document* d);
    void flattenDocumentItem(ProjViewDocumentTreeItem* docItem);
    void runLoadDocumentTask(Document* d);

	ProjViewDocumentTreeItem* findDocumentItem(Document* d) const;
	ProjViewObjectTreeItem* findGObjectItem(Document* d, GObject* r) const;


	QTreeWidget* tree;

	QAction* removeSelectedDocumentsAction;
	QAction* loadSelectedDocumentsAction;
    QAction* unloadSelectedDocumentsAction;
    QAction* addReadonlyFlagAction;
    QAction* removeReadonlyFlagAction;
    
	GObjectSelection  objectSelection;
	DocumentSelection documentSelection;
    const ProjectTreeControllerModeSettings mode;

public: 
	QIcon documentIcon;
    QIcon roDocumentIcon;
    QIcon gobjectIcon;

};


class GB2_COREAPI_EXPORT ProjViewTreeItem : public QTreeWidgetItem {
public:
	ProjViewTreeItem(ProjViewTreeItem *parent, ProjectTreeController* c) : QTreeWidgetItem(parent), controller(c) {}
	bool isRootItem() {return parent() == NULL;}
	virtual bool isDocumentItem() const {return false;}
	virtual bool isObjectItem() const {return false;}
	virtual void updateVisual(bool recursive = false) = 0;
	ProjectTreeController* controller;
};

class GB2_COREAPI_EXPORT ProjViewDocumentTreeItem : public ProjViewTreeItem {
public:
	ProjViewDocumentTreeItem(Document* _doc, ProjectTreeController* c);
	virtual bool isDocumentItem() const {return true;}
	virtual void updateVisual(bool recursive = false);
	Document* doc;
};

class GB2_COREAPI_EXPORT ProjViewObjectTreeItem : public ProjViewTreeItem {
public:
	ProjViewObjectTreeItem(GObject* _obj, ProjViewDocumentTreeItem* parent, ProjectTreeController* c);
	virtual bool isObjectItem() const {return true;}
	virtual void updateVisual(bool recursive = false);
	GObject* obj;
};



}//namespace
#endif
