/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GObjectComboBoxController.h"

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/GObject.h>

#include <gobjects/GObjectUtils.h>

namespace GB2 {


GObjectComboBoxController::GObjectComboBoxController(QObject* p, const GObjectComboBoxControllerConstraints& _c, QComboBox* _cb) 
: QObject(p), settings(_c), combo(_cb)
{
    connect(AppContext::getProject(), SIGNAL(si_documentAdded(Document*)), SLOT(sl_onDocumentAdded(Document*)));
    connect(AppContext::getProject(), SIGNAL(si_documentRemoved(Document*)), SLOT(sl_onDocumentRemoved(Document*)));
    foreach(Document* d, AppContext::getProject()->getDocuments()) {
        sl_onDocumentAdded(d);
    }
    objectIcon = QIcon(":core/images/gobject.png");
    combo->setInsertPolicy(QComboBox::InsertAlphabetically);
    updateCombo();
}

void GObjectComboBoxController::updateCombo() {
    QString lastFocus = combo->currentText();
    combo->clear();
    foreach(Document* d, AppContext::getProject()->getDocuments()) {
        addDocumentObjects(d);
    }
}

void GObjectComboBoxController::addDocumentObjects(Document* d) {
    foreach(GObject* obj, d->getObjects()) {
        addObject(obj);
    }
}

void GObjectComboBoxController::removeDocumentObjects(Document* d) {
    foreach(GObject* obj, d->getObjects()) {
        removeObject(obj);
    }
}

static QString itemText(GObject* o) {
    return o->getGObjectName() + " (" + o->getDocument()->getName() + ")";
}

static int findItem(QComboBox* cb, GObject* obj) {
    GObjectReference objRef(obj);
    for (int i=0; i < cb->count(); i++) {
        GObjectReference ref = cb->itemData(i).value<GObjectReference>();
        if (ref == objRef) {
            return i;
        }
    }
    return -1;
}

void GObjectComboBoxController::addObject(GObject* obj) {
    if (!settings.typeFilter.isEmpty() && obj->getGObjectType() != settings.typeFilter) {
        return;
    }
    if (settings.relationFilter.isValid() && !obj->hasObjectRelation(settings.relationFilter)) {
        return;
    }
    
    obj->disconnect(this);
    connect(obj, SIGNAL(si_lockedStateChanged()), SLOT(sl_lockedStateChanged()));

    if (settings.onlyWritable && obj->isStateLocked()) {
        return;
    }

#ifdef _DEBUG
    int n = findItem(combo, obj);
    assert(n==-1);
#endif

    combo->addItem(objectIcon, itemText(obj), QVariant::fromValue<GObjectReference>(GObjectReference(obj)));
}

void GObjectComboBoxController::removeObject(GObject* obj) {
    int n = findItem(combo, obj);
    if (n >= 0) {
        combo->removeItem(n);
    }
}

bool GObjectComboBoxController::setSelectedObject(GObject* obj) {
    int n = findItem(combo, obj);
    if (n < 0) {
        return false;
    }
    combo->setCurrentIndex(n);
    return true;
}

GObject* GObjectComboBoxController::getSelectedObject() const {
    int n = combo->currentIndex();
    if (n == -1) {
        return NULL;
    }
    GObjectReference r = combo->itemData(n).value<GObjectReference>();
    assert(r.isValid());
    GObject* obj = GObjectUtils::selectObjectByReference(r, GObjectUtils::findAllObjects());
    assert(obj!=NULL);
    return obj;
}


void GObjectComboBoxController::sl_onDocumentAdded(Document* d) {
    connect(d, SIGNAL(si_objectAdded(GObject*)), SLOT(sl_onObjectAdded(GObject*)));
    connect(d, SIGNAL(si_objectRemoved(GObject*)), SLOT(sl_onObjectRemoved(GObject*)));
    if (d->isLoaded()) {
        addDocumentObjects(d);
    }
}

void GObjectComboBoxController::sl_onDocumentRemoved(Document* d) {
    if (d->isLoaded()) {
        removeDocumentObjects(d);
    }
}

void GObjectComboBoxController::sl_onObjectAdded(GObject* obj) {
    addObject(obj);
}

void GObjectComboBoxController::sl_onObjectRemoved(GObject* obj) {
    removeObject(obj);
    obj->disconnect(this);
}

void GObjectComboBoxController::sl_lockedStateChanged() {
    if (!settings.onlyWritable) {
        return;
    }
    GObject* obj = qobject_cast<GObject*>(sender());
    if (obj->isStateLocked()) {
        removeObject(obj);
    } else {
        addObject(obj);
    }
}


}//namespace
