/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotationSettingsDialogController.h"

#include <gobjects/AnnotationSettings.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/AnnotationTableObject.h>

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/DocumentModel.h>

#include <util_gui/GUIUtils.h>

#include <QtGui/QPainter>
#include <QtGui/QPixmap>
#include <QtGui/QIcon>
#include <QtGui/QColorDialog>

namespace GB2 {

AnnotationSettingsDialogController::AnnotationSettingsDialogController(QWidget* p)
: QDialog(p) 
{
    setupUi(this);

    connect(tree, SIGNAL(itemClicked(QTreeWidgetItem*, int)), SLOT(sl_itemClicked(QTreeWidgetItem*, int)));
    connect(okButton, SIGNAL(clicked()), SLOT(sl_okClicked()));
    loadModel();
}


static void findAllAnnotationsNamesInProject(QSet<QString>& allAnnotations)  {
    Project* p = AppContext::getProject();
    foreach(Document* doc , p->getDocuments()) {
        foreach(GObject* obj, doc->getObjects()) {
            if (obj->getGObjectType() == GObjectTypes::ANNOTATION_TABLE) {
                AnnotationTableObject* ao = qobject_cast<AnnotationTableObject*>(obj);
                foreach(Annotation* a, ao->getAnnotations()) {
                    allAnnotations.insert(a->getAnnotationName());
                }
            }
        }
    }
}

static void findAllAnnotationsNamesInSettings(QSet<QString>& allAnnotations) {
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    foreach(const QString& s, asr->getAllSettings()) {
        allAnnotations.insert(s);
    }
}

void AnnotationSettingsDialogController::loadModel() {
    QSet<QString> allAnnotations;
    findAllAnnotationsNamesInProject(allAnnotations);
    findAllAnnotationsNamesInSettings(allAnnotations);
    
    QStringList sortedList = allAnnotations.toList();
    qSort(sortedList);
    
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    foreach(const QString& name, sortedList) {
        const AnnotationSettings* as = asr->getSettings(name);
        ASTreeItem* item = new ASTreeItem(as);
        tree->addTopLevelItem(item);    
    }
}


void AnnotationSettingsDialogController::storeModel() {
    AnnotationSettingsRegistry* asr = AppContext::getAnnotationsSettingsRegistry();
    QList<AnnotationSettings*> mods;
    for (int i=0; i<tree->topLevelItemCount(); i++) {
        ASTreeItem *item = (ASTreeItem*)tree->topLevelItem(i);
        item->fillModel();
        const AnnotationSettings* s = item->as;
        const AnnotationSettings* orig = asr->getSettings(s->name);
        if (!s->equals(orig)) {
            mods.append(new AnnotationSettings(s));
        }
    }
    if (!mods.isEmpty()) {
        asr->changeSettings(mods);
    }
}

void AnnotationSettingsDialogController::sl_okClicked() {
    storeModel();
    accept();
}


void AnnotationSettingsDialogController::sl_itemClicked(QTreeWidgetItem * i, int column) {
    if (column != 1) {
        return;
    }
    ASTreeItem* item = (ASTreeItem*)i;
    QColor c = QColorDialog::getColor(item->as->color, this);
    item->as->color = c;
    item->drawColorCell();
}

//////////////////////////////////////////////////////////////////////////

ASTreeItem::ASTreeItem(const AnnotationSettings* _as) {
    as = new AnnotationSettings(_as);    

    setText(0, as->name);
    drawColorCell();
    setCheckState(2, as->amino ? Qt::Checked : Qt::Unchecked);
    setCheckState(3, as->visible ? Qt::Checked : Qt::Unchecked);
}

ASTreeItem::~ASTreeItem() {
    delete as;
}

void ASTreeItem::fillModel() {
    as->amino = checkState(2) == Qt::Checked; 
    as->visible = checkState(3) == Qt::Checked;
}

void ASTreeItem::drawColorCell() {
    QIcon icon = GUIUtils::createSquareIcon(as->color, 14);
    setIcon(1, icon);
}

}//namespace
