/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TaskTests.h"

#include <core_api/AppContext.h>
#include <core_api/Task.h>
namespace GB2 {
    
#define VALUE_ATTR      "value"
#define DOC_ATTR        "doc"
#define OBJ_ATTR        "obj"
#define INDEX_ATTR      "index"
#define NAME_ATTR       "name"
#define FLAGS_ATTR      "flags"
#define SUBTASK_ATTR    "subtask"
#define STATE_ATTR      "state"
#define STATEINFO_PROGRESS_ATTR "progress"
#define STATEINFO_CANCEL_FLAG_ATTR "cancelflag"
#define TASK_TYPE_ATTR  "type"
#define DELETE_ATTR     "delete"
#define SERIAL_FLAG_ATTR "serial"
#define SUBTASK_NUM_ATTR "subtask_num"
#define CANCEL_FLAG_ATTR "cancel"
#define RUN_AFTER_ALL_SUBS_FINISHED_FLAG_ATTR "run_after_all_subs"
#define DELAY_ATTR       "ms"

static TaskFlags flagsFromString(QString str, bool *ok = NULL) {
    TaskFlags taskFlags = TaskFlags(TaskFlag_None);
    if(ok!=NULL) *ok = false;
    if(!str.isEmpty()) {
        QRegExp rx("(..)[(\\|)]");
        int pos = 0;
        QHash<QString, TaskFlag> hash;
        hash["TaskFlag_None"] = TaskFlag_None;
        hash["TaskFlag_NoRun"] = TaskFlag_NoRun;
        hash["TaskFlag_DeleteWhenFinished"] = TaskFlag_DeleteWhenFinished;
        hash["TaskFlag_RunAfterAllSubtasksFinished"] = TaskFlag_RunAfterAllSubtasksFinished;
        hash["TaskFlag_SerialSubtasks"]	= TaskFlag_SerialSubtasks;
        hash["TaskFlag_StopOnSubtaskError"] = TaskFlag_StopOnSubtaskError;    
        while ((pos = rx.indexIn(str, pos)) != -1) {
            pos += rx.matchedLength();
            TaskFlag flag = hash.value(rx.cap(1),(TaskFlag)-1);
            if(flag == -1) {
                return taskFlags;
            }
            taskFlags |= flag;
        }
        if(ok!=NULL) *ok = true;
    }
    return taskFlags;
}

static Task::State stateFromString(QString str, bool *ok = NULL) {
    Task::State taskState;
    if(ok!=NULL) *ok = false;
    if(!str.isEmpty()) {
        //int pos = 0;
        QHash<QString, Task::State> hash;
        hash["State_New"] = Task::State_New;
        hash["State_Prepared"] = Task::State_Prepared;
        hash["State_Running"] = Task::State_Running;
        hash["State_Finished"] = Task::State_Finished;
        taskState = hash.value(str,(Task::State)-1);
        if(taskState == -1) {
            return taskState;
        }
        if(ok!=NULL) *ok = true;
    }
    return taskState;
}

void InfiniteTestTask::run() {
    QMutex mutex;
    while(!stateInfo.cancelFlag) ;
}
StateOrderTestTask::StateOrderTestTask(StateOrderTestTaskCallback *ptr, TaskFlags _f)
: Task("calback_test_task",TaskFlags(TaskFlag_DeleteWhenFinished)|_f)
{
    callback = ptr;
    step = 0;
}
void StateOrderTestTask::prepare() {
    callback->func(this);
}
void StateOrderTestTask ::run() {
    callback->func(this);
}
Task::ReportResult StateOrderTestTask::report() {
    callback->func(this);
    return ReportResult_Finished;
}

void GTest_TaskCreateTest::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    deleteTask = false;

    resultContextName = el.attribute(INDEX_ATTR);
    QString taskName_str = el.attribute(NAME_ATTR);
    if (taskName_str.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(NAME_ATTR);
        return;
    }

    QString taskFlags_str = el.attribute(FLAGS_ATTR);
    TaskFlags taskFlags = TaskFlags(TaskFlag_None);
    if(!taskFlags_str.isEmpty()) {
        bool ok = false;
        taskFlags = flagsFromString(taskFlags_str,&ok);
        if(!ok) {
            stateInfo.error = GTest::tr("value not set %1").arg(FLAGS_ATTR);
            return;
        }
    }

    QString taskType_str = el.attribute(TASK_TYPE_ATTR);
    if(taskType_str.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(TASK_TYPE_ATTR);
        return;
    }

    QString deleteTask_str = el.attribute(DELETE_ATTR);
    if(!deleteTask_str.isEmpty()) {
        bool ok = false;
        deleteTask = deleteTask_str.toInt(&ok);
        if(!ok) {
            stateInfo.error = GTest::tr("value not set %1").arg(DELETE_ATTR);
            return;
        }
    }

    if(taskType_str == "base_task") {
        task = new Task(taskName_str,taskFlags);
    }
    else 
        if(taskType_str == "infinite_task") {
            task = new InfiniteTestTask(taskName_str,taskFlags);
        }


}

Task::ReportResult GTest_TaskCreateTest::report() {
    if (!resultContextName.isEmpty()) {
        addContext(resultContextName, task);
    }
    return ReportResult_Finished;
}
void GTest_TaskCreateTest::cleanup() {
    if (deleteTask) {
        delete task;
    }
}

void GTest_TaskAddSubtaskTest::init(GB2::XMLTestFormat *tf, const QDomElement &el){
    Q_UNUSED(tf);
    taskContextName = el.attribute(OBJ_ATTR);
    if (taskContextName .isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }
    subtaskContextName = el.attribute(SUBTASK_ATTR);
    if (subtaskContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(SUBTASK_ATTR);
        return;
    }
}
Task::ReportResult GTest_TaskAddSubtaskTest::report() {
    QObject *obj1 = getContext(taskContextName );
    if(obj1==NULL) {
        stateInfo.error = GTest::tr("invalid object context");
        return ReportResult_Finished;
    }
    assert(obj1!=NULL);
    Task *task = qobject_cast<Task*>(obj1);

    QObject  *obj2 = getContext(subtaskContextName );
    if(obj2==NULL) {
        stateInfo.error = GTest::tr("invalid object context");
        return ReportResult_Finished;
    }
    assert(obj2!=NULL);
    Task *subtask = qobject_cast<Task*>(obj2);

    task->addSubTask(subtask);

    if(!task->getSubtasks().contains(subtask)) {
        stateInfo.error = GTest::tr("subtask not add");
        return ReportResult_Finished;
    }
    if(subtask->getParentTask()!=task) {
        stateInfo.error = GTest::tr("subtask parent not set");
        return ReportResult_Finished;
    }

    return ReportResult_Finished;
}


void GTest_TaskCancelTest::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    objContextName = el.attribute(OBJ_ATTR);
    if (objContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }
}

Task::ReportResult GTest_TaskCancelTest::report() {
    QObject *obj = getContext(objContextName);
    if(obj==NULL) {
        stateInfo.error = GTest::tr("invalid object context");
        return ReportResult_Finished;
    }
    assert(obj!=NULL);
    Task *task = qobject_cast<Task*>(obj);
    task->cancel();
    if(!task->getStateInfo().cancelFlag) {
        stateInfo.error = GTest::tr("task state flag not matched: %1, expected %2 ").arg(task->getStateInfo().cancelFlag).arg(true);
        return ReportResult_Finished;
    }
    return ReportResult_Finished;
}

void GTest_TaskCheckFlag::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);
    taskContextName = el.attribute(OBJ_ATTR);
    if (taskContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }
    QString taskFlags_str = el.attribute(FLAGS_ATTR);
    if (taskFlags_str.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(FLAGS_ATTR);
        return;
    }
    bool ok = false;
    TaskFlags taskFlags = flagsFromString(taskFlags_str,&ok);
    if(!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(FLAGS_ATTR);
        return;
    }
}

Task::ReportResult GTest_TaskCheckFlag::report() {
    QObject *obj = getContext(taskContextName);
    if(obj==NULL) {
        stateInfo.error = GTest::tr("invalid object context");
        return ReportResult_Finished;
    }
    assert(obj!=NULL);
    Task *task = qobject_cast<Task*>(obj);
    if(task->getFlags().operator &(flag) == 0) {
        stateInfo.error = GTest::tr("task flags not matched %1, expected %2").arg(task->getFlags()).arg(flag);
        return ReportResult_Finished;
    }
    return ReportResult_Finished;
}

void GTest_TaskCheckState::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    checkState = false;
    checkProgress = false;
    checkCancelFlag = false;
    checkError = false;
    checkStateDesc = false;

    taskContextName = el.attribute(OBJ_ATTR);
    if (taskContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }
    QString taskState_str = el.attribute(STATE_ATTR);
    if (!taskState_str.isEmpty()) {
        bool ok = false;
        taskState = stateFromString(taskState_str,&ok);
        if(!ok) {
            stateInfo.error = GTest::tr("value not set %1").arg(STATE_ATTR);
            return;
        }
        checkState = true;
    }
    QString progress_str = el.attribute(STATEINFO_PROGRESS_ATTR);
    if (!progress_str.isEmpty()) {
        bool ok = false;
        taskStateInfo.progress = progress_str.toInt(&ok);
        if(!ok) {
            stateInfo.error = GTest::tr("value not set %1").arg(STATEINFO_PROGRESS_ATTR);
            return;
        }
        checkProgress = true;
    }

    QString cancelFlag_str = el.attribute(STATEINFO_CANCEL_FLAG_ATTR);
    if (!cancelFlag_str.isEmpty()) {
        bool ok = false;
        taskStateInfo.cancelFlag = cancelFlag_str.toInt(&ok);
        if(!ok) {
            stateInfo.error = GTest::tr("value not set %1").arg(STATEINFO_CANCEL_FLAG_ATTR);
            return;
        }
        checkCancelFlag = true;
    }
    //TODO: stateDesc, error
}

Task::ReportResult GTest_TaskCheckState::report() {
    QObject *obj = getContext(taskContextName);
    if(obj==NULL) {
        stateInfo.error = GTest::tr("invalid object context");
        return ReportResult_Finished;
    }
    assert(obj!=NULL);
    Task *task = qobject_cast<Task*>(obj);
    if(checkState) {
        if(task->getState() != taskState) {
            stateInfo.error = GTest::tr("task state not matched %1, expected %2").arg(task->getState()).arg(taskState);
            return ReportResult_Finished;
        }
    }
    if(checkProgress) {
        if(taskStateInfo.progress != task->getStateInfo().progress) {
            stateInfo.error = GTest::tr("task stateInfo.progress not matched %1, expected %2").arg(task->getStateInfo().progress).arg(taskStateInfo.progress);
            return ReportResult_Finished;
        }
    }
    if(checkCancelFlag) {
        if(taskStateInfo.cancelFlag!= task->getStateInfo().cancelFlag) {
            stateInfo.error = GTest::tr("task stateInfo.cancelFlag not matched %1, expected %2").arg(task->getStateInfo().cancelFlag).arg(taskStateInfo.cancelFlag);
            return ReportResult_Finished;
        }
    }
    return ReportResult_Finished;
}

void GTest_TaskExec::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);
    taskContextName = el.attribute(OBJ_ATTR);
    if (taskContextName.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(OBJ_ATTR);
        return;
    }
}

void GTest_TaskExec::prepare() {
    QObject *obj = getContext(taskContextName);
    if(obj==NULL) {
        stateInfo.error = GTest::tr("invalid object context");
        return;
    }
    assert(obj!=NULL);
    Task *task = qobject_cast<Task*>(obj);
    AppContext::getTaskScheduler()->registerTopLevelTask(task);
}
Task::ReportResult GTest_TaskExec::report() {
    return ReportResult_Finished;
}

void GTest_TaskStateOrder::init(XMLTestFormat *tf, const QDomElement& el) {
    Q_UNUSED(tf);

    serial_flag = true;
    subtask_num = 0;
    cancel_flag = false;
    run_after_all_subs_flag = false;

    QString subtask_num_str = el.attribute(SUBTASK_NUM_ATTR);
    if (!subtask_num_str.isEmpty()) {
        bool ok = false;
        subtask_num = subtask_num_str.toInt(&ok);
        if(!ok && subtask_num >= 0) {
            stateInfo.error = GTest::tr("value not set %1").arg(SUBTASK_NUM_ATTR);
            return;
        }
    }
    QString serial_flag_str = el.attribute(SERIAL_FLAG_ATTR);
    if (serial_flag_str.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(SERIAL_FLAG_ATTR);
        return;
    }
    bool ok = false;
    serial_flag = serial_flag_str.toInt(&ok);
    if(!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(SERIAL_FLAG_ATTR);
        return;
    }

    QString cancel_flag_str = el.attribute(CANCEL_FLAG_ATTR);
    if (!cancel_flag_str.isEmpty()) {
        ok = false;
        cancel_flag = cancel_flag_str.toInt(&ok);
        if(!ok) {
            stateInfo.error = GTest::tr("value not set %1").arg(CANCEL_FLAG_ATTR);
            return;
        }
    }
    QString run_after_all_subs_flag_str = el.attribute(RUN_AFTER_ALL_SUBS_FINISHED_FLAG_ATTR );
    if (!run_after_all_subs_flag_str.isEmpty()) {
        ok = false;
        run_after_all_subs_flag = run_after_all_subs_flag_str.toInt(&ok);
        if(!ok) {
            stateInfo.error = GTest::tr("value not set %1").arg(RUN_AFTER_ALL_SUBS_FINISHED_FLAG_ATTR );
            return;
        }
    }
    task = new StateOrderTestTask(this, TaskFlags((serial_flag?TaskFlag_SerialSubtasks:TaskFlag_None)|
        (run_after_all_subs_flag?TaskFlag_RunAfterAllSubtasksFinished:TaskFlag_None)));
    addSubTask(task);
    for (int i=0; i<subtask_num; i++) {
        StateOrderTestTask *sub = new StateOrderTestTask(this, TaskFlag_None);
        subs.append(sub);
        task->addSubTask(sub);
    }
}

void GTest_TaskStateOrder::func(StateOrderTestTask *_task) {

    int ind = -1;
    if(task == _task) {
    } else {
        ind = subs.indexOf((StateOrderTestTask *)_task);
        if(ind < 0) {
            stateInfo.error = GTest::tr("task test internal error: can't find subtask in list").arg(1);
            return; 
        }
    }
    int &step = _task->step;
    State newState = _task->getState();

    if(step == 0) {
        if(cancel_flag) {
            _task->cancel();
        }
        if(serial_flag) {
            for(int i=0; i<ind; i++) {
                if(!subs[i]->isFinished()) {
                    stateInfo.error = GTest::tr("task serial subtask promoting error");
                    return;
                }
            }
        }
        step++;
        if(newState != State_New) {
            stateInfo.error = GTest::tr("task promoting error: state value not matched %1, expected %2").arg(newState).arg(State_New);
            return;
        }
        return;
    } else
        if(step == 1) {
            step++;
            if(task == _task) {
                if(run_after_all_subs_flag) {
                    for(int i=0; i<subs.count(); i++) {
                        if(!subs[i]->isFinished()) {
                            stateInfo.error = GTest::tr("task promoting error: run after all subtasks finished");
                            return;
                        }
                    }
                }
            }
            if(task->getStateInfo().cancelFlag) {
                if(newState == State_Running) {
                    stateInfo.error = GTest::tr("task promoting error: run canceled task");
                    return;
                }
            } else
                if(newState != State_Running) {
                    stateInfo.error = GTest::tr("task promoting error: state value not matched %1, expected %2").arg(newState).arg(State_Running);
                    return;
                }
        }
}

Task::ReportResult GTest_TaskStateOrder::report() {
    subs.clear();
    return ReportResult_Finished;
}

void GTest_Wait::init(GB2::XMLTestFormat *tf, const QDomElement &el) {
    Q_UNUSED(tf);
    QString ms_str = el.attribute(DELAY_ATTR);
    if (ms_str.isEmpty()) {
        stateInfo.error = GTest::tr("value not set %1").arg(DELAY_ATTR);
        return;
    }
    bool ok = false;
    ms = ms_str.toInt(&ok);
    if(!ok) {
        stateInfo.error = GTest::tr("value not set %1").arg(DELAY_ATTR);
        return;
    }
}

Task::ReportResult GTest_Wait::report() {
    QTime timer;
    timer.start();
    while(timer.elapsed() < ms) ;
    return ReportResult_Finished;
}

QList<XMLTestFactory*> TaskTests::createTestFactories() {
    QList<XMLTestFactory*> res;
    res.append(GTest_TaskStateOrder::createFactory());
    res.append(GTest_TaskCreateTest::createFactory());
    res.append(GTest_TaskAddSubtaskTest::createFactory());
    res.append(GTest_TaskCancelTest::createFactory());
    res.append(GTest_TaskCheckState::createFactory());
    res.append(GTest_TaskCheckFlag::createFactory());
    res.append(GTest_TaskExec::createFactory());
    res.append(GTest_Wait::createFactory());
    return res;
}

}//namespace
