/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_ANNOTATION_TABLE_OBJECT_H_
#define _GB2_ANNOTATION_TABLE_OBJECT_H_

#include <core_api/GObject.h>
#include <datatype/AnnotationData.h>

#include <QtCore/QSet>

namespace GB2 {

class Annotation;
class Qualifier;
class AnnotationGroup;
class AnnotationTableObject;

enum AnnotationModificationType {
    AnnotationModification_NameChanged, 
    AnnotationModification_QualifierAdded,
    AnnotationModification_QualifierRemoving,
    AnnotationModification_QualifierValueChanged,
    AnnotationModification_LocationChanged,
    AnnotationModification_AddedToGroup,
    AnnotationModification_RemovedFromGroup
};


class GB2_COREAPI_EXPORT AnnotationModification {
public:
    AnnotationModification(AnnotationModificationType _type, Annotation* a) : type(_type), annotation(a) {}
    AnnotationModificationType type;
    Annotation* annotation;
};

class  GB2_COREAPI_EXPORT QualifierModification : public AnnotationModification {
public:
    QualifierModification(AnnotationModificationType t, Annotation* a, const Qualifier& q) 
        : AnnotationModification(t, a), qualifier(q) {}

    Qualifier qualifier;
};

class  GB2_COREAPI_EXPORT AnnotationGroupModification : public AnnotationModification {
public:
    AnnotationGroupModification(AnnotationModificationType t, Annotation* a, AnnotationGroup* g) 
        : AnnotationModification(t, a), group(g) {}

    AnnotationGroup* group;
};


class  GB2_COREAPI_EXPORT Annotation {
    friend class AnnotationGroup;
    friend class AnnotationTableObject;

public:
    Annotation(SharedAnnotationData data); 

    virtual ~Annotation();

    AnnotationTableObject* getGObject() const { return obj;}

    const QString& getAnnotationName() const;
    
    void setAnnotationName(QString& name);

    bool isOnComplementStrand() const;
    
    TriState getAminoStrand() const;

    void setOnComplementStrand(bool v);

    void setAminoStrand(TriState s);

    const QList<LRegion>& getLocation() const;

    void addLocationRegion(const LRegion& reg);

    void removeLocationRegion(const LRegion& reg);

    const QList<Qualifier>& getQualifiers() const;

    void removeQualifier(const Qualifier& q);

    const QList<AnnotationGroup*>& getGroups() const {return groups;}

    void findQualifiers(const QString& name, QList<Qualifier>& res) const {return d->findQualifiers(name, res);}
    
    QString findFirstQualifierValue(const QString& name) const {return d->findFirstQualifierValue(name);}

    SharedAnnotationData data() const {return d;}
    QString getQualifiersTip(int maxRows) const;
        
private:
    AnnotationTableObject*                  obj;
    SharedAnnotationData                    d;
    QList<AnnotationGroup*>                 groups;
};


class GB2_COREAPI_EXPORT AnnotationGroup {
    friend class AnnotationTableObject;
public:
    AnnotationGroup(AnnotationTableObject* p, const QString& _name, AnnotationGroup* parentGrp);
    ~AnnotationGroup();
    
    static const QString ROOT_GROUP_NAME;

    void findAllAnnotationsInGroupSubTree(QSet<Annotation*>& set) const;

    const QList<Annotation*>& getAnnotations() const {return annotations;}

    void addAnnotation(Annotation* a);
    
    void removeAnnotation(Annotation* a);

    void removeAnnotations(const QList<Annotation*>& annotations);

    const QList<AnnotationGroup*>& getSubgroups() const {return subgroups;}

    AnnotationGroup* addSubgroup(const QString& name, bool create);

    void removeSubgroup(AnnotationGroup* g);

    const QString& getGroupName() const {return name;}
    
    QString getGroupPath() const; 

    void setGroupName(const QString& newName);

    AnnotationTableObject* getGObject() const {return obj;}

    AnnotationGroup* getParentGroup() const {return parentGroup;}
    
    AnnotationGroup* getSubgroup(const QString& path, bool create);

    /// remove all subgroups and annotation refs
    void clear();

    int getGroupDepth() const {return 1 + (parentGroup == NULL ? 0 : parentGroup->getGroupDepth());}

    bool isParentOf(AnnotationGroup* g) const;

private:
    QString                 name;
    AnnotationTableObject*  obj;
    AnnotationGroup*        parentGroup;
    QList<Annotation*>      annotations;
    QList<AnnotationGroup*> subgroups;
};

class GB2_COREAPI_EXPORT AnnotationTableObject: public GObject {
    Q_OBJECT
    friend class Annotation;
    friend class AnnotationGroup;

public:
    AnnotationTableObject(const QString& objectName, const QVariantMap& hintsMap = QVariantMap());
    ~AnnotationTableObject();
    
    const QList<Annotation*>& getAnnotations() const {return annotations;}
    
    AnnotationGroup* getRootGroup() const {return rootGroup;}

    void addAnnotation(Annotation* a, const QString& groupName = QString());

    void addAnnotations(const QList<Annotation*>& annotations, const QString& groupName = QString());

    void removeAnnotation(Annotation* a);
    
    void removeAnnotations(const QList<Annotation*>& annotations);

    virtual GObject* clone() const;

    void selectAnnotationsByName(const QString& name, QList<Annotation*>& res);

    bool checkConstraints(const GObjectConstraints* c) const;

protected:

    void emit_onAnnotationModified(const AnnotationModification& md) {emit si_onAnnotationModified(md);}

    void emit_onGroupCreated(AnnotationGroup* g) {emit si_onGroupCreated(g);}
    void emit_onGroupRemoved(AnnotationGroup* p, AnnotationGroup* g) {emit si_onGroupRemoved(p, g);}
    void emit_onGroupRenamed(AnnotationGroup* g, const QString& oldName) {emit si_onGroupRenamed(g, oldName);}

    void _removeAnnotation(Annotation* a);

    QList<Annotation*>      annotations;
    AnnotationGroup*        rootGroup;

signals:
    //annotations added to the object and have valid groups assigned
    void si_onAnnotationsAdded(const QList<Annotation*>& a);
    //annotations removed from the object and will be deleted, but still keeps references to groups and object
    void si_onAnnotationsRemoved(const QList<Annotation*>& a);
    void si_onAnnotationModified(const AnnotationModification& md);

    void si_onGroupCreated(AnnotationGroup*);
    void si_onGroupRemoved(AnnotationGroup* p, AnnotationGroup* removed);
    void si_onGroupRenamed(AnnotationGroup*, const QString& oldName);
};


class AnnotationTableObjectConstraints : public GObjectConstraints {
    Q_OBJECT
public:
    AnnotationTableObjectConstraints(QObject* p = NULL);
    AnnotationTableObjectConstraints(const AnnotationTableObjectConstraints& c, QObject* p = NULL);
    int sequenceSizeToFit;
};

bool annotationLessThanByRegion(const Annotation* a1, const Annotation* a2);


}//namespace


#endif
