/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AnnotationSettings.h"

#include <util_gui/GUIUtils.h>
#include <core_api/Settings.h>
#include <datatype/BioStruct3D.h>

namespace GB2 {

#define SETTINGS_ROOT QString("annotation_settings/")
#define MAX_CACHE_SIZE 1000


AnnotationSettingsRegistry::AnnotationSettingsRegistry(Settings* _s) : s(_s) {
    read();
}

AnnotationSettingsRegistry::~AnnotationSettingsRegistry() {
    save();
    
    foreach(AnnotationSettings*  v, persistentMap.values()) {
        delete v;
    }

    foreach(AnnotationSettings*  v, transientMap.values()) {
        delete v;
    }

}

void AnnotationSettingsRegistry::changeSettings(const QList<AnnotationSettings*>& settings) {
    if (settings.isEmpty()) {
        return;
    }
    QStringList changedNames;
    foreach(AnnotationSettings* s, settings) {
        AnnotationSettings* prev = persistentMap.value(s->name);
        if (prev!=NULL) {
            persistentMap.remove(s->name);
            delete prev;
        } 
        prev = transientMap.value(s->name);
        if (prev!=NULL) {
            transientMap.remove(s->name);
            delete prev;
        }
        persistentMap[s->name] = s;
        changedNames.append(s->name);

    }
    emit si_annotationSettingsChanged(changedNames);
}


const AnnotationSettings* AnnotationSettingsRegistry::getSettings(const QString& name) {
    //Search in persistent settings:
    AnnotationSettings* s = persistentMap.value(name);
    if (s!=NULL) {
        return s;
    }
    
    //search in cache:
    s = transientMap.value(name);
    if (s!=NULL)  {
        return s;
    }

    s = new AnnotationSettings();
    s->name = name;
    s->color = GUIUtils::genLightColor(name);
    s->amino = true;
    s->visible = true;
    if (transientMap.size() == MAX_CACHE_SIZE) {
        //todo: mutex!?
        transientMap.erase(transientMap.begin());
    }
    transientMap[name]=s;
    return s;
}


void AnnotationSettingsRegistry::read() {
    QStringList keys = s->getAllKeys(SETTINGS_ROOT);
    QList<AnnotationSettings*> list;
    foreach(const QString& key, keys) {
        AnnotationSettings* as = new AnnotationSettings();
        as->name = key.split('/').first();
        as->color = s->getValue(SETTINGS_ROOT + as->name + "/color", GUIUtils::genLightColor(as->name)).value<QColor>();
        as->visible = s->getValue(SETTINGS_ROOT + as->name + "/visible", true).toBool();
        as->amino = s->getValue(SETTINGS_ROOT + as->name + "/amino", true).toBool();
        list.append(as);
    }
    changeSettings(list);
    addPredefined();
}

void AnnotationSettingsRegistry::save() {
    QStringList keys = s->getAllKeys(SETTINGS_ROOT);
    foreach(AnnotationSettings* as, persistentMap.values()) {
        s->setValue(SETTINGS_ROOT + as->name + "/color", as->color);
        s->setValue(SETTINGS_ROOT + as->name + "/visible", as->visible);
        s->setValue(SETTINGS_ROOT + as->name + "/amino", as->amino);
    }
}

void AnnotationSettingsRegistry::addPredefined() {
    AnnotationSettings predefined[] = {
        //                  name            //amino //color                 //visible
        AnnotationSettings("3'UTR",         false,  QColor(255, 205, 230),  true ), 
        AnnotationSettings("5'UTR",         false,  QColor(255, 105, 180),  true ), 
        AnnotationSettings("CDS",           false,  QColor(150, 255, 255),  true ), 
        AnnotationSettings("contig",        false,  QColor(  0, 170, 127),  true ), 
        AnnotationSettings("exon",          true,   QColor(128, 128, 200),  true ), 
        AnnotationSettings("gene",          true,   QColor(  0, 255, 200),  true ), 
        AnnotationSettings("mat_peptide",   false,  QColor(170, 255,   0),  true ), 
        AnnotationSettings("misc_feature",  false,  QColor(100, 200, 100),  true ), 
        AnnotationSettings("mRNA",          false,  QColor(140, 140, 140),  true ), 
        AnnotationSettings("polyA_signal",  false,  QColor(100, 100, 255),  true ), 
        AnnotationSettings("primer_bind",   false,  QColor( 85, 170, 255),  true ), 
        AnnotationSettings("promoter",      false,  QColor(200, 250, 200),  true ), 
        AnnotationSettings("repeat_region", false,  QColor(204, 204, 255),  true ), 
        AnnotationSettings("rRNA",          false,  QColor(150, 250, 150),  true ), 
        AnnotationSettings("sig_peptide",   false,  QColor(  0, 170, 127),  true ), 
        AnnotationSettings("source",        false,  Qt::gray,               false), 
        AnnotationSettings("STS",           false,  QColor(  0, 220, 220),  true ), 
        AnnotationSettings("terminator",    false,  QColor(250, 180, 180),  true ), 
        AnnotationSettings("tRNA",          false,  QColor(200, 250, 200),  true ), 
        AnnotationSettings("variation",     false,  QColor(255, 255, 155),  true ), 
        AnnotationSettings(BioStruct3D::AlphaHelixAnnotationTag, true, QColor(102,255, 0), true),
        AnnotationSettings(BioStruct3D::BetaStrandAnnotationTag, true, QColor(255,255,153), true),
        AnnotationSettings(BioStruct3D::TurnAnnotationTag, true, QColor(255,85,127), true)
    };

    QList<AnnotationSettings*> toAdd;
    for (int i=0, n = sizeof(predefined)/sizeof(AnnotationSettings); i<n; i++) {
        const AnnotationSettings& s = predefined[i];
        if (!persistentMap.contains(s.name)) {
            toAdd.append(new AnnotationSettings(&s));
        }
    }
    if (!toAdd.isEmpty()) {
        changeSettings(toAdd);
    }
}

//////////////////////////////////////////////////////////////////////////
AnnotationSettings::AnnotationSettings() {
    amino = false;
    color = Qt::black;
    visible = true;
}

AnnotationSettings::AnnotationSettings(const AnnotationSettings* as) {
    name = as->name;
    amino = as->amino;
    color = as->color;
    visible = as->visible;
}

AnnotationSettings::AnnotationSettings(const QString& _name, bool _amino, const QColor& _color, bool _visible) 
: name(_name), amino(_amino), color(_color), visible(_visible)
{
}

bool AnnotationSettings::equals(const AnnotationSettings* as) const {
    return name == as->name 
        && amino == as->amino
        && color == as->color
        && visible == as->visible;
}


}//namespace
