// -*- Mode: Go; indent-tabs-mode: t -*-

/*
 * Copyright (C) 2014-2015 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package snappy

import (
	"bytes"
	"regexp"
	"text/template"

	"launchpad.net/snappy/logger"
)

func verifyBusName(busName string) error {
	// http://dbus.freedesktop.org/doc/dbus-specification.html
	rs := "^[A-Za-z0-9][A-Za-z0-9_-]*(\\.[A-Za-z0-9][A-Za-z0-9_-]*)+$"
	r, err := regexp.Compile(rs)
	if err != nil {
		return err
	}

	if !r.MatchString(busName) {
		return &ErrStructIllegalContent{
			content:   busName,
			whitelist: rs,
		}
	}
	return nil
}

func genBusPolicyFile(busName string) (string, error) {
	if err := verifyBusName(busName); err != nil {
		return "", err
	}

	policyTemplate := `<?xml version="1.0"?> <!--*-nxml-*-->
<!DOCTYPE busconfig PUBLIC "-//freedesktop//DTD D-BUS Bus Configuration 1.0//EN"
        "http://www.freedesktop.org/standards/dbus/1.0/busconfig.dtd">

<!--
  This file is autogenerated by snappy
-->

<busconfig>
        <policy user="root">
                <allow own="{{.BusName}}"/>
                <allow send_destination="{{.BusName}}"/>
        </policy>

        <policy context="default">
                <allow send_destination="{{.BusName}}"/>
        </policy>
</busconfig>
`
	var templateOut bytes.Buffer
	t := template.Must(template.New("wrapper").Parse(policyTemplate))
	wrapperData := struct {
		BusName string
	}{
		busName,
	}
	if err := t.Execute(&templateOut, wrapperData); err != nil {
		// this can never happen, except we forget a variable
		logger.Panicf("Unable to execute template: %v", err)
	}

	return templateOut.String(), nil
}
