/* ==================================================== ======== ======= *
 *
 *  umsproto.hpp: Ubit Protocol for the UMS (Ubit Mouse/Message Server)
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _umsproto_hpp_
#define	_umsproto_hpp_
//pragma ident	"@(#)umsproto.hpp	ubit:03.05.03"
#include <ubit/udefs.hpp>

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Ubit Protocol for the UMS (Ubit Mouse/Message Server)
 */
struct UMSprotocol {

  static const int UMS_DEFAULT_PORT = 9666;
  /**< the default port of the UMS (Ubit Mouse/Message Server).
   * umsd (the UMS demon) is available in directory ubit/ums
   */

  static const u_id UBIT_EVENT_FLOW = 1<<14;
  /**< state field of X and U Events when an alternate mouse pointer is used.
   * Notes:
   * - UEvent::getState() return the ORed state of the event
   * - UEvent::getFlow() and getFlowID() identifies the mouse pointer
   * - the No of the mouse pointer is also stored in XEvent.subwindow
   */

  static const char *UBIT_SELECTION, *UBIT_MESSAGE, *UBIT_WINDOW;
  /**< X Window atoms that identifie Ubit actions and windows.
   * Note: the ASCII name of the atoms are "_UBIT_SELECTION", etc.
   */

  static void setUbitProtocols(class UNatDisp*, class UNatWin*);
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** UMS Request (from a Client to the UMS server).
 * <pre>
 * size    reqtype    
 * (octal) (u_char)   (c = u_char / c* = char* / l = long)
 * ----------------------------------------------------------------------
 *   s   MOUSE_CTRL   (c)evtype (c)evflow (l)x (l)y (l)btn|key|mvm
 *   s   SEND_EVENT   (c)evtype (c)evflow (l)x (l)y (l)btn|key|mvm (c*)target...
 *   s   SEND_MESSAGE (c*)target... (c*)message...
 *   s   OPEN_WIN     (l)win_no
 *   s   CLOSE_WIN    (l)win_no
 *   s   OPEN_CNX     (c*)apname...
 *   s   CLOSE_CNX    (c*)apname...
 * ----------------------------------------------------------------------
 * target can be:
 * winid[:evflow]   with winid = ascii-name-without-blanks or quoted 'asci name'
 *                  or decimal (1234) or hexa number (0x1234a)
 * #win:/[display]/winid[:evflow]    (display is opt and ... never used)
 * #ptr:/[display]/ptrid[:evflow]    with ptrid = 0, 1 ... (useless)
 * </pre>
 */
struct UMSrequest {
  enum RequestType { 
    MOUSE_CTRL        = 1,   
    SEND_EVENT        = 2,   
    SEND_MESSAGE      = 3,
    OPEN_WIN          = 5, 
    CLOSE_WIN         = 6,
    OPEN_CNX          = 7, 
    CLOSE_CNX         = 8
  };

  // comm status
  enum Stat {Error = -1, CloseCnx = 0, Ok = 1, KeepCnx = 2};

  UMSrequest();
  UMSrequest(unsigned char reqtype);

  int size() {return data[0]*8;}
  ///< returns the size of the request (which is always a multiple of 8).

  void readEvent(unsigned char& event_type, 
		 unsigned char& event_flow, 
		 long& x, long& y, unsigned long& detail);

  void writeEvent(unsigned char event_type, 
		  unsigned char event_flow, 
		  long x, long y, unsigned long detail);

  char readChar();
  void writeChar(char);

  short readShort();
  void writeShort(short);

  long readLong();
  void writeLong(long);

  bool writeString(const char*);
  char* getString();
  ///< NB: writeString() or getString must be the last call (only one string).
  
  unsigned char data[8*256];
  /**< data sent to the UMS.
    * Notes:
    * - data[0]*8 is the size of the request (we send 8 byte data blocks)
    * - data[1] is the RequestType
    */
  
  int count;
  /**< char count.
    * this variable is used by the write*() and read*() functions
    * to store or retreive the data (but it is not sent to the UMS,
    * the size/8 being eventually stored in data[0]
    */
};

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
#endif
