#ifndef SIGNALBASE_HH
#define SIGNALBASE_HH

/** Copyright (c) 1995-1999 Ohio Board of Regents and the University of
    Cincinnati.  All Rights Reserved.

    You may modify, distribute, and use the software contained in this
    package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
    version 2, June 1991. A copy of this license agreement can be found
    in the file "LGPL", distributed with this archive. */

#include <vector>
#include "VHDLKernel.hh"
#include "DefaultAttributes.hh"
#include "FanDest.hh"
#include "TransactionList.hh"

using std::vector;

class Block;
class VHDLKernel;
class SourceBase;
class SigEvent;
class Transaction;

class SignalBase : public ObjectBase {
public:

  /** fanout list size.  This variable holds the current number of entries in
      the fanout list (fanDest). */
  int getFanOut() const {
    return fanDest.size();
  }

  const string getName() const { return name; }
  int getSigId() const { return id; }
  SigType getType() const { return type; }
  const vector<FanDest *> &getFanDest() const { return fanDest; }
  int getNumAttributes() const { return attributeList.size(); }
  vector<Attribute *> &getAttributeList(){ return attributeList; }
  DefaultAttributes &getAllAttributes(){ return allAttributes; }
  int getBusResFn() const { return busResFn; }
  Block *getAdditionalDriverList() const { return additionalDriverList; }
  bool isSensitive() const { return sensitive; }

  /**  Default constructor. */
  SignalBase();

  /** Constructor */
  SignalBase( const char *initName );

  /** Constructor */  
  SignalBase( int intNumSources );

  /** Constructor */  
  SignalBase( const char *initName, int initNumSources );

  /** Constructor */  
  SignalBase( SigType initType,
	      bool initSensitive,
	      VHDLData::UniversalType initUniversalType );
  
  /** Constructor */  
  SignalBase( const SignalBase& sb );
  
  virtual ~SignalBase() {}
  
  SignalBase& operator=(const SignalBase& sb);

  /** Add VHDLKernel object to the fan out list.  Adds the given VHDLKernel
      process object to the fan out list. Optionally may the signal identifier
      be specified. */
  SignalBase &Add( VHDLKernel * );
  SignalBase &Add( VHDLKernel *, int sigid );

  /** Remove VHDLKernel object to the fan out list.  Removes the given
      VHDLKernel process object to the fan out list. Optionally may the signal
      identifier be specified. */
  SignalBase &Remove( VHDLKernel * );
  SignalBase &Remove( VHDLKernel *, int sigid );
  
  virtual bool _is_resolved() const; 

  bool _is_signal() const { return true; }
  
  void setBusKind() { type = G_BUS;  }

  virtual void print(ostream& os = cout) const;
  
  virtual ArrayInfo* readArrayInfo();
  
  virtual void updateDriver( const int, 
			     int sigSrc,
			     const VHDLData *, 
			     const ArrayInfo *,
			     const ArrayInfo * = NULL) { 
    (void)sigSrc;
    cerr << getName() << " updateDriver ERROR!" << endl;
    abort();
  }

  virtual void disconnectDriver( VHDLKernel * ){ 
    cerr << getName() << " disconnectDriver(VHDLKernel *) ERROR!" << endl;
    abort();
  }
  
  virtual VHDLType* readDriver( VHDLKernel * ){ 
    cerr << getName() << " readDriver ERROR!" << endl;
    abort();
    return NULL;
  }
  
  virtual void updateDrvVal(const VHDLData*, const ArrayInfo* = NULL,
			    const ArrayInfo* = NULL) { 
    cerr << getName() << " updateDrvVal ERROR!" << endl;
    abort();
  }
  
  virtual VHDLData* readDrvVal() { 
    cerr << getName() << " readDrvVal ERROR!" << endl;
    abort();
    return NULL;
  }
  
  virtual void updateEffVal(const VHDLData&, const ArrayInfo* = NULL,
			    const ArrayInfo* = NULL) { 
    cerr << getName() << " updateEffVal ERROR!" << endl;
    abort();
  }
  
  virtual void updateEffVal(const VHDLData*, const ArrayInfo* = NULL,
			    const ArrayInfo* = NULL) { 
    cerr << getName() << " updateEffVal ERROR!" << endl;
    abort();
  }

  virtual VHDLData* readEffVal() { 
    cerr << getName() << " readEffVal ERROR!" << endl;
    abort();
    return NULL;
  }
  
  virtual const VHDLVTime& readTimeVal() {
    cerr << getName() << " readTimeVal ERROR!" << endl;
    abort();
    // this is bogus, but it should never be called
    return VHDLVTime::getVHDLVTimePositiveInfinity();
  }
  
  virtual VHDLVTime& updateTimeVal( const VHDLVTime& v = VHDLVTime::getVHDLVTimePositiveInfinity() ){
    cerr << getName() << " updateTimeVal ERROR!" << endl;
    abort();
    return (VHDLVTime &) v;
  }

  virtual void setSource(SourceBase*) {
   cerr << "ERROR: SignalBase::setSource(SourceBase *) called" << endl;
   abort();
  } 

  void setElaborationInfo(ObjectBase& obj);

  void setAttrib(AttribType typ, VHDLType& attrib);

  virtual bool isCompositeResolvedSignal() const {
    cerr << "ERROR: SignalBase::isCompositeResolvedSignal() called " << endl;
    return false;
  }

  virtual SignalBase *findSigInBlock( int sigId, VHDLKernel *srcId );
  virtual void set_sourcebase_delete_flag(bool flag);

  virtual SourceBase *getSource();
  
  /** Assigning a value to a signal does not immediately change the value
      of the signal. Instead, only a transaction get's posted on a
      signal. Posting a transaction on a signal involves 2 main steps:

      (a) the transaction is conveyed to all processes that are sensitive to
      the signal

      (b) marking of the signal transactions is done.

      This method triggers/implements the above two steps:
      
      Post the signal transaction to all the processes that are sensitive
      to this signal i.e., to all the process entries in the fanout of this
      signal:

      Create a new transaction to be posted for this signal at the given
      transaction time... 
  */
  void assignSignal( SigType signalType, 
		     VHDLKernel *srcId, 
		     const VHDLData &src,
		     const VHDLVTime &delay, 
		     const VHDLVTime &rejTime, 
		     const ArrayInfo &destInfo,
		     const ArrayInfo &srcInfo );
  /**
     Called by TransactionList during marking.
  */
  void cancelTransaction( const Transaction * );
  
protected:
  SigEvent *generateEvent( VHDLKernel *sender,
			   const string &receiverName,
			   const Transaction *transaction,
			   int destSigId, 
			   int senderSigId,
			   SigType type, 
			   const VHDLVTime &delay, 
			   const VHDLVTime &rejTime,
			   const VHDLData &sigValue, 
			   const ArrayInfo &destInfo,
			   const ArrayInfo &srcInfo );

  
private:
  /**
     Our transaction list.
  */
  TransactionList transactionList;
  /** Signal type */
  SigType type;  
  
  /** global signal ID */
  int id;            

  /** Human-readable signal name */
  string name;

  /**  does it belong to the processes sensitivity list? */
  bool sensitive;

  /** Index of busResFn for this signal */
  int busResFn;
  int numSources;


  /** fanout list.  This contains the simulation object IDs that this
      signal gets transmitted to.  This data should never change during
      a simulation, so it is allocated only once, during initState, and
      deleted after the simulation is over.  The copy constructor and
      operator= simply copy the pointer to the new object. */

  vector<FanDest *> fanDest;
  vector<Attribute *> attributeList;
  DefaultAttributes allAttributes;

  Block *downTypeConversionFnListPtr;

  /** we should search for updating the Drivers. */
  Block *additionalDriverList;

  unsigned int getNextSequenceNumber();
  unsigned int transactionSequenceNumber;

  void cleanTransactionList( const VHDLVTime &beforeTime );

  /**
     Sends the transaction to everyone in our fanout list.
  */
  void sendTransactionToFanoutList( SigType signalType, 
				    VHDLKernel *sourceProcess, 
				    const VHDLData &src,
				    const VHDLVTime &delay, 
				    const VHDLVTime &rejTime,
				    const ArrayInfo &destInfo, 
				    const ArrayInfo &srcInfo,
				    const Transaction *newTransaction );
};

#endif
