#include <clutils/Debug.h>
#include "FanDest.hh"
#include "SignalBase.hh"
#include "VHDLType.hh"
#include "SignalNetinfo.hh"
#include "SigEvent.hh"
#include "VHDLKernel.hh"
#include "Transaction.hh"
#include <algorithm>
#include <functional>

SignalBase::SignalBase() : 
  transactionList( *this ),
  type( EXPLICIT ),
  id( -1 ),
  name( "" ),
  sensitive( false ),
  busResFn( -1 ),
  numSources( -1 ),
  downTypeConversionFnListPtr( 0 ),
  additionalDriverList( 0 ),
  transactionSequenceNumber( 0 ){ }
  
SignalBase::SignalBase( const char *initName ) :
  transactionList( *this ),
  type( EXPLICIT ),
  id( -1 ),
  name( initName ),
  sensitive( false ),
  busResFn( -1 ),
  numSources( -1 ),
  downTypeConversionFnListPtr( 0 ),
  additionalDriverList( 0 ),
  transactionSequenceNumber( 0 ){ }
  

SignalBase::SignalBase ( int initNumSources ):
  transactionList( *this ),
  type( EXPLICIT ),
  id( -1 ),
  name( "" ),
  sensitive( false ),
  busResFn( -1 ),
  numSources( initNumSources ),
  downTypeConversionFnListPtr( 0 ),
  additionalDriverList( 0 ),
  transactionSequenceNumber( 0 ){ }

SignalBase::SignalBase( SigType initType,
			bool initSensitive,
			VHDLData::UniversalType initUniversalType ) :
  transactionList( *this ),
  type( initType ),
  id( -1 ),
  name( "" ),
  sensitive( initSensitive ),
  busResFn( -1 ),
  numSources( 0 ),
  downTypeConversionFnListPtr( 0 ),
  additionalDriverList( 0 ),
  transactionSequenceNumber( 0 ){ 
  allAttributes.sigtype = initUniversalType;
}

SignalBase::SignalBase( const char *initName, 
			int initNumSources ) :
  transactionList( *this ),
  type( EXPLICIT ),
  id( -1 ),
  name( initName ),
  sensitive( false ),
  busResFn( -1 ),
  numSources( initNumSources ),
  downTypeConversionFnListPtr( 0 ),
  additionalDriverList( 0 ),
  transactionSequenceNumber( 0 ){ }
  

SignalBase::SignalBase( const SignalBase &sb ) : 
  ObjectBase(),
  transactionList( *this ),
  type( sb.type ),
  id( sb.id ),
  name( sb.name ),
  sensitive( sb.sensitive ),
  busResFn( sb.busResFn ),
  numSources( sb.numSources ),
  fanDest( sb.fanDest ),
  downTypeConversionFnListPtr( sb.downTypeConversionFnListPtr ),
  transactionSequenceNumber( sb.transactionSequenceNumber ){ 
  
  if ( sb.additionalDriverList != NULL ){
    additionalDriverList  = new Block();
    *additionalDriverList = *(sb.additionalDriverList);
  }
  else{
    additionalDriverList = 0;
  }
}

SignalBase &
SignalBase::operator=(const SignalBase &sb) {
  type          = sb.type;
  id            = sb.id;
  name          = sb.name;
  sensitive     = sb.sensitive;
  busResFn      = sb.busResFn;
  numSources    = sb.numSources;
  attributeList = sb.attributeList;
  allAttributes = sb.allAttributes;

  // Copy Fan-out list
  fanDest       = sb.fanDest;
  
  downTypeConversionFnListPtr = sb.downTypeConversionFnListPtr;
  
  if ( sb.additionalDriverList != NULL ) {
    additionalDriverList  = new Block();
    *additionalDriverList = *(sb.additionalDriverList);
  }
  else{
    additionalDriverList = 0;
  }
  
  parentCompositeType = sb.parentCompositeType;
  
  // Copy necessary transactions from one list to another...
  transactionList.copy( sb.transactionList );

  return *this;
}

// Add a VHDLKernel process object to the fan out list.

SignalBase&
SignalBase::Add( VHDLKernel* objid ){
  return Add(objid, id);
}

SignalBase&
SignalBase::Add( VHDLKernel *objid, int  sigid ){
  // If there exist a fan out list,
  
  // Check here for the destination process id and the signal id already in
  // the list.
  int present = 0;
  for(unsigned int j = 0; j < fanDest.size(); j++){
    // If there is an object having the correct process Id, and if the
    // signal identifier is different, the object signal identifier matches,
    // then we have an object in the fan out list which is.operator==ent to the
    // one added.
    
    if ( fanDest[j]->getProcessName() == objid->getName() &&
	 ( sigid == id || fanDest[j]->getSigId() == sigid )){
      present = 1;
      break;
    }
  }
  if(present == 0){
    fanDest.push_back( new FanDest(objid->getName(), sigid) );
  }
  return *this;
}

// Remove a VHDLKernel process object from the fan out list.

SignalBase&
SignalBase::Remove(VHDLKernel* objid){
  return Remove(objid, id);
}

SignalBase&
SignalBase::Remove( VHDLKernel *objid, int  sigid ){
  // If there exist a fan out list,
  
  // Check here for the destination process id and the signal id already in
  // the list.
  
  vector<FanDest *>::iterator pos = fanDest.end();
  for( unsigned int j = 0; j < fanDest.size(); j++ ){
    // If there is an object having the correct process Id, and if the
    // signal identifier is different, the object signal identifier matches,
    // then we have an object in the fan out list which is.operator==ent to the
    // one added.
    if ( fanDest[j]->getProcessName() == objid->getName() &&  
	 (sigid == id || fanDest[j]->getSigId() == sigid )){
      advance( pos, j );
      break;
    }
  }
  
  if( pos != fanDest.end() ){
    FanDest *erased = *pos;
    fanDest.erase( pos );
    delete erased;
  }

  return *this;
}

void 
SignalBase::setAttrib(AttribType typ, VHDLType &attrib) {
  Attribute *newAttribute = new Attribute();
  newAttribute->attrib = typ;
  newAttribute->value = &attrib;
  newAttribute->value->initializeImplicitSignal(typ);
  attributeList.push_back( newAttribute );
}

SignalBase*
SignalBase::findSigInBlock(int, VHDLKernel *) {
  cerr << "ERROR: SignalBase::findSigInBlock(int sigId, int srcId) called" << endl;
  return NULL;
}

void
SignalBase::set_sourcebase_delete_flag(bool){
  cerr << "ERROR: SignalBase::set_sourcebase_delete_flag(bool) called" << endl;
}

SourceBase*
SignalBase::getSource() {
  cerr << "ERROR: SignalBase::getSource() called" << endl;
  return NULL;
}

bool
SignalBase::_is_resolved() const {
  cerr << "SignalBase::_is_resolved() called.  Aborting." << endl;
  abort();
  return false;
}

void
SignalBase::print( ostream &os ) const {
  os << "SignalBase " << name << " id " << id;
  os.flush();
}

ArrayInfo*
SignalBase::readArrayInfo() {
  cerr << name << " readArrayInfo ERROR!" << endl;
  return NULL;
}

void
SignalBase::setElaborationInfo( ObjectBase &obj ){
  ASSERT (obj.getKind() == SIGNAL_NETINFO);
  
  SignalNetinfo &temp_obj     = dynamic_cast<SignalNetinfo &>(obj);
  id                          = temp_obj.id;
  setSource(temp_obj.getSource());
  
  downTypeConversionFnListPtr = &temp_obj.getDownTypeConversionFnList();
  sensitive                   = true;

  // Copy Fan-out list
  fanDest                     = temp_obj.getFanoutTable();
  
  // Copy also the additionalDriverList
  if (temp_obj.getAdditionalDriverList() != NULL) {
    additionalDriverList  = new Block();
    *additionalDriverList = *(temp_obj.getAdditionalDriverList());
  }
}

void
SignalBase::sendTransactionToFanoutList( SigType signalType, 
					 VHDLKernel *sourceProcess, 
					 const VHDLData &src,
					 const VHDLVTime &delay, 
					 const VHDLVTime &rejTime,
					 const ArrayInfo &destInfo, 
					 const ArrayInfo &srcInfo,
					 const Transaction *newTransaction ){
  for( int i = 0; i < getFanOut(); i++) {
    SigEvent *transEvent = generateEvent( sourceProcess,
					  fanDest[i]->getProcessName(),
					  newTransaction, 
					  fanDest[i]->getSigId(),
					  id, 
					  signalType, 
					  delay,
					  rejTime, 
					  src, 
					  destInfo, 
					  srcInfo );
    transEvent->setSigSrc( sourceProcess->getVHDLProcessId() );

#ifdef VHDLDBG
    cout << "Sending event " << *transEvent << endl;
#endif

    fanDest[i]->sendEvent( transEvent, sourceProcess );
  }
}

void
SignalBase::assignSignal( SigType signalType, 
			  VHDLKernel *sourceProcess, 
			  const VHDLData &src,
			  const VHDLVTime &delay, 
			  const VHDLVTime &rejTime,
			  const ArrayInfo &destInfo, 
			  const ArrayInfo &srcInfo ){

  ASSERT ( sourceProcess != NULL );
  
  VHDLVTime transactionTime(sourceProcess->getTimeNow() + delay);
  Transaction *newTransaction = new Transaction( sourceProcess,
						 transactionTime,
						 getNextSequenceNumber(),
						 src.clone() );
  sendTransactionToFanoutList( signalType,
			       sourceProcess,
			       src,
			       delay,
			       rejTime,
			       destInfo,
			       srcInfo,
			       newTransaction );

  if( !transactionList.empty() ){
    transactionList.clean( sourceProcess->getTimeNow() );
    transactionList.doMarking( transactionTime,
			       newTransaction,
			       rejTime );
  }

}
  
SigEvent*
SignalBase::generateEvent( VHDLKernel *sender,
			   const string &receiverName,
			   const Transaction *transaction, 
			   int destSigId, 
			   int senderSigId, 
			   SigType type, 
			   const VHDLVTime &, 
			   const VHDLVTime &rejTime,
			   const VHDLData &sigValue, 
			   const ArrayInfo &destInfo,
			   const ArrayInfo &srcInfo ){
  ASSERT( transaction != 0 );

  // This method generates an event for a given destination when a transaction
  // gets posted on a signal.
  
  ASSERT (srcInfo.length() == destInfo.length());

  SigEvent *newEvent = new SigEvent( sender->getTimeNow(), // Should be "now()"
				     transaction->getTime(),
				     sender->getName(),
				     receiverName,
				     type,
				     destSigId,
				     0,
				     rejTime,
				     srcInfo,
				     destInfo,
				     senderSigId,
				     transaction->getSequenceNumber(),
				     NORMALTRANSACTION,
				     sigValue );
  
  return newEvent;
}


void
SignalBase::cancelTransaction( const Transaction* transaction ){
  // A transaction posted on this signal is being removed due to the marking
  // process. We now send a cancel message to each process indicating that a 
  // given transaction is being cancelled.
  VHDLKernel *sourceProcess = transaction->getSourceProcess();

  if( transaction->getTime() <= sourceProcess->getTimeNow() ){
    // cerr << "Warning: Attempt to cancel transaction at time "
    // << transaction->getTime() << " at simulation time " 
    // << process->getTimeNow() << ".  This operation cannot be "
    // << "performed since it violates semantics of DES and is ignored.\n";
    return;
  }
 
  for( int i = 0; i < getFanOut(); i++ ){
    SigEvent *transEvent = generateEvent( sourceProcess,
					  fanDest[i]->getProcessName(),
					  transaction, 
					  fanDest[i]->getSigId(),
					  id, 
					  type,
					  VHDLVTime::getVHDLVTimeZero(), 
					  VHDLVTime::getVHDLVTimeZero(),
					  SAVANT_BOOLEAN_TRUE.readVal(), 
					  nullInfo,
					  nullInfo );
    
    transEvent->setTransactionType( CANCELTRANSACTION );
    // Ask the process to dispatch this cancel Transaction event for us...
    transEvent->setSigSrc( sourceProcess->getVHDLProcessId() );
    fanDest[i]->sendEvent( transEvent, sourceProcess );
  }
}

unsigned int
SignalBase::getNextSequenceNumber(){
  return transactionSequenceNumber++;
}
