/*
 * twclock:  A world clock implemented with Motif widgets
 * Copyright (C) 1997 Ted Williams WA0EIR
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge. MA 02139, USA.
 * See the COPYING file in this directory
 *
 * Version: 2.1 - Mar 2003
 */

/*
 * CALLBACKS FOR TWCLOCK 
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

#include "twclock.h"
char timepath[100];
Pixel saveColor;

/*
 * CATCH INPUT EVENTS
 * This catches all input events in the GUI.  If it was a
 * button 3 press, then popup the menu, else go away.
 */
void input_event (Widget w, XtPointer client_data, XEvent *event, Boolean *cont)
{
   Widget popup_menu = (Widget) client_data;

   if(event->type == ButtonPress && event->xbutton.button == Button3)
   {
      XmMenuPosition(popup_menu, &(event->xbutton));
      XtManageChild(popup_menu); 
   }
}


/*
 * POPUP MENU CALLBACK
 * All popup menu button presses come here.  The switch checks for
 * which menu button was pressed and either sets the TZ variable 
 * to Local, GMT or, in the case of the Others: button, pops
 * up the file selection box dialog.  The tzset() forces the
 * new value in TZ to be recognized.  The clock will adjust
 * to the new time on the next update.
 */
void popup_cb(Widget w, XtPointer client_data, XtPointer cbs)
{
   int set_zone = (int) client_data;
   Widget zone_fsb;

   switch (set_zone)
   {
      case LOCAL_BTN:
         strcpy (timepath, "");
         doit = 1;     /* force the date label to be updated next time */
         break;

      case GMT_BTN:
         strcpy (timepath, "TZ=GMT");
         doit = 1;     /* force the date label to be updated next time */
         break;

      case OTHERS_BTN:
         zone_fsb = XmCreateFileSelectionDialog(clock_shell, "zone_fsb",
            (ArgList) NULL, 0);

         XtVaSetValues (XtParent(zone_fsb),
            XmNtitle, "Timezone Selection",
            NULL);

         XtVaSetValues (zone_fsb,
            XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
            NULL);

         /* Unmanage some fsb widgets */
         XtUnmanageChild (XmFileSelectionBoxGetChild (zone_fsb,
            XmDIALOG_FILTER_TEXT));

         XtUnmanageChild (XmFileSelectionBoxGetChild (zone_fsb,
            XmDIALOG_FILTER_LABEL));

         XtUnmanageChild (XmFileSelectionBoxGetChild (zone_fsb,
            XmDIALOG_HELP_BUTTON));

         /* Add callbacks for file selection box */
         XtAddCallback (zone_fsb, XmNokCallback, fsbOkCB,
               (XtPointer) NULL);

         /* XtAddCallback (zone_fsb, XmNapplyCallback, fsbOkCB,
            (XtPointer) NULL); */

         XtAddCallback (zone_fsb, XmNcancelCallback, fsbCancelCB,
            (XtPointer) NULL);

         XtManageChild (zone_fsb);   /* Show the file selection box */
         break;

      case SET_TIMER_BTN:
         settimer();
         break;

      default:
         fprintf (stderr, "Error: invalid menu button\n");
         break;
   }
}
 

/*
 * GETTIME FUNCTION
 * A call to this function is hardcoded near the end of widget
 * creation.  At the end of this function, it registers a timeout
 * that forces itself to be called again in one second.  We're off!
 *
 * This function calls localtime(), which bases time on TZ.  The
 * time values are then set into the widgets.  No need to put the
 * same string in the date label 86400 times a day.  So, if the day
 * of the week value has changed, change the label string.  Or, if
 * doit is true, we update the label.  popup_cb() sets doit when
 * the time zone is changed, which requires updating the label.
 */
void  gettime ()
{
   time_t current;
   struct tm *timestruct;
   char  datestr[40];
   static int oldday;
   XmString labelstr;
   XtIntervalId time_id;

   unsigned long time_delta = 1000;

   putenv (timepath);                   /* put selected time in TZ */
   time (&current);
   timestruct = localtime (&current);   /* get localtime as per TZ */

   XtVaSetValues (hr_scale,  XmNvalue, timestruct->tm_hour, NULL);
   XtVaSetValues (min_scale, XmNvalue, timestruct->tm_min,  NULL);
   XtVaSetValues (sec_scale, XmNvalue, timestruct->tm_sec,  NULL);

   if ((timestruct->tm_mday != oldday) | doit)
   {
      strftime (datestr, sizeof(datestr), "%e %b %Y %Z", timestruct);
      labelstr = XmStringCreateLocalized (datestr);

      XtVaSetValues (date_label,
         XmNlabelString, labelstr,
         NULL);

      XmStringFree (labelstr);
      oldday = timestruct->tm_mday;
      doit = 0;
   }
   
   /* register the time out again */
   time_id=XtAppAddTimeOut(app, time_delta,
      (XtTimerCallbackProc)gettime, (XtPointer)NULL);
}


/*
 * FSB OK CALLBACK
 * This function gets data from the fsb and builds a string of
 * the form TZ=pathname.  If the pathname ends with a "/",
 * then you didn't select a city in the fsb, so exit and leave
 * TZ alone.  Otherwise, set TZ to the new timezone and set doit.
 * This forces gettime() to use the new TZ and update the date label.
 */
void fsbOkCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XmFileSelectionBoxCallbackStruct *fsbcbs =
      (XmFileSelectionBoxCallbackStruct *) cbs;

   int  len;
   char *path;

   XmStringGetLtoR (fsbcbs->value, XmFONTLIST_DEFAULT_TAG, &path);

   len = strlen (path);
   if (path[len-1] != '/')       /* If the path ends with /, then do nothing */
   {
      strcpy (timepath, "TZ=");  /* build the new TZ value */
      strcat (timepath, path);   /* and put it in timepath */
      doit = 1;                  /* force date label to be updated next time */

   }
   /* XtUnmanageChild (w); */
   XtDestroyWidget (XtParent (w)); /* Remove the fsb */
   XtFree (path);
}


/*
 * FSB CANCEL CALLBACK
 * Cancel button was pressed, so unmanage the file selection box
 */
void fsbCancelCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XtDestroyWidget(XtParent (w));
}


/*
 * TBCHANGECB CALLBACK FUNCTION
 * Registers or unregisters the ID timeout, depending on the state of the
 * toggle button.
 */
void TBchangeCB(Widget w, XtPointer cdata, XtPointer cbs)
{

   XmToggleButtonCallbackStruct *alarmCBS=(XmToggleButtonCallbackStruct *)cbs;

   switch(alarmCBS->set)
   {
      case True:
         flashFlag = TRUE;
         XtVaGetValues (w,
            XmNselectColor, &saveColor,
            NULL);

         alarm_id = XtAppAddTimeOut(app,
                       ((60 * appRes.minutes) + appRes.seconds) * 1000,
                       (XtTimerCallbackProc)timer, (XtPointer)NULL);
         break;

      case False:
         XtVaSetValues (w,
            XmNselectColor, saveColor,
            NULL);
         XtRemoveTimeOut (alarm_id);
         break;

      default:
         fprintf (stderr, "Error: invalid toggle button state\n");
         break;
  }
} 

/*
 * ALARM TIMER CALLBACK
 * First called when the ID timer times out.  It will blinks the toggle button
 * and beep as requested in the resource file.  It then changes the timer
 * delay to 1 sec and registers another timeout.  Unseting the toggle button
 * unregisters the timeout.
 */
void timer (void)
{
   FILE *con;

   if ((con = fopen ("/dev/console", "w")) == 0)
   {
      fprintf (stderr, "can't open console\n");
   }
   else
   {
      /* flash and beep */
      if (appRes.beep == TRUE)
      {
         fprintf (con, "\a");
      }
      fclose (con);
   }

   if (appRes.blink == TRUE && flashFlag == TRUE)
   {
      XtVaSetValues (call_toggleB,
         XmNselectColor, appRes.flashColor,
         NULL);
      flashFlag = FALSE;
   }
   else
   {
      XtVaSetValues (call_toggleB,
         XmNselectColor, saveColor,
         NULL);
      flashFlag = TRUE;
      
   }
   alarm_id=XtAppAddTimeOut(app, 1000,(XtTimerCallbackProc)timer, NULL);   
}


/*
 * SETTIMER FUNCTION
 * Create a dialog box to input new time
 * Add callbacks for min, sec, and close
 */
void settimer (void)
{
   char minutes[10], seconds[10];
   static Widget settimerDiag = NULL;
   static Widget timeTFs[2];
   Widget setForm, min_label, sec_label;
   XmString set;
   
   /* Convert the current minutes and seconds to strings */ 
   sprintf (minutes, "%d", appRes.minutes);
   sprintf (seconds, "%d", appRes.seconds);

   /* Lazy Create */   
   if (settimerDiag != NULL)
   {

     XtVaSetValues (timeTFs[0],
         XmNvalue, minutes,
         NULL);

      XtVaSetValues (timeTFs[1],
         XmNvalue, seconds,
         NULL);

      XtManageChild (settimerDiag);
      return;
   }

   /* Create a message box dialog and set the dialog shell values */   
   settimerDiag = XmCreateMessageDialog (clock_shell, "settimer",
     (ArgList) NULL, 0);
   
   XtVaSetValues (XtParent(settimerDiag),
      XmNtitle, "Set Timer",
      NULL); 

   set = XmStringCreateLocalized ("Set");

   XtVaSetValues (settimerDiag,          
      XmNokLabelString, set,
      XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
      NULL);
   
   XmStringFree(set);

   /* Don't want a help button */
   XtUnmanageChild (XmMessageBoxGetChild (settimerDiag,XmDIALOG_HELP_BUTTON));   
   /* Add form, text fields and labels to the message box workspace */
   setForm = XtVaCreateWidget ("setForm", xmFormWidgetClass, settimerDiag,
         NULL);
   
   min_label = XtVaCreateManagedWidget ("Minutes", xmLabelWidgetClass, setForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);
         
   timeTFs[0] = XtVaCreateManagedWidget ("settimerDiag",
      xmTextFieldWidgetClass, setForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, min_label,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      XmNvalue, minutes,
      XmNmaxLength, 4,
      XmNcolumns, 4,
      NULL);

   sec_label = XtVaCreateManagedWidget ("Seconds", xmLabelWidgetClass, setForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, timeTFs[0],
      XmNbottomAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);
   
   timeTFs[1] = XtVaCreateManagedWidget ("settimerDiag",
      xmTextFieldWidgetClass, setForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, sec_label,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      XmNvalue, seconds,
      XmNmaxLength, 4,
      XmNcolumns, 4,
      NULL);

   /* Add the callbacks and 2 modifyVerify */

   XtAddCallback (settimerDiag, XmNokCallback, timerOkCB,
               (XtPointer) timeTFs);

   XtAddCallback (timeTFs[0], XmNmodifyVerifyCallback, timerModVerCB,
               (XtPointer) NULL);

   XtAddCallback (timeTFs[1], XmNmodifyVerifyCallback, timerModVerCB,
               (XtPointer) NULL);

   /* Pop up the timer dialog */
   XtManageChild (setForm);
   XtManageChild (settimerDiag);
}


/*
 * Callbacks for setting the timer
 */

void timerOkCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   Widget *wids = (Widget *) client_data;
   char *minStr, *secStr;
   int min, sec;

   minStr = XmTextFieldGetString (wids[0]);
   secStr = XmTextFieldGetString (wids[1]);

   min = atoi(minStr);
   sec = atoi(secStr);

   /* just in case */
   min = min + sec / 60;
   sec = sec % 60;

   appRes.minutes = min;
   appRes.seconds = sec;
}


void timerModVerCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int i;
   XmTextVerifyCallbackStruct *pt = (XmTextVerifyCallbackStruct *) cbs;

   for (i=0; i < pt->text->length; i++)
   {
      if ( !isdigit (pt->text->ptr[i]))  /* allow only digits, beep if not */
      {
         pt->doit = False;
      }
   }
}
