/**
 * Copyright (C) 2001, 2002, 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "sdloutput.h"

#ifdef HAVE_SDL

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>
#include <ctype.h>
#include <SDL/SDL.h>
#include "utils.h"
#include "input.h"

const int resize_frame_delay = 10;
const int sdlflags = SDL_HWSURFACE | SDL_RESIZABLE;

static SDL_Surface *screen;
static SDL_Overlay *frame;
static int sdlaspect;
static int outwidth;
static int outheight;
static int fs;
static int needresize;
static int window_width;
static int window_height;

static unsigned char *sdl_get_output( void )
{
    return frame->pixels[ 0 ];
}

static void sdl_reset_display( void )
{
    screen = SDL_SetVideoMode( outwidth, outheight, 0, sdlflags );
}

static int sdl_init( int outputheight, int aspect, int verbose )
{
    int ret;

    sdlaspect = aspect;
    outheight = outputheight;

    if( sdlaspect ) {
        outwidth = (outheight / 9) * 16;
    } else {
        outwidth = (outheight / 3) * 4;
    }

    frame = 0;
    fs = 0;
    needresize = 0;
    window_width = 0;
    window_height = 0;

    /* Initialize SDL. */
    ret = SDL_Init( SDL_INIT_VIDEO );
    if( ret < 0 ) {
        fprintf( stderr, "sdloutput: SDL_Init failed.\n" );
        return 0;
    } 

    /* Create screen surface. */
    /* Unfortunately, we always assume square pixels for now. */
    screen = SDL_SetVideoMode( outwidth, outheight, 0, sdlflags );
    if( !screen ) {
        fprintf( stderr, "sdloutput: Can't open output!\n" );
        return 0;
    }

    SDL_ShowCursor( 0 );
    SDL_EnableKeyRepeat( SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL );
    return 1;
}

static int sdl_set_input_size( int inputwidth, int inputheight )
{
    /* Create overlay surface. */
    frame = SDL_CreateYUVOverlay( inputwidth, inputheight,
                                  SDL_YUY2_OVERLAY, screen );
    if( !frame ) {
        fprintf( stderr, "sdloutput: Couldn't create overlay surface.\n" );
        return 0;
    }

    if( !frame->hw_overlay ) {
        fprintf( stderr, "sdloutput: Can't get a hardware overlay surface!!\n"
                         "sdloutput: Using SDL's software fallbacks!  Performance will be poor!\n" );
    }

    return 1;
}

static int sdl_toggle_fullscreen( int fullscreen_width, int fullscreen_height )
{
    if( !fs && fullscreen_width && fullscreen_height ) {
        screen = SDL_SetVideoMode( fullscreen_width, fullscreen_height, 0, sdlflags );
        window_width = outwidth;
        window_height = outheight;
        outwidth = fullscreen_width;
        outheight = fullscreen_height;
    }

    SDL_WM_ToggleFullScreen( screen );

    if( fs ) {
        if( window_width && window_height ) {
            outwidth = window_width;
            outheight = window_height;
        }
        sdl_reset_display();
    }

    fs = !fs;
    return fs;
}

static int sdl_toggle_aspect( void )
{
    sdlaspect = !sdlaspect;
    sdl_reset_display();
    return sdlaspect;
}

static int sdl_show_frame( int x, int y, int width, int height )
{
    SDL_Rect r;
    int curwidth;
    int curheight;
    int widthratio = sdlaspect ? 16 : 4;
    int heightratio = sdlaspect ? 9 : 3;

    if( needresize == 1 ) {
        sdl_reset_display();
    }
    if( needresize ) needresize--;

    curwidth = outwidth;
    curheight = ( outwidth / widthratio ) * heightratio;
    if( curheight > outheight ) {
        curheight = outheight;
        curwidth = ( outheight / heightratio ) * widthratio;
    }

    r.x = ( outwidth - curwidth ) / 2;
    r.y = ( outheight - curheight ) / 2;
    r.w = curwidth;
    r.h = curheight;

    SDL_DisplayYUVOverlay( frame, &r );
    return 1;
}

static void sdl_poll_events( input_t *in )
{
    SDL_Event event;
    int curcommand = 0, arg = 0;
    SDLMod mods;

    while( SDL_PollEvent( &event ) ) {

        if( event.type == SDL_QUIT ) {
            curcommand = I_QUIT;
            input_callback( in, curcommand, arg );
        }

        if( event.type == SDL_VIDEORESIZE ) {
            outwidth = event.resize.w;
            outheight = event.resize.h;
            needresize = resize_frame_delay;
        }

        if( event.type == SDL_KEYDOWN ) {
            curcommand = I_KEYDOWN;

            mods = event.key.keysym.mod;
            if( mods & KMOD_SHIFT ) arg |= I_SHIFT;
            if( mods & KMOD_CTRL ) arg |= I_CTRL;
            if( mods & (KMOD_META | KMOD_ALT) ) arg |= I_META;

            switch ( event.key.keysym.sym ) {

            case SDLK_KP_PLUS:     arg |= '+'; break;
            case SDLK_KP_MINUS:    arg |= '-'; break;
            case SDLK_KP_DIVIDE:   arg |= '/'; break;
            case SDLK_KP_PERIOD:   arg |= '.'; break;
            case SDLK_KP_MULTIPLY: arg |= '*'; break;
            case SDLK_KP_EQUALS:   arg |= '='; break;

            case SDLK_KP0: arg |= '0'; break;
            case SDLK_KP1: arg |= '1'; break;
            case SDLK_KP2: arg |= '2'; break;
            case SDLK_KP3: arg |= '3'; break;
            case SDLK_KP4: arg |= '4'; break;
            case SDLK_KP5: arg |= '5'; break;
            case SDLK_KP6: arg |= '6'; break;
            case SDLK_KP7: arg |= '7'; break;
            case SDLK_KP8: arg |= '8'; break;
            case SDLK_KP9: arg |= '9'; break;

            case SDLK_KP_ENTER:
                arg |= I_ENTER;
                break;

            default:
                if( (mods & KMOD_SHIFT) || (mods & KMOD_CAPS) ) {
                    arg |= toupper( event.key.keysym.sym );
                } else {
                    arg |= event.key.keysym.sym;
                }
                break;
            }

            input_callback( in, curcommand, arg );
        }

        if( event.type == SDL_MOUSEBUTTONDOWN ) {
            input_callback( in, I_BUTTONPRESS, event.button.button );
        }
    }
}

static void sdl_quit( void )
{
    if( fs ) {
        SDL_WM_ToggleFullScreen( screen );
        fs = 0;
    }
    SDL_FreeYUVOverlay( frame );
    SDL_Quit();
}

static int sdl_get_stride( void )
{
    return frame->w * 2;
}

static int sdl_is_interlaced( void )
{
    return 0;
}

static int sdl_can_read_from_buffer( void )
{
    return 1;
}

static int sdl_is_exposed( void )
{
    return 1;
}

static int sdl_is_fullscreen( void )
{
    return fs;
}

static int sdl_is_fullscreen_supported( void )
{
    return 1;
}

static int sdl_is_alwaysontop_supported( void )
{
    return 0;
}

static int sdl_is_alwaysontop( void )
{
    return 0;
}

static void sdl_wait_for_sync( int field )
{
}

static void sdl_lock_output( void )
{
    SDL_LockYUVOverlay( frame );
}

static void sdl_unlock_output( void )
{
    SDL_UnlockYUVOverlay( frame );
}
static void sdl_set_window_caption( const char *caption )
{
    SDL_WM_SetCaption( caption, 0 );
}

static int sdl_get_visible_width( void )
{
    int curwidth;
    int curheight;
    int widthratio = sdlaspect ? 16 : 4;
    int heightratio = sdlaspect ? 9 : 3;

    curwidth = outwidth;
    curheight = ( outwidth / widthratio ) * heightratio;
    if( curheight > outheight ) {
        curheight = outheight;
        curwidth = ( outheight / heightratio ) * widthratio;
    }

    return curwidth;
}

static int sdl_get_visible_height( void )
{
    int curwidth;
    int curheight;
    int widthratio = sdlaspect ? 16 : 4;
    int heightratio = sdlaspect ? 9 : 3;

    curwidth = outwidth;
    curheight = ( outwidth / widthratio ) * heightratio;
    if( curheight > outheight ) {
        curheight = outheight;
        curwidth = ( outheight / heightratio ) * widthratio;
    }

    return curheight;
}

static void sdl_resize_window_fullscreen( void )
{
}

static void sdl_set_window_position( int x, int y )
{
}

static void sdl_set_window_height( int window_height )
{
    outheight = window_height;
    if( sdlaspect ) {
        outwidth = (outheight * 16) / 9;
    } else {
        outwidth = (outheight * 4) / 3;
    }
    sdl_reset_display();
}

static void sdl_set_fullscreen_position( int pos )
{
}

static void sdl_set_matte( int width, int height )
{
}

static int sdl_toggle_alwaysontop( void )
{
    return 0;
}

static int sdl_is_overscan_supported( void )
{
    return 0;
}

static output_api_t sdloutput =
{
    sdl_init,

    sdl_set_input_size,

    sdl_lock_output,
    sdl_get_output,
    sdl_get_stride,
    sdl_can_read_from_buffer,
    sdl_unlock_output,

    sdl_is_exposed,
    sdl_get_visible_width,
    sdl_get_visible_height,
    sdl_is_fullscreen,
    sdl_is_alwaysontop,

    sdl_is_fullscreen_supported,
    sdl_is_alwaysontop_supported,
    sdl_is_overscan_supported,

    sdl_is_interlaced,
    sdl_wait_for_sync,

    sdl_show_frame,

    sdl_toggle_aspect,
    sdl_toggle_alwaysontop,
    sdl_toggle_fullscreen,
    sdl_resize_window_fullscreen,

    sdl_set_window_caption,
    sdl_set_window_position,
    sdl_set_window_height,
    sdl_set_fullscreen_position,
    sdl_set_matte,

    sdl_poll_events,
    sdl_quit
};

output_api_t *get_sdl_output( void )
{
    return &sdloutput;
}

#else

output_api_t *get_sdl_output( void )
{
    return 0;
}

#endif
