//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#include "tulip/Graph.h"
#include "tulip/ConnectedTest.h"
#include "tulip/TreeTest.h"
#include "tulip/AcyclicTest.h"
#include "tulip/ForEach.h"

using namespace std;
using namespace tlp;
#ifdef _WIN32 
#ifdef DLL_EXPORT
TreeTest * TreeTest::instance=0;
#endif
#else
TreeTest * TreeTest::instance=0;
#endif
TreeTest::TreeTest(){
}

bool TreeTest::isTree(Graph *graph) {
  if (instance==0)
    instance=new TreeTest();
  return instance->compute(graph);
}

//====================================================================
//Determines if a graph is a topological tree.  This means that
//if the graph were undirected, there would be no cycles.
bool TreeTest::isFreeTree(Graph *graph) {
  if (instance==0) instance = new TreeTest();
  MutableContainer<bool> visited;
  visited.setAll (false);
  node firstNode = graph->getOneNode();
  return (firstNode.isValid() &&
	  instance->isFreeTree (graph, firstNode, firstNode,
				visited))
    ? ConnectedTest::isConnected(graph)
    : false;
}//isFreeTree

//====================================================================
//Turns a topological tree graph into a directed tree starting at
//the node root.
void TreeTest::makeDirectedTree(Graph *graph, node root) {
  if (instance==0) instance=new TreeTest();
  graph->removeObserver(instance);
  instance->resultsBuffer.erase((unsigned long)graph);
  if (!graph->isElement (root)) {
    cerr << "makeDirectedTree:  Passed root is not element of graph" << endl;
    return;
  }//end if
  if (!TreeTest::isFreeTree (graph)) {
    cerr << "makeDirectedTree:  Graph is not topological tree, so directed " 
	 << "tree cannot be made." << endl;
    return;
  }//end if
  instance->makeDirectedTree (graph, root, root);
  assert (TreeTest::isTree (graph));
}//end makeDirectedTree

//====================================================================
//Determines if the passed graph is topologically a tree.  The 
//passed mutable container returns if we have visited a node
bool TreeTest::isFreeTree (Graph *graph, node curRoot, node cameFrom,
			   MutableContainer<bool> &visited) {
  if (visited.get (curRoot.id)) return false;
  visited.set (curRoot.id, true);
  node curNode;
  forEach (curNode, graph->getInOutNodes(curRoot)) { 
    if (curNode != cameFrom)
      if (!isFreeTree (graph, curNode, curRoot, visited)) return false;
  }//end forEach
  return true;
}//end isFreeTree 

//====================================================================
//given that graph is topologically a tree, The function turns graph
//into a directed tree.
void TreeTest::makeDirectedTree (Graph *graph, node curRoot, node cameFrom) {
  edge curEdge;
  forEach (curEdge, graph->getInOutEdges(curRoot)) {
    node opposite = graph->opposite(curEdge, curRoot);
    if (opposite != cameFrom) {
      if (graph->target (curEdge) == curRoot)
	graph->reverse(curEdge);
      makeDirectedTree (graph, opposite, curRoot);
    }//end if
  }//end forEach
}//end makeDirectedTree

//====================================================================

bool TreeTest::compute(Graph *graph) { 
  if (resultsBuffer.find((unsigned long)graph)!=resultsBuffer.end()) {
    return resultsBuffer[(unsigned long)graph];
  }
  if (graph->numberOfEdges()!=graph->numberOfNodes()-1) {
    resultsBuffer[(unsigned long)graph]=false;
    graph->addObserver(this);
    return false;
  }
  bool rootNodeFound=false;
  Iterator<node> *it=graph->getNodes();
  while (it->hasNext()) {
    node tmp=it->next();
    if (graph->indeg(tmp)>1) {
      delete it;
      resultsBuffer[(unsigned long)graph]=false;
      graph->addObserver(this);
      return false;
    }
    if (graph->indeg(tmp)==0) {
      if (rootNodeFound) {
	delete it;
	resultsBuffer[(unsigned long)graph]=false;
	graph->addObserver(this);
	return false;
      }
      else
	rootNodeFound=true;
    }
  } delete it;
  if (AcyclicTest::isAcyclic(graph)) {
    resultsBuffer[(unsigned long)graph]=true;
    graph->addObserver(this);
    return true;
  }
  else {
    resultsBuffer[(unsigned long)graph]=false;
    graph->addObserver(this);
    return false;
  }
}

void TreeTest::addEdge(Graph *graph,const edge) {
  graph->removeObserver(this);
  resultsBuffer.erase((unsigned long)graph);
}
void TreeTest::delEdge(Graph *graph,const edge) {
  graph->removeObserver(this);
  resultsBuffer.erase((unsigned long)graph);
}
void TreeTest::reverseEdge(Graph *graph,const edge) {
  graph->removeObserver(this);
  resultsBuffer.erase((unsigned long)graph);
}
void TreeTest::addNode(Graph *graph,const node) {
  graph->removeObserver(this);
  resultsBuffer.erase((unsigned long)graph);
}
void TreeTest::delNode(Graph *graph,const node) {
  graph->removeObserver(this);
  resultsBuffer.erase((unsigned long)graph);
}

void TreeTest::destroy(Graph *graph) {
  graph->removeObserver(this);
  resultsBuffer.erase((unsigned long)graph);
}
