//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#if (QT_REL == 3)
#include <qevent.h>
#include <qimage.h>
#include <qgl.h>
#else
#ifdef  _WIN32
// compilation pb workaround
#include <windows.h>
#endif
#include <QtGui/qevent.h>
#include <QtGui/qimage.h>
#include <QtGui/qtooltip.h>
#include "tulip/Qt3ForTulip.h"
#endif

#include "tulip/GlGraphWidget.h"

#include <tulip/Graph.h>
#include <tulip/Iterator.h>
#include <tulip/BooleanProperty.h>
#include <tulip/LayoutProperty.h>
#include <tulip/SizeProperty.h>
#include <tulip/GlGraph.h>
#include <tulip/GlTools.h>
#include <tulip/StlIterator.h>

using namespace std;
using namespace tlp;

// indicates if we can deal with GL_AUX* buffers
bool _glAuxBufferAvailable = false;
// This has to be done just before the first time we draw
// to ensure a right QGLContext already exist
static void checkIfGlAuxBufferAvailable() {
  static bool done = false;
  if (!done) {
    GLint res;
    glGetIntegerv(GL_AUX_BUFFERS, &res);
    tlp::glTest(__PRETTY_FUNCTION__);
    _glAuxBufferAvailable = (res > 0);
    done = true;
#ifndef NDEBUG
    if (_glAuxBufferAvailable)
      cerr << "Auxillary Buffer Available" << endl;
    else
      cerr << "Auxillary Buffer Not Available" << endl;
#endif
  }
}
//==================================================
static void setRasterPosition(unsigned int x, unsigned int y) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  float val[4];
  unsigned char tmp[10];
  glGetFloatv(GL_CURRENT_RASTER_POSITION, (float*)&val);
  glBitmap(0,0,0,0,-val[0] + x, -val[1] + y, tmp);
  glGetFloatv(GL_CURRENT_RASTER_POSITION, (float*)&val);
  tlp::glTest(__PRETTY_FUNCTION__);
}
//==================================================
static QGLFormat GlInit() {
  QGLFormat tmpFormat;
  tmpFormat.setDirectRendering(true);
  tmpFormat.setDoubleBuffer(true);
  tmpFormat.setAccum(false);
  tmpFormat.setStencil(true);
  tmpFormat.setOverlay(false);
  tmpFormat.setDepth(true);
  tmpFormat.setRgba(true);
  tmpFormat.setAlpha(true);
  tmpFormat.setOverlay(false);
  tmpFormat.setStereo(false);
  return tmpFormat;
}
//==================================================
GlGraphWidget::GlGraphWidget(QWidget *parent, const char *name):
  QGLWidget(GlInit(), parent, name),
  _id(GWInteractor::invalidID) {
#if (QT_REL == 3)
  setName(name);
#else
  setObjectName(name);
#endif
  //  cerr << __PRETTY_FUNCTION__ << endl;
  TRACE_EXEC();
  setFocusPolicy(StrongFocus);
  _composite = new GlADComposite();
}
//==================================================
GlGraphWidget::~GlGraphWidget() {
  if (_composite != 0 ) {
    _composite->reset(false);
    delete _composite;
  }
  for (unsigned int i = 0; i > _interactors.size(); ++i)
    delete _interactors[i];
}
//==================================================
void GlGraphWidget::paintEvent( QPaintEvent* e) {
#if (QT_REL == 3)
  QRegion rect = this->clipRegion();
#else
  QRegion rect = this->visibleRegion();
#endif
  //If the visible are changed we need to draw the entire scene
  //Because the saved snapshot only backup the visibile part of the
  //Graph
  if (rect.boundingRect() != _visibleArea.boundingRect()) {
    _visibleArea = rect;
    draw();
  }
  else {
    redraw();
  }
  _visibleArea = rect; //Save the new visible area.
}
//==================================================
void GlGraphWidget::closeEvent(QCloseEvent *e) {
  emit closing(this, e);
}
//==================================================
void GlGraphWidget::setupOpenGlContext() {
  //  cerr << __PRETTY_FUNCTION__ << " (" << (int)this << ")" << endl;
  assert(context()->isValid());
  makeCurrent();
}
//==================================================
void GlGraphWidget::paintGL() { 
  //  cerr << __PRETTY_FUNCTION__ << endl;
}
//==================================================
void GlGraphWidget::glInit() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
}
//==================================================
void GlGraphWidget::glDraw() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
}
//==================================================
void GlGraphWidget::updateGL() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
}
//==================================================
void GlGraphWidget::initializeGL() {
  //  cerr << __PRETTY_FUNCTION__ << endl;
}
//==================================================
void GlGraphWidget::redraw() {
  if (isDrawing()) return; //the graph drawn should be tested 
  //when multithreading will be enable.
  checkIfGlAuxBufferAvailable();
  //Gl_AUX is not available we must redraw everything
  //TODO : make copy into memory in that case
  if (!_glAuxBufferAvailable) {
    draw();
    return;
  }
   
  makeCurrent();
  //Restore the graph image
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_STENCIL_TEST);
  glDrawBuffer(GL_BACK);
  glReadBuffer(GL_AUX0);
  setRasterPosition(0,0);
  glDisable(GL_BLEND);
  glDisable(GL_LIGHTING);
  glCopyPixels(0,0,width(), height(), GL_COLOR);
  //draw the glAugmented on top of the graph
  _composite->draw(this);
  //draw the interactors on top of graph and augmented display
  for(vector<GWInteractor *>::iterator it =
	_interactors.begin(); it != _interactors.end(); ++it) {
    if ((*it)->draw(this))
      break;
  }
  tlp::glTest(__PRETTY_FUNCTION__);
  swapBuffers();
}
//==================================================
bool GlGraphWidget::isDrawing() {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << " : not implemented" << endl;
#endif
  return false;
}
//==================================================
void GlGraphWidget::draw() {
  checkIfGlAuxBufferAvailable();
  makeCurrent();
  GlGraph::draw();
  //save the drawing of the graph in order to prevent redrawing during
  //Interactor draw
  if (_glAuxBufferAvailable) {
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_STENCIL_TEST);
    glDisable(GL_BLEND);
    glDisable(GL_LIGHTING);
    glReadBuffer(GL_BACK);
    glDrawBuffer(GL_AUX0);
    glClear(GL_COLOR_BUFFER_BIT);
    setRasterPosition(0,0);
    glCopyPixels(0,0,width(), height(), GL_COLOR);
    glFlush();
    glDrawBuffer(GL_BACK);
  }
  glTest(__PRETTY_FUNCTION__);  
  //draw the AugmentedDisplay over the graph
  _composite->draw(this);
  //draw the interactors over the graph and the augmented display
  drawInteractors();
  swapBuffers();
  emit graphRedrawn(this);
}
//==================================================
void GlGraphWidget::stopDrawing() {
  cerr << __PRETTY_FUNCTION__ << " :: Not implemented" << endl;
  //  _drawTimer->stop();
}
//==================================================
void GlGraphWidget::waitDrawing() {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << " :: Not implemented" << endl;
#endif
}
//==================================================
void GlGraphWidget::drawInteractors() {
  makeCurrent();
  for(vector<GWInteractor *>::iterator it =
	_interactors.begin(); it != _interactors.end(); ++it) {
    if ((*it)->draw(this))
      break;
  }
}
//==================================================
Iterator<GWInteractor *> *GlGraphWidget::getInteractors() const {
  return new StlIterator<GWInteractor *, vector<GWInteractor *>::const_iterator>(_interactors.begin(), _interactors.end());
}
//==================================================
GWInteractor::ID GlGraphWidget::pushInteractor(GWInteractor* interactor) {
  if (interactor) {
    interactor = interactor->clone();
    interactor->setID(++_id);
    _interactors.push_back(interactor);
    installEventFilter(interactor);
    updateGL();
  }
  return _id;
}
//==================================================
void GlGraphWidget::popInteractor() {
  if (_interactors.size()) {
    GWInteractor *interactor = _interactors[_interactors.size() - 1];
    _interactors.pop_back();
    removeEventFilter(interactor);
    delete interactor;
  }
}
//==================================================
void GlGraphWidget::removeInteractor(GWInteractor::ID i) {
  for(vector<GWInteractor *>::iterator it =
	_interactors.begin(); it != _interactors.end(); ++it) {
    if ((*it)->getID() == i) {
      removeEventFilter(*it);
      delete *it;
      _interactors.erase(it);
      break;
    }
  }
}
//==================================================
GWInteractor::ID GlGraphWidget::resetInteractors(GWInteractor *interactor) {
  for(vector<GWInteractor *>::iterator it =
	_interactors.begin(); it != _interactors.end(); ++it) {
    removeEventFilter(*it);
    delete *it;
  }
  _interactors.clear();
  return pushInteractor(interactor);
}
//==================================================
std::vector<tlp::GWInteractor::ID> GlGraphWidget::resetInteractors(const std::vector<GWInteractor *> &new_interactors) {
  for(vector<GWInteractor *>::iterator it =
	_interactors.begin(); it != _interactors.end(); ++it) {
    removeEventFilter(*it);
    delete *it;
  }
  _interactors.clear();
  vector<GWInteractor::ID> ids;
  for (vector<GWInteractor *>::const_iterator it =
       new_interactors.begin(); it != new_interactors.end(); ++it)
    ids.push_back(pushInteractor(*it));
  return ids;
}
//==================================================
//QGLWidget
//==================================================
QImage GlGraphWidget::grabFrameBuffer(bool withAlpha) {
  waitDrawing();
  QImage img = QGLWidget::grabFrameBuffer(withAlpha);
  return img;
}
//==================================================
//QGLWidget slots
//==================================================
void GlGraphWidget::resizeGL(int w, int h) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << endl;
#endif
  if (w == 0 || h == 0) {
    cerr << "warning: GlGraphWidget::resizeGL(" << w << ", " << h << ")" <<  endl;
    return ;
  }
  GlGraphRenderingParameters param = getRenderingParameters();
  param.setViewport(0,0,w,h);
  setRenderingParameters(param);
}
//==================================================
const stdext::hash_map<std::string, GlAugmentedDisplay*> * GlGraphWidget::getAugmentedDisplays() {
  if (_composite != 0) 
    return _composite->getDisplays();
  else 
    return 0;
}
//==================================================
void GlGraphWidget::addGlAugmentedDisplay(GlAugmentedDisplay *augmentedDisplay, const std::string &name) {
  _composite->addGlAugmentedDisplay(augmentedDisplay, name);
}
//==================================================
void GlGraphWidget::removeGlAugmentedDisplay(GlAugmentedDisplay *augmentedDisplay) {
  _composite->deleteGlAugmentedDisplay(augmentedDisplay);
}
//==================================================
void GlGraphWidget::removeGlAugmentedDisplay(const std::string &name) {
  _composite->deleteGlAugmentedDisplay(name);
}
//==================================================
GlAugmentedDisplay* GlGraphWidget::findGlAugmentedDisplay(const std::string &name) {
  return _composite->findGlAugmentedDisplay(name);
}
//==================================================
bool GlGraphWidget::doSelectAugmentedDisplay (const int x, const int y, 
					      const int width, const int height,
					      std::vector <std::pair <string, tlp::GlAugmentedDisplay *> >
					      &pickedAugmentedDisplays) {
  if (_composite != 0)
    return _composite->doSelect(x, y, 
				width, height, 
				pickedAugmentedDisplays, this);
  else 
    return false;
}
//==================================================
bool GlGraphWidget::doSelectAugmentedDisplay (const int x, const int y,
					      std::vector <std::pair <string, 
					      tlp::GlAugmentedDisplay *> >
					      &pickedAugmentedDisplays) {
  if (_composite != 0)
    return _composite->doSelect(x-3, y-3, 
				6, 6, 
				pickedAugmentedDisplays, this);
  else 
    return false;
}
//==================================================
void GlGraphWidget::doSelect(const int x, const int y, 
			     const int width ,const int height,
			     vector<node> &sNode, vector<edge> &sEdge) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << " x:" << x << ", y:" <<y <<", wi:"<<width<<", height:" << height << endl;
#endif
  makeCurrent();
  GlGraph::doSelect(x, y, width, height, sNode, sEdge);
}
//=====================================================
bool GlGraphWidget::doSelect(const int x, const int y, tlp::ElementType &type ,node &n, edge &e ) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << endl;
#endif
  makeCurrent();
  return GlGraph::doSelect(x, y, type, n, e);
}
//=====================================================
unsigned char * GlGraphWidget::getImage() {
  makeCurrent();
  return GlGraph::getImage();
}
//=====================================================
bool GlGraphWidget::outputEPS(int size, int doSort, const char *filename) {
  makeCurrent();
  return GlGraph::outputEPS(size, doSort, filename);
}
//=====================================================
bool GlGraphWidget::outputSVG(int size, const char* filename) {
  makeCurrent();
  return GlGraph::outputSVG(size, filename);
}
