//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <GL/glu.h>

#include <tulip/BooleanProperty.h>
#include <tulip/StringProperty.h>
#include <tulip/DoubleProperty.h>
#include <tulip/GraphProperty.h>
#include <tulip/IntegerProperty.h>
#include <tulip/LayoutProperty.h>
#include <tulip/SizeProperty.h>
#include <tulip/ColorProperty.h>
#include "tulip/GlRenderer.h"
#include "tulip/GlLines.h"
#include "tulip/GlGraph.h"
#include "tulip/Glyph.h"
#include "tulip/TLPPixmapGlyph.h"
#include "tulip/TLPPixmapFont.h"
#include "tulip/TextRenderer.h"
#include "tulip/DrawingTools.h"
#include "tulip/Curves.h"
#include "tulip/GlTools.h"
#include "tulip/OcclusionTest.h"

using namespace std;
using namespace tlp;

//====================================================
static const unsigned int TLPMAXDEPTH=10;
static const Color COLORSELECT= Color(255, 102, 255, 255);

namespace {
  void drawPoint(const Color &color, const Coord &pos, float size) {
    setColor(color);
    glPointSize(size);
    glBegin(GL_POINTS);
    glVertex3f(pos[0], pos[1], pos[2]);
    glEnd();
  }
}
//====================================================
void GlGraph::drawNode(node n, unsigned int depth) {
  if (depth>TLPMAXDEPTH) return;

  const Coord &nodeCoord = elementLayout->getNodeValue(n);
  const Size &nodeSize = elementSize->getNodeValue(n);
  float lod = projectSize(nodeCoord, nodeSize, projectionMatrix, modelviewMatrix, _renderingParameters.getViewport());
  if (lod < 0) return; //node is not visible
  glPassThrough(n.id); //id of the node for the feed back mode
  
  if (lod < 10.0) { //less than four pixel on screen, we use points instead of glyphs
    if (lod < 1) lod = 1;
    glDisable(GL_LIGHTING);
    const Color &nodeColor = elementColor->getNodeValue(n);
    if (!elementSelected->getNodeValue(n)) {
      drawPoint(nodeColor, nodeCoord, sqrt(lod));
    }
    else {
      glStencilFunc(GL_ALWAYS, 1, 0xFFFF);
      drawPoint(COLORSELECT, nodeCoord, sqrt(lod) + 1);
      glStencilFunc(GL_LEQUAL, 2, 0xFFFF);
    }
    glEnable(GL_LIGHTING);
  }
  else { //draw a glyph or make recursive call for meta nodes
    this->desactivateTexture();
    glPushMatrix();
    glTranslatef(nodeCoord[0], nodeCoord[1], nodeCoord[2]);
    glRotatef(elementRotation->getNodeValue(n), 0., 0., 1.);
    glScalef(nodeSize[0], nodeSize[1], nodeSize[2]);
    
    if (elementGraph->getNodeValue(n) == 0) {
      glyphs.get(elementShape->getNodeValue(n))->draw(n);
    }
    else {
      glStencilFunc(GL_LEQUAL, 3, 0xFFFF);
      glDisable(GL_DEPTH_TEST);
      glyphs.get(elementShape->getNodeValue(n))->draw(n);
      glEnable(GL_DEPTH_TEST);
      glStencilFunc(GL_LEQUAL, 2, 0xFFFF);
      drawMetaNode(n,depth);
    }
    
    if (elementSelected->getNodeValue(n)) {
      glStencilFunc(GL_ALWAYS, 1, 0xFFFF);
      assert(glIsList(selectionDL));
      glCallList(selectionDL);
      glStencilFunc(GL_LEQUAL, 2, 0xFFFF);
    }
    glPopMatrix();
  }
}
//====================================================
unsigned int GlGraph::drawNodes(unsigned int number, Iterator<node> *itN, unsigned int depth) {
  if (!itN->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp = number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glMatrixMode(GL_MODELVIEW); 
  glEnable(GL_LIGHTING);
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_LINE_SMOOTH);
  glDisable(GL_COLOR_MATERIAL);
  glEnable(GL_STENCIL_TEST);
  glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);
  glStencilFunc(GL_LEQUAL, 2, 0xFFFF);
  initProxies();
  glPassThrough(-1.0);	//mark beginning of nodes drawing for feedback mode
  while ((itN->hasNext()) && (number >0)) {
    --number;
    drawNode(itN->next(), depth);
  }
  glPassThrough(-1.0);  //mark end of nodes drawing
  glPopAttrib();
  GLenum error = glGetError();
  if ( error != GL_NO_ERROR)
    cerr << "[OpenGL Error] => " << gluErrorString(error) << endl << "\tin : " << __PRETTY_FUNCTION__ << endl;
  return tmp-number;
}
//====================================================
void GlGraph::drawEdge(edge ite) {
  //cerr << this << "::" << __PRETTY_FUNCTION__ << endl;
  const node source = _graph->source(ite);
  const node target = _graph->target(ite);
  Coord srcCoord = elementLayout->getNodeValue(source);
  Coord tgtCoord = elementLayout->getNodeValue(target);
  const LineType::RealType &bends = elementLayout->getEdgeValue(ite);
  unsigned nbBends = bends.size();
  
  if (nbBends==0 && (source==target)) { //a loop without bends
    //draw a nice loop;
    //TODO !!
    return;
  }
  //  if (bends.size()==0 && (srcCoord - tgtCoord).norm() < 1E-4) 
  //    return; //two nodes very closed
  if (nbBends == 0 && 
      (segmentVisible(srcCoord, tgtCoord, transformMatrix, _renderingParameters.getViewport()) < 15.)) {
    //cerr << ".";
    return;
  }
  //take source and target information for edge clipping
  const Size &srcSize  = elementSize->getNodeValue(source);
  const Size &tgtSize  = elementSize->getNodeValue(target);
  double srcRot = elementRotation->getNodeValue(source);
  double tgtRot = elementRotation->getNodeValue(target);

  //if first bend is inside the glyph (srcCoord is an anchor point it must be  replaced the node position)
  //if last bend is in the glyph (Coord is an anchor point it should replace the node position)
  //Be carefull if there is only one node.
  //  if (bends.size() > 0) {
  //    Coord firstBend = bends.front();
  //    Coord lastBend  = bends.back();
  //  }
  
  // set srcAnchor, tgtAnchor. tmpAnchor will be on the point just before tgtAnchor
  Coord srcAnchor, tgtAnchor, endLineAnchor, tmpAnchor;

  //compute anchor, (clip line with the glyph)
  float lod;
  lod = projectSize(srcCoord, srcSize, projectionMatrix, modelviewMatrix, _renderingParameters.getViewport()); 
  if (lod > 10) { //clip edges with the glyph
    //    int srcGlyphId = 1; //cube outlined
    //    if (elementGraph->getNodeValue(source)==0)
    int srcGlyphId = elementShape->getNodeValue(source);
    Glyph *sourceGlyph = glyphs.get(srcGlyphId);
    tmpAnchor = (nbBends > 0) ? bends.front() : tgtCoord;
    srcAnchor = sourceGlyph->getAnchor(srcCoord, tmpAnchor, srcSize, srcRot);
  } else {
    srcAnchor = srcCoord;
  }

  //compute anchor, (clip line with the glyph)
  lod = projectSize(tgtCoord, tgtSize, projectionMatrix, modelviewMatrix, _renderingParameters.getViewport()); 
  if (lod > 10) { //clip edges with the glyph
    int tgtGlyphId = 1; //cube outlined
    if (elementGraph->getNodeValue(target)==0)
      tgtGlyphId = elementShape->getNodeValue(target);
    Glyph *targetGlyph = glyphs.get(tgtGlyphId);
    //this time we don't take srcCoord but srcAnchor to be oriented to where the line comes from
    tmpAnchor = (nbBends > 0) ? bends.back() : srcAnchor;
    tgtAnchor = targetGlyph->getAnchor(tgtCoord, tmpAnchor, tgtSize, tgtRot);
  } 
  else {
    tgtAnchor = tgtCoord;
  }

  Size edgeSize; //the edge radius and arrow radius
  if (_renderingParameters.isEdgeSizeInterpolate()) {
    edgeSize[0] = std::min(srcSize[0], srcSize[1]) / 16.;
    edgeSize[1] = std::min(tgtSize[0], tgtSize[1]) / 16.;
    if (_renderingParameters.isViewArrow()) {
      edgeSize[2] = std::min(tgtSize[0], tgtSize[1]) / 4.;
    }
  }
  else {
    edgeSize = elementSize->getEdgeValue(ite);
  }

  Color srcCol,tgtCol;
  bool selected = elementSelected->getEdgeValue(ite);
  if (selected) {
    srcCol = COLORSELECT;
    tgtCol = COLORSELECT;
    edgeSize[0] += 0.05;
    edgeSize[1] += 0.05;
  } else {
    if (_renderingParameters.isEdgeColorInterpolate()) {
      srcCol = elementColor->getNodeValue(source);
      tgtCol = elementColor->getNodeValue(target);
    }
    else {
      srcCol = tgtCol = elementColor->getEdgeValue(ite);
    }
  }

  //draw Arrow
  if (_renderingParameters.isViewArrow()) {
    float sizeT = edgeSize.getD();
    sizeT = std::min(sizeT, (float)(tmpAnchor- tgtAnchor).norm()/2.0f);
    MatrixGL matrix(makeArrowMatrix(tmpAnchor, tgtAnchor));
    glEnable(GL_LIGHTING);
    glPushMatrix();
    glMultMatrixf((GLfloat *)&matrix);
    glScalef(sizeT, sizeT, sizeT);
    setMaterial(tgtCol);
    assert(glIsList(arrowDL));
    glCallList(arrowDL);
    glPopMatrix();
    if (selected) {
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glPushMatrix();
      glMultMatrixf((GLfloat *)&matrix);
      glScalef(sizeT+0.1, sizeT+0.1, sizeT+0.1);
      setMaterial(COLORSELECT);
      assert(glIsList(arrowDL));
      glCallList(arrowDL);
      glPopMatrix();
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    }
    endLineAnchor = tmpAnchor - tgtAnchor;
    float nrm = endLineAnchor.norm();
    if( nrm > 0.00000001f ) {
      endLineAnchor *= sizeT / nrm;
      endLineAnchor += tgtAnchor;
    } else {
      endLineAnchor = tgtAnchor;
    }
    tgtCoord = tgtAnchor; //this defines in drawEdge the arrow head as being the final node
    glDisable(GL_LIGHTING);
  }
  else {
    endLineAnchor = tgtAnchor;
  }
  //  cerr << "DRAW EDGE" << endl;
  //draw Edge
  drawEdge(srcCoord, tgtCoord, srcAnchor, endLineAnchor, bends, srcCol, tgtCol,
	   edgeSize, elementShape->getEdgeValue(ite));
  //  cerr << this << "::" << "[END]" << endl;
}
//====================================================
unsigned int GlGraph::drawEdges(unsigned int number,Iterator<edge> *itE, unsigned int depth) {
  if (!itE->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp = number;
  glPushAttrib(GL_ALL_ATTRIB_BITS); 
  glDisable(GL_LIGHTING);
  glEnable(GL_STENCIL_TEST);
  glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
  glStencilFunc(GL_EQUAL, 3, 0xFFFF);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  glDisable(GL_CULL_FACE);
  //  glEnable(GL_POLYGON_SMOOTH);
  initProxies();
  glGetFloatv (GL_PROJECTION_MATRIX, (GLfloat *)&projectionMatrix);
  glGetFloatv (GL_MODELVIEW_MATRIX,  (GLfloat *)&modelviewMatrix);
  transformMatrix = modelviewMatrix * projectionMatrix;
  while ((itE->hasNext()) && (number>0)) {
    number--;
    drawEdge(itE->next());
  }
  glPopAttrib();
  GLenum error = glGetError();
  if ( error != GL_NO_ERROR)
    cerr << "[OpenGL Error] => " << gluErrorString(error) << endl << "\tin : " << __PRETTY_FUNCTION__ << endl;
  return tmp-number;
}
//====================================================

//=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*=*
#define L3D_BIT (1<<9)
void GlGraph::drawEdge(const Coord &srcNodePos, const Coord &tgtNodePos,
                       const Coord &startPoint, const Coord &endPoint, const LineType::RealType &bends,
                       const Color &startColor, const Color &endColor, const Size &size, int shape) {
  bool drawLine = true;
  bool drawPoly = false;
  tlp::curveVisibility(startPoint, bends, endPoint, size, drawPoly, drawLine,
		       projectionMatrix, modelviewMatrix, _renderingParameters.getViewport());
  if (!drawLine && !drawPoly) {
    return;
  }

  //================================
  bool drawEdge3D = _renderingParameters.isEdge3D();
  if (drawEdge3D)
    shape |= L3D_BIT;
  if (shape & L3D_BIT) {
    if (shape < (L3D_BIT+16)) {
      glEnable(GL_LIGHTING);
      glEnable(GL_COLOR_MATERIAL);
      //      if (selected) glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    }
    else shape  = shape & ~L3D_BIT; //no 3D model defined, falling back to Line model
  }
  Coord srcDir(srcNodePos);
  Coord tgtDir(tgtNodePos);
  vector<Coord> tmp =
    tlp::computeCleanVertices(bends, startPoint, endPoint, srcDir, tgtDir);  

  if (tmp.size()<2) {
    return;
  }

  switch (shape) {
  case POLYLINESHAPE:
    if (drawPoly) {
      tlp::polyQuad(tmp, startColor, endColor, size[0], size[1], srcDir, tgtDir, this);
    }
    if (drawLine) {
      tlp::polyLine(tmp, startColor, endColor);
    }
    break;
  case BEZIERSHAPE:
    if (drawPoly)
      tlp::bezierQuad(tmp, startColor, endColor, size[0], size[1], srcDir, tgtDir, this);
    if (drawLine) 
      tlp::bezierLine(tmp, startColor, endColor);
    break;
  case SPLINESHAPE:
    if (drawPoly)
      tlp::splineQuad(tmp, startColor, endColor, size[0], size[1], srcDir, tgtDir, this);
    if (drawLine) 
      tlp::splineLine(tmp, startColor, endColor);
    break;
    //3D lines
  case L3D_BIT + POLYLINESHAPE: 
    GlLines::glDrawExtrusion(srcDir, tgtDir, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
			     GlLines::LINEAR, startColor, endColor); 
    if (drawLine) tlp::polyLine(tmp, startColor, endColor);
    break;
  case L3D_BIT + BEZIERSHAPE:
    GlLines::glDrawExtrusion(srcDir, tgtDir, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
			     GlLines::BEZIER, startColor, endColor); break;
  case L3D_BIT + SPLINESHAPE:
    GlLines::glDrawExtrusion(srcDir, tgtDir, startPoint, bends, endPoint, 10, size, GlLines::TLP_PLAIN,
			     GlLines::SPLINE3, startColor, endColor); break;
  default:
    if (drawPoly)
      tlp::polyQuad(tmp, startColor, endColor, size[0], size[1], srcDir, tgtDir, this);
    if (drawLine) 
      tlp::polyLine(tmp,startColor,endColor);
    break;
  }
  
  if ((shape & L3D_BIT) && (shape < (L3D_BIT+16))) {
    //    if (selected) glPolygonMode(GL_FRONT, GL_FILL);
    glDisable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);
  }
}
//====================================================
//====================================================
// Drawing of labels
// .................
//====================================================
unsigned int GlGraph::drawEdgeLabels(unsigned int number, Iterator<edge> *itE, bool mode, unsigned int depth) {
  if (!itE->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp = number;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);
  glDepthFunc(GL_ALWAYS );
  glDisable(GL_COLOR_MATERIAL);
  initProxies();
  if(mode)
    fontRenderer->setContext(_renderingParameters.getFontsPath() + "font.ttf", 20, 0, 0, 255);
  else	
    fontRenderer->setContext(_renderingParameters.getFontsPath() + "font.ttf", 18, 255, 255, 255);
  while ((itE->hasNext()) && (number >0)) {
    edge ite=itE->next();
    if (elementSelected->getEdgeValue(ite)==mode){
      number--;
      drawEdgeLabel(ite, mode);
    }
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================
unsigned int GlGraph::drawNodeLabels(unsigned int number, Iterator<node> *itN, bool mode, unsigned int depth) {
  if (!itN->hasNext() || number==0) return 0;
  if (depth>TLPMAXDEPTH) return 0;
  unsigned int tmp = number;
  //  setupOpenGlContext();
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_LIGHTING);
  glDepthFunc(GL_ALWAYS );
  glDisable(GL_CULL_FACE);
  glDisable(GL_COLOR_MATERIAL);
  initProxies();
  if(mode)
    fontRenderer->setContext(_renderingParameters.getFontsPath() + "font.ttf", 20, 0, 0, 255);
  else	
    fontRenderer->setContext(_renderingParameters.getFontsPath() + "font.ttf", 18, 255, 255, 255);
  while ((itN->hasNext()) && (number >0)) {
    node itv = itN->next();
    --number;
    drawNodeLabel(itv, mode, depth);
  }
  glPopAttrib();
  return tmp-number;
}
//====================================================
void GlGraph::drawPixmapFont(const string &str, const Color &col,  const Coord &position,
			     int labelPos, bool selected, float width) {
  int rastPos[4];
  float w_max = 300;
  float w,h;
  unsigned int labelsBorder = _renderingParameters.getLabelsBorder();
  //need to be done before glRasterPos3f to set GL_RASTER_COLOR correctly  
  //Gl_RASTER_COLOR is used by FTGL in BITMAP and PIXMAP mode instead of GL_COLOR
  glColor4ub(col[0], col[1], col[2], 255);

  glRasterPos3f(position[0], position[1], position[2]);

  glGetIntegerv(GL_CURRENT_RASTER_POSITION, rastPos);
  if(occlusionTest->testRectangle(RectangleInt2D(rastPos[0] - labelsBorder - 5,
						 rastPos[1]  - labelsBorder - 5,
						 rastPos[0]  + labelsBorder + 5,
						 rastPos[1]  + labelsBorder + 5)))
    return;

  fontRenderer->setMode(TLP_PIXMAP);
  fontRenderer->setString(str, VERBATIM);
  //fontRenderer->setString(str, XML);

  fontRenderer->setColor(col[0], col[1], col[2]);
  //  w_max = width;
  fontRenderer->getBoundingBox(w_max, h, w);

  if(!occlusionTest->addRectangle(RectangleInt2D(rastPos[0]-(int)(w/2.0) - labelsBorder,
						 rastPos[1]-(int)(h/2.0) - labelsBorder,
						 rastPos[0]+(int)(w/2.0) + labelsBorder,
						 rastPos[1]+(int)(h/2.0) + labelsBorder))) {
    fontRenderer->draw(w, w, labelPos);
  }
}

//====================================================
void GlGraph::drawEdgeLabel(edge e, bool mode) {
  string tmp=elementLabel->getEdgeValue(e);
  if (tmp.length()==0) return;
  const Coord & srcCoord = elementLayout->getNodeValue(_graph->source(e));
  const Coord & tgtCoord = elementLayout->getNodeValue(_graph->target(e));
  const LineType::RealType &bends = elementLayout->getEdgeValue(e);
  Coord position;
  if (bends.empty()) {
    position = (srcCoord+tgtCoord) / 2.0;
  }
  else {
    if (bends.size()%2 == 0)
      position = (bends[bends.size()/2-1]+bends[bends.size()/2]) / 2.0;
    else
      position = bends[bends.size()/2];
  }

  Color fontColor;
  if (elementSelected->getEdgeValue(e))
    fontColor.set(255,0,0,255);
  else {
    fontColor = elementLabelColor->getEdgeValue(e);
  }
  drawPixmapFont(tmp, fontColor, position, ON_CENTER, elementSelected->getEdgeValue(e), 50);
}
//====================================================
void GlGraph::drawNodeLabel(node n, bool mode, unsigned int depth) {
  const Coord &nodeCoord = elementLayout->getNodeValue(n);
  const Size  &nodeSize  = elementSize->getNodeValue(n);
  int labelPos = elementLabelPosition->getNodeValue(n);
  Coord nodePos(nodeCoord);
  switch (labelPos) {
    case ON_TOP:
      nodePos.setY(nodeCoord.getY() + nodeSize.getH()/2);
      break;
    case ON_BOTTOM:
      nodePos.setY(nodeCoord.getY() - nodeSize.getH()/2);
      break;
    case ON_LEFT:
      nodePos.setX(nodeCoord.getX() - nodeSize.getW()/2);
      break;
    case ON_RIGHT:
      nodePos.setX(nodeCoord.getX() + nodeSize.getW()/2);
    default:
      break;
    }
  //Recursive call for node inside meta-node
  if (_renderingParameters.isViewMetaLabel() && elementGraph->getNodeValue(n)!=0) {
    glPushMatrix();
    glTranslatef(nodePos[0], nodePos[1], nodePos[2]);
    glRotatef(elementRotation->getNodeValue(n), 0., 0., 1.);
    glScalef(nodeSize[0], nodeSize[1], nodeSize[2]);
    drawMetaNode(n, depth, true, mode);
    glPopMatrix();
  }
  if (elementSelected->getNodeValue(n) != mode) return; 

  const string &tmp = elementLabel->getNodeValue(n);
  if (tmp.length() < 1) {
    return;
  }

  Color fontColor = elementLabelColor->getNodeValue(n);

  if (mode)
    fontColor = COLORSELECT;

  float w_max = 300;
  float w,h;
  float div_w, div_h;
  switch(_renderingParameters.getFontsType()){
  case 0:
    if (projectSize(nodeCoord, nodeSize, projectionMatrix, modelviewMatrix, _renderingParameters.getViewport()) < 8.0) return;
    fontRenderer->setMode(TLP_POLYGON);
    fontRenderer->setColor(fontColor[0], fontColor[1], fontColor[2]);
    fontRenderer->setString(tmp, VERBATIM);
    //      w_max = nodeSize.getW()*50.0;
    fontRenderer->getBoundingBox(w_max, h, w);
    glPushMatrix();
    glTranslatef(nodePos[0], nodePos[1], nodePos[2]);
    glRotatef(elementRotation->getNodeValue(n), 0., 0., 1.);
    div_w = nodeSize.getW()/w;
    div_h = nodeSize.getH()/h;
    if(div_h > div_w) 
      glScalef(div_w, div_w, 1);
    else
      glScalef(div_h, div_h, 1);  
    fontRenderer->draw(w,w, labelPos);
    glPopMatrix();
    break;
  case 1:
    drawPixmapFont(tmp, fontColor, nodePos, labelPos, elementSelected->getNodeValue(n), nodeSize.getW());
    break;
  case 2:
    if (projectSize(nodeCoord, nodeSize, projectionMatrix, modelviewMatrix, _renderingParameters.getViewport()) < 8.0) return;

    fontRenderer->setMode(TLP_TEXTURE);
    fontRenderer->setColor(fontColor[0], fontColor[1], fontColor[2]);
    fontRenderer->setString(tmp, VERBATIM);

    //      w_max = nodeSize.getW();
    fontRenderer->getBoundingBox(w_max, h, w);
    glPushMatrix();
    glTranslatef(nodePos[0], nodePos[1], nodePos[2]);
    glRotatef(elementRotation->getNodeValue(n), 0., 0., 1.);
    div_w = nodeSize.getW()/w;
    div_h = nodeSize.getH()/h;
    if(div_h > div_w) 
      glScalef(div_w, div_w, 1);
    else 
      glScalef(div_h, div_h, 1);
    glEnable( GL_TEXTURE_2D);
    glBlendFunc(GL_ONE_MINUS_DST_COLOR,GL_ONE_MINUS_SRC_COLOR);
    fontRenderer->draw(w, w, labelPos);
    glDisable( GL_TEXTURE_2D);
    glPopMatrix();
    break;
  default:
    cerr << "GlGraph::DrawNodes unknown fonts" << endl;
    break;
  }
}
//====================================================
void GlGraph::drawMetaNode(node n,unsigned int depth, bool labels, bool selectedLabels) {
  Graph *sp = _graph;
  _graph = elementGraph->getNodeValue(n);
  string layoutNameBackup(_renderingParameters.getInputLayout());
  _renderingParameters.setInputLayout("viewLayout");
  initProxies();
  pair<Coord, Coord> bboxes = tlp::computeBoundingBox(_graph, elementLayout, elementSize, elementRotation);
  //  cerr << bboxes.first << "/" << bboxes.second << endl;
  Coord maxC = bboxes.first;
  Coord minC = bboxes.second;
  MatrixGL saveMatrix(modelviewMatrix);
  Coord translate= (maxC + minC)/ (-2.0);
  double dept  = maxC[2] - minC[2];
  double width  = maxC[0] - minC[0];
  double height = maxC[1] - minC[1];
  if (width<0.0001) width=1;
  if (height<0.0001) height=1;
  if (dept<0.0001) dept=1;
  glPushMatrix();
  glScalef(1.0/width, 1.0/height, 1.0/dept);
  glTranslatef(translate[0],translate[1],translate[2]);
  glGetFloatv (GL_MODELVIEW_MATRIX, (GLfloat *)&modelviewMatrix);
  transformMatrix = modelviewMatrix * projectionMatrix;

  Iterator<node> *itN=_graph->getNodes();
  if (!labels)
    drawNodes(_graph->numberOfNodes(), itN , depth+1);
  else
    drawNodeLabels(_graph->numberOfNodes(), itN , selectedLabels, depth+1);
  delete itN;
  if (_renderingParameters.isDisplayEdges()) {
    Iterator<edge> *itE=_graph->getEdges();
    if (!labels)
      drawEdges(_graph->numberOfEdges(), itE, depth+1);
    else if (_renderingParameters.isViewEdgeLabel())
      drawEdgeLabels(_graph->numberOfEdges(), itE , selectedLabels, depth+1);
    delete itE;
  }
  glPopMatrix();
  _graph = sp;
  _renderingParameters.setInputLayout(layoutNameBackup);
  initProxies();
  modelviewMatrix = saveMatrix;
  transformMatrix = modelviewMatrix * projectionMatrix;
}
