//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 01/05/2002
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/

#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif
#include <iostream>
#include <time.h>

#include <tulip/TulipPlugin.h>
#include <tulip/Coord.h>
#include <tulip/PropertyProxy.h>

#define TLP_FILE_VERSION "2.0"

using namespace std;


static string convert(string tmp) {
  string newStr;
  for (unsigned int i=0;i<tmp.length();i++) {
    if (tmp[i]=='\"')
      newStr+="\\\"";
    else
      if (tmp[i]=='\n')
	newStr+="\\n";
      else
	newStr+=tmp[i];
  }
  return newStr;
}

static const char* boolTN, *colorTN, *coordTN, *doubleTN, *floatTN, *intTN, *stringTN, *uintTN;
static bool typesInited = false;

namespace {
  const char * paramHelp[] = {
    // params
    HTML_HELP_OPEN() \
    HTML_HELP_DEF( "type", "string" ) \
    HTML_HELP_DEF( "default", "" ) \
    HTML_HELP_BODY() \
    "Indicates the author of this graph." \
    HTML_HELP_CLOSE(),
    HTML_HELP_OPEN() \
    HTML_HELP_DEF( "type", "string" ) \
    HTML_HELP_DEF( "default", "This file was generated by Tulip." ) \
    HTML_HELP_BODY() \
    "adds some comments." \
    HTML_HELP_CLOSE(),
  };
}
/** \addtogroup export */
/*@{*/
struct TLP:public ExportModule {

  DataSet displaying;
  
  TLP(ClusterContext context):ExportModule(context) {
    /* Tulip 3 only
    addParameter<string>("author", paramHelp[0]);
    addParameter<string>("text::comments", paramHelp[1], "This file was generated by Tulip."); */
    addParameter<DataSet>("displaying");
  }
    //=====================================================
  ~TLP(){}
    //=====================================================
  void saveGraphElements(ostream &os,SuperGraph *graph) {
    if (graph->getFather() != graph) {
      os << "(cluster " << graph->getId() << " \"" << graph->getAttribute<string>("name") << "\"" << endl;
      Iterator<node> *itN = graph->getNodes();
      if (itN->hasNext()) {
	os << "(nodes ";
	while (itN->hasNext()) {
	  os << itN->next().id;
	  if (itN->hasNext()) os << " ";
	}
	os << ")" << endl;
      } delete itN;
      Iterator<edge> *itE = graph->getEdges();
      if (itE->hasNext()) {
	os << "(edges ";
	while (itE->hasNext()) {
	  os << itE->next().id;
	  if (itE->hasNext()) os << " ";
	}
	os << ")" << endl;
      } delete itE;
    }
    else
      os << graph << endl;
    Iterator<SuperGraph *> *itS = graph->getSubGraphs();
    while (itS->hasNext())
      saveGraphElements(os,itS->next());
    delete itS;
    if (graph->getFather() != graph)  os << ")" << endl;
  }
  //=====================================================
  void saveLocalProperties(ostream &os,SuperGraph *graph) {
    Iterator<string> *itS=graph->getLocalProperties();
    PProxy *pproxy;
    while (itS->hasNext()) {
      string its=itS->next();
      pproxy=graph->getProperty(its);
      if (graph->getFather()==graph)
	os << "(property " << " 0 " << pproxy->getTypename() << " " ;
      else
	os << "(property " << " " << graph->getId() << " " << pproxy->getTypename() << " " ;
      os << "\"" << convert(its) << "\"" << endl;
      string nDefault = pproxy->getNodeDefaultStringValue();
      string eDefault = pproxy->getEdgeDefaultStringValue();
      os <<"(default \"" << convert(nDefault) << "\" \"" << convert(eDefault) << "\" )" << endl; 
      Iterator<node> *itN=graph->getNodes();
      while (itN->hasNext()) {
	node itn=itN->next();
	string tmp = pproxy->getNodeStringValue(itn);
	if (strcmp(tmp.c_str(),nDefault.c_str())!=0) os << "(node " << itn.id << " \"" << convert(tmp) << "\")" << endl ;
      } delete itN;
      Iterator<edge> *itE=graph->getEdges();
      while (itE->hasNext()) {
	edge ite=itE->next();
	string tmp = pproxy->getEdgeStringValue(ite);
	if (strcmp(tmp.c_str(),eDefault.c_str())!=0) os << "(edge " << ite.id << " \"" << convert(tmp) << "\")" << endl ;
      } delete itE;
      os << ")" << endl;
    }
    delete itS;
  }
  //=====================================================
  void saveProperties(ostream &os,SuperGraph *graph) {
    saveLocalProperties(os,graph);
    Iterator<SuperGraph *> *itS=graph->getSubGraphs();
    while (itS->hasNext())
      saveProperties(os,itS->next());
    delete itS;
  }
  //=====================================================
  void initTypeNames() {
    if (!typesInited) {
      boolTN = typeid(typesInited).name();
      Color color;
      colorTN = typeid(color).name();
      Coord coord;
      coordTN = typeid(coord).name();
      double d;
      doubleTN = typeid(d).name();
      float f;
      floatTN = typeid(f).name();
      int i;
      intTN = typeid(i).name();
      string s;
      stringTN = typeid(s).name();
      unsigned int ui;
      uintTN = typeid(ui).name();
      typesInited = true;
    }
  }      
  //=====================================================
  void saveDataSet(ostream &os, DataSet &data) {
    initTypeNames();
    // get iterator over pair attribute/value
    Iterator< pair<string, DataType> > *it = data.getValues();
    while( it->hasNext() ) {
      pair<string, DataType> p;
      p = it->next();
      const string tn = p.second.typeName;
      char *otn = 0;
      // get output type name
      if (tn == boolTN)
	otn = "bool";
      else if (tn == colorTN)
	otn = "color";
      else if (tn == coordTN)
	otn = "coord";
      else if (tn == doubleTN)
	otn = "double";
      else if (tn == floatTN)
	otn = "float";
      else if (tn == intTN)
	otn = "int";
      else if (tn == stringTN)
	otn = "string";
      else if (tn == uintTN)
	otn = "uint";
      // general case do nothing 
      else continue;
      // output opened parenthesis and otn
      os << '(' << otn;
      // output attribute name
      os << " \"" << p.first << "\" ";
      // output value
      // special cases
      if (tn == boolTN)
	os << ((*(bool*)p.second.value) ? "true" : "false");
      else if (tn == colorTN) {
	Color *color = (Color*) p.second.value;
	os << "\"(" << (int)color->getR() << ',' << (int)color->getG() << ',' << (int) color->getB() << ",0)\"";
      } else if (tn == coordTN) {
	Coord *coord =  (Coord*) p.second.value;
	os << "\"(" << coord->getX() << "," << coord->getY() << "," << coord->getZ() << ")\"";
      }	else if (tn == doubleTN)
	os << *(double*)p.second.value;
      else if (tn == floatTN)
	os << *(float*)p.second.value;
      else if (tn == intTN)
	os << *(int*)p.second.value;
      else if (tn == stringTN)
	os << '"' << *(string *)p.second.value << '"';
      else if (tn == uintTN)
	os << *(unsigned int *) p.second.value;
      // output closed parenthesis
      os << ')' << endl;
    } delete it;
  }    
  //=====================================================
  void saveAttributes(ostream &os, SuperGraph *graph) {
    DataSet &dataSet = graph->getAttributes();
    os << "(attributes " << endl;
    saveDataSet(os, graph->getAttributes());
    os << ")" << endl;
  }
  //=====================================================
  void saveDisplaying(ostream &os, DataSet &data) {
    os << "(displaying " << endl;
    saveDataSet(os, data);
    os << ")" << endl;
  }
  
  bool exportGraph(ostream &os,SuperGraph *currentGraph) {
    superGraph=currentGraph->getRoot();
    /* Tulip 3 only
    string author;
    string comments; */

    if (dataSet != NULL) {
      dataSet->get<DataSet>("displaying", displaying);
      /* Tulip 3 only
      dataSet->get("author", author);
      dataSet->get("text::comments", comments); */
    }

    /* for Tulip 3 only
    // get ostime
    time_t ostime = time(NULL);
    // get local time
    struct tm *currTime = localtime(&ostime);
    // format date
    char currDate[32];
    strftime(currDate, 32, "%m-%d-%Y", currTime);

    // output tlp format version
    os << "(tlp \"" << TLP_FILE_VERSION << '"' << endl;
    // current date
    os << "(date \"" << currDate << "\")" << endl;
    // author
    if (author.length() > 0)
      os << "(author \"" << author << "\")" << endl;
    // comments
    os << "(comments \"" << comments << "\")" << endl;
    */

    saveGraphElements(os,superGraph);
    saveProperties(os,superGraph);
    saveAttributes(os,superGraph);

    //Save displaying
    //    DataSet displaying;
    if (dataSet != NULL && dataSet->get<DataSet>("displaying", displaying))
      saveDisplaying(os, displaying);
    
    // for Tulip 3 only
    //os << ')' << endl; // end of (tlp ...
    return true;
  }
};
/*@}*/
EXPORTPLUGIN(TLP,"tlp","Auber David","31/07/2001","0","1","0");
