#include <qlabel.h>
#include <qmessagebox.h>

#include <tulip/PropertyManager.h>
#include <tulip/PropertyProxy.h>

#include "tulip/TulipTableWidget.h"
#include "tulip/TulipElementProperties.h"

using namespace tlp;
//==========================================
TulipElementProperties::TulipElementProperties(SuperGraph *supergraph,
                                               const QStringList &nodeListedProperties,
                                               const QStringList &edgeListedProperties,
                                               QWidget *parent, const char *name, WFlags fl) :
  TulipElementPropertiesData(parent, name, fl), 
  displayMode(NODE), 
  nodeSet(false), 
  edgeSet(false),
  supergraph(0)
{
  setSuperGraph(supergraph);
  setNodeListedProperties(nodeListedProperties);
  setEdgeListedProperties(edgeListedProperties);
  propertyTable->horizontalHeader()->setLabel(0, tr("Property"));
  propertyTable->horizontalHeader()->setLabel(1, tr("Value"));
  propertyTable->setColumnStretchable(1, false);
  displayAllProperties=false;
  connect(propertyTable,SIGNAL(valueChanged(int,int)),this,SLOT(propertyTableValueChanged(int,int)));
}
//==========================================
TulipElementProperties::TulipElementProperties(QWidget *parent, const char *name, WFlags fl) :
  TulipElementPropertiesData(parent, name, fl), 
  displayMode(NODE), 
  nodeSet(false), 
  edgeSet(false),
  supergraph(0)
{
  setSuperGraph(0);
  propertyTable->horizontalHeader()->setLabel(0, tr("Property"));
  propertyTable->horizontalHeader()->setLabel(1, tr("Value"));
  displayAllProperties=true;
  connect(propertyTable,SIGNAL(valueChanged(int,int)),this,SLOT(propertyTableValueChanged(int,int)));
}
//==========================================
TulipElementProperties::~TulipElementProperties() {
}
//==========================================
QStringList TulipElementProperties::getCurrentListedProperties() const{
  switch(displayMode) {
  case NODE: return nodeListedProperties;
  case EDGE: return edgeListedProperties;
  default:
    qWarning(QString(__PRETTY_FUNCTION__) + ": current displayMode is invalid!");
    return QStringList();
  }
}
//==========================================
QStringList TulipElementProperties::getNodeListedProperties() const {
  return nodeListedProperties;
}
//==========================================
QStringList TulipElementProperties::getEdgeListedProperties() const {
  return edgeListedProperties;
}
//==========================================
SuperGraph *TulipElementProperties::getSuperGraph() const {
  return supergraph;
}
//==========================================
node TulipElementProperties::getCurrentNode() const {
  return currentNode;
}
//==========================================
edge TulipElementProperties::getCurrentEdge() const {
  return currentEdge;
}
//==========================================
tlp::ElementType TulipElementProperties::getDisplayMode() const {
  return displayMode;
}
//==========================================
bool TulipElementProperties::isElementDisplayed() const {
  switch(displayMode) {
  case NODE: return nodeSet; break;
  case EDGE: return edgeSet; break;
  }
  return true; //never reached, but suppresses a compiler warning
}
//==========================================
void TulipElementProperties::setDisplayMode(const tlp::ElementType mode) {
  switch(mode) {
  case NODE:
  case EDGE:
    nodeSet = false;
    edgeSet = false;
    displayMode = mode;
    updateTable();
    break;
  }
}
//==========================================
void TulipElementProperties::setCurrentNode(SuperGraph *graph, const node &n) {
  if (supergraph!=graph) setSuperGraph(graph);
  displayMode = NODE;
  nodeSet = true;
  currentNode = n;
  this->setCaption(QString("Node %1").arg(n.id));
  label->setText(QString("Node %1").arg(n.id));
  updateTable();
}
//==========================================
void TulipElementProperties::setCurrentEdge(SuperGraph *graph,const edge &e) {
  if (supergraph!=graph) setSuperGraph(graph);
  displayMode = EDGE;
  edgeSet = true;
  currentEdge = e;
  this->setCaption(QString("Edge %1").arg(e.id));
  label->setText(QString("Edge %1").arg(e.id));
  updateTable();
}
//==========================================
void TulipElementProperties::setSuperGraph(SuperGraph *s, bool destroy) {
  if (supergraph!=0 && !destroy)
    supergraph->removeObserver(this);
  supergraph = s;
  nodeSet = false;
  edgeSet = false;
  this->setCaption("");
  label->setText("No element selected");
  for (int i=0;i<propertyTable->numRows();++i) {
    propertyTable->clearCell(i,0);
    propertyTable->clearCell(i,1);
  }
  propertyTable->setNumRows(0);
  if (supergraph!=0)
    supergraph->addObserver(this);
}
//==========================================
void TulipElementProperties::setCurrentListedProperties(const QStringList &l) {
  switch(displayMode) {
  case NODE: return setNodeListedProperties(l);
  case EDGE: return setEdgeListedProperties(l);
  }
}
//==========================================
void TulipElementProperties::setNodeListedProperties(const QStringList &l) {
  nodeListedProperties = l;
  propertyTable->setNumRows(nodeListedProperties.size());
  updateTable();
}
//==========================================
void TulipElementProperties::setEdgeListedProperties(const QStringList &l) {
  edgeListedProperties = l;
  propertyTable->setNumRows(edgeListedProperties.size());
  updateTable();
}
//==========================================
void TulipElementProperties::updateTable() {
  if (supergraph == NULL) return;
  int i=0;
  if (displayAllProperties) {
    propertyTable->setNumRows(0);    
    propertyTable->setNumRows(50);    
    Iterator<std::string> *it=supergraph->getLocalProperties();
    while(it->hasNext()) {
      std::string pname=it->next();
      propertyTable->setText(i, 0, QString(pname.c_str()));
      switch(displayMode) {
      case NODE:
	if (nodeSet) {
	  PProxy *editedProperty = supergraph->getProperty(pname);
	  propertyTable->setTulipNodeItem(editedProperty, pname, currentNode, i, 1);
	  ++i;
	}
	break;
      case EDGE:
	if (edgeSet) {
	  PProxy *editedProperty = supergraph->getProperty(pname);
	  propertyTable->setTulipEdgeItem(editedProperty, pname, currentEdge, i, 1);
	  ++i;
	}
	break;
      }

    }delete it;
    it=supergraph->getInheritedProperties();
    while(it->hasNext()) {
      std::string pname=it->next();
      propertyTable->setText(i, 0, QString(pname.c_str()));
      switch(displayMode) {
      case NODE:
	if (nodeSet) {
	  PProxy *editedProperty = supergraph->getProperty(pname);
	  propertyTable->setTulipNodeItem(editedProperty, pname, currentNode, i, 1);
	  ++i;
	}
	break;
      case EDGE:
	if (edgeSet) {
	  PProxy *editedProperty = supergraph->getProperty(pname);
	  propertyTable->setTulipEdgeItem(editedProperty, pname, currentEdge, i, 1);
	  ++i;
	}
	break;
      }
    }delete it;
    propertyTable->setNumRows(i);
  }
  else {
    QStringList *listedProperties = NULL;
    switch(displayMode) {
    case NODE: listedProperties = &nodeListedProperties; break;
    case EDGE: listedProperties = &edgeListedProperties; break;
    }
    propertyTable->setNumRows(listedProperties->count());
    for (QStringList::const_iterator it=listedProperties->begin();
	 it != listedProperties->end();
	 ++it, ++i) {
      propertyTable->setText(i, 0, *it);
      if (supergraph->existProperty((*it).latin1())) {
        switch(displayMode) {
        case NODE:
          if (nodeSet) {
            PProxy *editedProperty = supergraph->getProperty((*it).latin1());
            propertyTable->setTulipNodeItem(editedProperty, (*it).latin1(), currentNode, i, 1);
          }
          break;
        case EDGE:
          if (edgeSet) {
            PProxy *editedProperty = supergraph->getProperty((*it).latin1());
            propertyTable->setTulipEdgeItem(editedProperty, (*it).latin1(), currentEdge, i, 1);
          }
        }
      }
    }
  }
  propertyTable->adjustColumn(0);
  propertyTable->adjustColumn(1);
}
using namespace std;
//==========================================
void TulipElementProperties::propertyTableValueChanged(int row, int col) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  bool *elementSet;
  QStringList *listedProperties;
  switch(displayMode) {
  case NODE: elementSet = &nodeSet; listedProperties = &nodeListedProperties; break;
  case EDGE: elementSet = &edgeSet; listedProperties = &edgeListedProperties; break;
  }

  QString property = propertyTable->text(row, 0);
  QString value = propertyTable->text(row, col);
  PProxy *editedProperty = supergraph->getProperty(property.latin1());
  if (editedProperty==0) return;
  std::string tmp = value.latin1();
  //  cerr << "Value :" << value << endl;
  bool result=true;
  switch(displayMode) {
  case NODE:
    result=editedProperty->setNodeStringValue( currentNode, tmp);
    break;
  case EDGE:
    result=editedProperty->setEdgeStringValue( currentEdge, tmp);
    break;
  }
  
  if (!result) {
    QMessageBox::critical( 0, "Tulip Property Editor Change Failed",
			   QString("The value entered for this property is not correct,\n"
				   "The change won't be applied\n"
				   "Modify the entered value to apply the changes.")
			   );
  }
  else {
    switch(displayMode) {
    case NODE:
      emit tulipNodePropertyChanged(supergraph, currentNode, property, value);
      break;
    case EDGE:
      emit tulipEdgePropertyChanged(supergraph, currentEdge, property, value);
      break;
    }
  }
  //}
}
//==========================================
void TulipElementProperties::nodePropertyChanged(SuperGraph *s, node const &n, QString const &property, QString const &value) {
  if ((displayMode == NODE) && (supergraph == s) && (currentNode == n))
    updateTable();
}
//==========================================
void TulipElementProperties::edgePropertyChanged(SuperGraph *s, edge const &e, QString const &property, QString const &value) {
  if ((displayMode == EDGE) && (supergraph == s) && (currentEdge == e))
    updateTable();
}
//==========================================
void TulipElementProperties::delNode(SuperGraph *graph, node n) {
  if (graph!=supergraph && supergraph!=0 )
    supergraph->removeObserver(this);
  if (displayMode==NODE && currentNode==n)
    setSuperGraph(graph);
}
void TulipElementProperties::delEdge(SuperGraph *graph, edge e) {
  if (graph!=supergraph && supergraph!=0)
    supergraph->removeObserver(this);
  if (displayMode==EDGE && currentEdge==e)
    setSuperGraph(graph);
}
void TulipElementProperties::destroy(SuperGraph *) {
  setSuperGraph(0, true);
}
