/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 20/08/2001
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "tulip/SuperGraphView.h"
#include "tulip/PropertyManager.h"
#include "tulip/SelectionProxy.h"
#include "tulip/SuperGraph.h"
#include "tulip/GraphIterator.h"
#include "tulip/StableIterator.h"
#include "tulip/SelectionProxy.h"

using namespace std;
//----------------------------------------------------------------
SuperGraphView::SuperGraphView(SuperGraph *father,SelectionProxy *filter):
  SuperGraphAbstract(father),
  _superGraph(father->getRoot()),
  nNodes(0),
  nEdges(0)
{
  adaptativeFilter =new SelectionProxy(this);
  adaptativeFilter->setAllNodeValue(false);
  adaptativeFilter->setAllEdgeValue(false);
  Iterator<node> *iteN=getFather()->getNodes();
  while (iteN->hasNext()) {
    node n=iteN->next();
    if (filter->getNodeValue(n)) addNode(n);
  } delete iteN;
  Iterator<edge> *iteE=getFather()->getEdges();
  while (iteE->hasNext()) {
    edge e=iteE->next();
    if (filter->getEdgeValue(e)) addEdge(e);
  } delete iteE;
}

//----------------------------------------------------------------
SuperGraphView::~SuperGraphView() {
  notifyDestroy(this);
  //  cerr << "SuperGraphView::~SuperGraphView" << endl;
  delete adaptativeFilter;
  //  cerr << "SuperGraphView::~SuperGraphView Ok......" << endl;
}
//----------------------------------------------------------------
bool SuperGraphView::isElement(const node n) 
{return adaptativeFilter->getNodeValue(n);}
//----------------------------------------------------------------
bool SuperGraphView::isElement(const edge e) 
{return adaptativeFilter->getEdgeValue(e);}
//----------------------------------------------------------------
node SuperGraphView::addNode() {
  _superGraph=getFather();
  node tmp=_superGraph->addNode();
  adaptativeFilter->setNodeValue(tmp,true);
  if (nNodes>0) nNodes++;
  notifyAddNode(this, tmp);
  return tmp;
}
//----------------------------------------------------------------
void SuperGraphView::addNode(const node n) {
  if (!isElement(n)) {
    _superGraph=getFather();
    if (!_superGraph->isElement(n)) _superGraph->addNode(n);
    adaptativeFilter->setNodeValue(n,true);
    if (nNodes>0) nNodes++;
    notifyAddNode(this, n);
  }
}
//----------------------------------------------------------------
edge SuperGraphView::addEdge(const node n1,const node n2) {
  assert(isElement(n1));
  assert(isElement(n2));
  _superGraph=getFather();
  edge tmp=_superGraph->addEdge(n1,n2);
  adaptativeFilter->setEdgeValue(tmp,true);
  if (nEdges>0) nEdges++;
  notifyAddEdge(this, tmp);
  return tmp;
}
//----------------------------------------------------------------
void SuperGraphView::addEdge(const edge e) {
  //  assert(!isElement(e));
  if (!adaptativeFilter->getEdgeValue(e)) {
    _superGraph=getFather();
    if (!_superGraph->isElement(e)) _superGraph->addEdge(e);
    adaptativeFilter->setEdgeValue(e,true);
    if (nEdges>0) nEdges++;
    notifyAddEdge(this, e);
  }
}
//----------------------------------------------------------------
void SuperGraphView::delNode(const node n) {
  if (isElement(n)) {
    Iterator<SuperGraph *>*itS=getSubGraphs();
    for (;itS->hasNext();)
      itS->next()->delNode(n);
    delete itS;
    set<edge> tmpSet;
    Iterator<edge> *itE=getInOutEdges(n);
    for (;itE->hasNext();) 
      tmpSet.insert(itE->next());
    delete itE;
    for (set<edge>::const_iterator ite=tmpSet.begin();ite!=tmpSet.end();++ite)
      delEdge(*ite);
    adaptativeFilter->setNodeValue(n,false);
    getPropertyManager()->erase(n);
    if (nNodes>0) nNodes--;
    notifyDelNode(this, n);
  }
}
//----------------------------------------------------------------
void SuperGraphView::delEdge(const edge e) {
  if (isElement(e)) {
    Iterator<SuperGraph *>*itS=getSubGraphs();
    for (;itS->hasNext();)
      itS->next()->delEdge(e);
    delete itS;
    adaptativeFilter->setEdgeValue(e,false);
    getPropertyManager()->erase(e);
    if (nEdges>0) nEdges--;
    notifyDelEdge(this,e);
  }
}
//----------------------------------------------------------------
void SuperGraphView::delAllNode(const node n){
  getFather()->delAllNode(n);
}
//----------------------------------------------------------------
void SuperGraphView::delAllEdge(const edge e){
  getFather()->delAllEdge(e);
}
//----------------------------------------------------------------
void SuperGraphView::setEdgeOrder(const node n,const std::vector<edge> &v ) {
  getFather()->setEdgeOrder(n,v);
}
//----------------------------------------------------------------
void SuperGraphView::swapEdgeOrder(const node n,const edge e1 , const edge e2) {
  getFather()->swapEdgeOrder(n,e1,e2);
}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getNodes() const {
  Iterator<unsigned int> *it=0;
  try {
    it=adaptativeFilter->nodeProperties.findAll(true);
  } catch (ImpossibleOperation &e) {
    it=0;
  }
  if (it==0)
    return (new SGraphNodeIterator(this,adaptativeFilter));
  else
    return (new UINTIterator<node>(it));
}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getInNodes(const node n) const 
{return (new InNodesIterator(this,adaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getOutNodes(const node n)const 
{return (new OutNodesIterator(this,adaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphView::getInOutNodes(const node n)const 
{return (new InOutNodesIterator(this,adaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getEdges()const {
  Iterator<unsigned int> *it=0;
  try {
    it=adaptativeFilter->edgeProperties.findAll(true);
  } catch (ImpossibleOperation &e) {
    it=0;
  }
  if (it==0)
    return (new SGraphEdgeIterator(this,adaptativeFilter));
  else
    return (new UINTIterator<edge>(it));
}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getInEdges(const node n)const 
{return (new InEdgesIterator(this,adaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getOutEdges(const node n)const 
{return (new OutEdgesIterator(this,adaptativeFilter,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphView::getInOutEdges(const node n)const
{return (new InOutEdgesIterator(this,adaptativeFilter,n));}
//----------------------------------------------------------------
unsigned int SuperGraphView::numberOfNodes()const {
  if (nNodes<=0) 
    return (nNodes=this->SuperGraphAbstract::numberOfNodes());
  else
    return nNodes;
}
//----------------------------------------------------------------
unsigned int SuperGraphView::numberOfEdges() const {
  if (nEdges<=0) 
    return (nEdges=this->SuperGraphAbstract::numberOfEdges());
  else
    return nEdges;
}
//----------------------------------------------------------------
