//-*-c++-*-
/**
 Author: David Auber
 Email : auber@labri.fr
 Last modification : 20/08/2001
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "tulip/SuperGraph.h"
#include "tulip/SuperGraphImpl.h"
#include "tulip/SuperGraphView.h"
#include "tulip/SelectionProxy.h"
#include "tulip/GraphIterator.h"

using namespace std;

//----------------------------------------------------------------
SuperGraphImpl::SuperGraphImpl():SuperGraphAbstract(this) {
  _superGraph=this;
  nbNodes=0;
  nbEdges=0;
  outDegree.setAll(0);
}
//----------------------------------------------------------------
SuperGraphImpl::~SuperGraphImpl() {
  notifyDestroy(this);
#ifndef NDEBUG
  cerr << "SuperGraphImpl::~SuperGraphImpl delete proxyContainer" << endl;
#endif
#ifndef NDEBUG
  cerr << "SuperGraphImpl::~SuperGraphImpl Ok..." << endl;
#endif
  for (Nodes::iterator i=nodes.begin();i!=nodes.end();++i) {
    i->deallocateAll();
  }
}
//----------------------------------------------------------------
bool SuperGraphImpl::isElement(const node n) 
{return !nodeIds.is_free(n.id);}
//----------------------------------------------------------------
bool SuperGraphImpl::isElement(const edge e) {return !edgeIds.is_free(e.id);}
//----------------------------------------------------------------
node SuperGraphImpl::addNode() {
  node newNode(nodeIds.get());
  outDegree.set(newNode.id,0);
  while (nodes.size()<=newNode.id){
    nodes.push_back(EdgeContainer());
  }
  assert(nodes[newNode.id].empty());
  assert(nodes[newNode.id].empty());
  nbNodes++;
  notifyAddNode(this,newNode);
  return newNode;
}
//----------------------------------------------------------------
void SuperGraphImpl::addNode(const node n) {
  cerr << "SuperGraphImpl::addNode(const node n), The "
       << "current version of Tulip doesn't manage this"
       << "operation on Graph Implementation (coming soon)." << endl;
}
//----------------------------------------------------------------
edge SuperGraphImpl::addEdge(const node s,const node t) {
  assert(isElement(s) && isElement(t));
  pair< node , node > tmp(s,t);
  outDegree.set(s.id,1+outDegree.get(s.id));
  edge newEdge(edgeIds.get());
  while (edges.size()<=newEdge.id){
    edges.push_back(tmp);
  }
  edges[newEdge.id]=tmp;
  nodes[s.id].push_back(newEdge);
  nodes[t.id].push_back(newEdge);
  nbEdges++;
  notifyAddEdge(this,newEdge);
  return newEdge;
}
//----------------------------------------------------------------
void SuperGraphImpl::addEdge(const edge e) {
  cerr << "SuperGraphImpl::addEdge(const edge e), The"
       << " current version of Tulip doesn't manage this"
       << "operation on Graph Implementation (coming soon)." << endl;
}
//----------------------------------------------------------------
void SuperGraphImpl::delNode(const node n) {
  //Warning, the current implementation doesn't manage the 
  //updating of properties for upper_subgraph in the case of 
  //Super Graph Implementation
  externRemove(n);
  for(EdgeContainer::iterator i=nodes[n.id].begin();i!=nodes[n.id].end();++i) {
    externRemove(*i);
    node s=opposite(*i,n);
    removeEdge(nodes[s.id],*i);
    if (s==edges[(*i).id].first)
      outDegree.set(s.id,outDegree.get(s.id)-1);
    notifyDelEdge(this,*i);
  }
  nodes[n.id].clear();
  notifyDelNode(this,n);
  //  cout << "Nodes ID" << endl << nodeIds << endl;
  //  nodes.erase(n);
}
//----------------------------------------------------------------
void SuperGraphImpl::delEdge(const edge e) {
  //Warning, the current implementation doesn't manage the updating of 
  //properties for upper_subgraph in the case of Super Graph Implementation
  externRemove(e);
  node s=edges[e.id].first;
  node t=edges[e.id].second;
  outDegree.set(s.id,outDegree.get(s.id)-1);
  removeEdge(nodes[s.id],e);
  removeEdge(nodes[t.id],e);
  notifyDelEdge(this,e);
  //cout << "Edges ID" << endl << edgeIds << endl;
}
//----------------------------------------------------------------
void SuperGraphImpl::delAllNode(const node n){delNode(n);}
//----------------------------------------------------------------
void SuperGraphImpl::delAllEdge(const edge e){delEdge(e);}
//----------------------------------------------------------------
void SuperGraphImpl::setEdgeOrder(const node n,const vector<edge> &v ) {
  //  cerr << __PRETTY_FUNCTION__ << "not tested function" << endl;
  if (v.size()==0) return;

  MutableContainer<unsigned int> elements;
  elements.setAll(0);
  for (vector<edge>::const_iterator it=v.begin();it!=v.end();++it) {
    elements.set(it->id,elements.get(it->id)+1);
  }

  vector<edge> tmp;
  EdgeContainer adjacency=nodes[n.id];
  for (unsigned int i=0;i<v.size();++i) {
    if (elements.get(adjacency[i].id)>0) {
      elements.set(adjacency[i].id,elements.get(adjacency[i].id)-1);
    } else {
      tmp.push_back(adjacency[i]);
    }
    adjacency[i]=v[i];
  }

  unsigned int i=v.size();
  for (vector<edge>::const_iterator it=tmp.begin();it!=tmp.end();++it,++i) {
    adjacency[i]=*it;
  }
  
}
//----------------------------------------------------------------
void SuperGraphImpl::swapEdgeOrder(const node n,const edge e1 , const edge e2) {
  //  cerr << __PRETTY_FUNCTION__ << " not tested function" << endl;
  if (e1==e2) return;
  EdgeContainer adjacency=nodes[n.id];
  unsigned int e1Pos=UINT_MAX,e2Pos=UINT_MAX;
  for (unsigned int i=0;i<deg(n);++i) {
    if (adjacency[i]==e1) e1Pos=i;
    if (adjacency[i]==e2) e2Pos=i;
    if (e1Pos!=UINT_MAX && e2Pos!=UINT_MAX) break;
  }
  assert(e1Pos!=UINT_MAX && e2Pos!=UINT_MAX);
  adjacency[e1Pos]=e2;
  adjacency[e2Pos]=e1;
}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getNodes()const
{return (new xSGraphNodeIterator(this));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getInNodes(const node n)const
{return (new xInNodesIterator(this,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getOutNodes(const node n)const
{return (new xOutNodesIterator(this,n));}
//----------------------------------------------------------------
Iterator<node>* SuperGraphImpl::getInOutNodes(const node n)const
{return (new xInOutNodesIterator(this,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getEdges()const
{return (new xSGraphEdgeIterator(this));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getInEdges(const node n)const
{return (new xInEdgesIterator(this,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getOutEdges(const node n)const
{return (new xOutEdgesIterator(this,n));}
//----------------------------------------------------------------
Iterator<edge>* SuperGraphImpl::getInOutEdges(const node n)const
{return (new xInOutEdgesIterator(this,n));}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::deg(const node n) const {return nodes[n.id].size();}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::indeg(const node n) const {return nodes[n.id].size()-outDegree.get(n.id);}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::outdeg(const node n) const {return outDegree.get(n.id);}
//----------------------------------------------------------------
node SuperGraphImpl::source(const edge e)const{return edges[e.id].first;}
//----------------------------------------------------------------
node SuperGraphImpl::target(const edge e)const{return edges[e.id].second;}
//----------------------------------------------------------------
void SuperGraphImpl::reverse(const edge e) {
  assert(isElement(e));
  node s=edges[e.id].first;
  edges[e.id].first=edges[e.id].second;
  edges[e.id].second=s;
  outDegree.set(s.id,outDegree.get(s.id)-1);
  outDegree.set(edges[e.id].first.id,outDegree.get(edges[e.id].first.id)+1);
  notifyReverseEdge(this,e);
}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::numberOfEdges()const{return nbEdges;}
//----------------------------------------------------------------
unsigned int SuperGraphImpl::numberOfNodes()const{return nbNodes;}
//----------------------------------------------------------------
void SuperGraphImpl::removeEdge(EdgeContainer &c, const edge e) {
  edge e1=c.back();
  c.pop_back();
  if (e!=e1)
    for (EdgeContainer::iterator i=c.begin();i!=c.end();++i)
      if (*i==e) { *i=e1; break; }
}
//----------------------------------------------------------------
void SuperGraphImpl::externRemove(const edge e) {
  assert(isElement(e));
  Iterator<SuperGraph *>*itS=getSubGraphs();
  while (itS->hasNext())
    itS->next()->delEdge(e);
  delete itS;
  getPropertyManager()->erase(e);
  edgeIds.free(e.id);
  nbEdges--;
}
//----------------------------------------------------------------
void SuperGraphImpl::externRemove(const node n) {
  assert(isElement(n));
  Iterator<SuperGraph *>*itS=getSubGraphs();
  while (itS->hasNext())
    itS->next()->delNode(n);
  delete itS;
  getPropertyManager()->erase(n);
  nodeIds.free(n.id);
  nbNodes--;
}
//----------------------------------------------------------------
