#include <iostream>
#include <sstream>
#include <string>

#include <qstring.h>
#include <qpushbutton.h>
#include <qfiledialog.h>
#include <qcolordialog.h>
#include <qpainter.h>
#include <qfontmetrics.h>
#include <qapplication.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qvalidator.h>
#include <qsizepolicy.h>
#include <qmessagebox.h>
#include <qinputdialog.h>

#include <tulip/SuperGraph.h>
#include <tulip/PropertyManager.h>
#include <tulip/MetricProxy.h>
#include <tulip/StringProxy.h>
#include <tulip/SelectionProxy.h>
#include <tulip/LayoutProxy.h>
#include <tulip/IntProxy.h>
#include <tulip/ColorsProxy.h>
#include <tulip/SizesProxy.h>
#include <tulip/GlGraph.h>

#include "tulip/PropertyWidgets.h"

#define TABLEBUFSIZE 100

using namespace std;

//================================================================================
TulipPropertyTable::TulipPropertyTable(QWidget *parent, const char *name) :
  TulipTableWidget(parent, name), supergraph(0) {
  resetBackColor1();
  resetBackColor2();
  editedProperty=0;
  vScroll=verticalScrollBar();
  setColumnReadOnly(0, true);
  horizontalHeader()->setLabel(0, tr("Undefined"));
  connect(vScroll,SIGNAL(valueChanged(int)),SLOT(scroll(int)));
  connect(this,SIGNAL(valueChanged(int,int)),SLOT(changePropertyValue(int,int)));
  vScrollPos=0;
}

TulipPropertyTable::~TulipPropertyTable() {}

SuperGraph *TulipPropertyTable::getSuperGraph() const {return supergraph;}
void TulipPropertyTable::setSuperGraph(SuperGraph *s) {
  editedProperty=0;
  for (int i=0;i<numRows();++i) {
    clearCell(i,0);
    clearCell(i,1);
  }
  setNumRows(0);
  vScrollPos = 0;
  supergraph = s;
  update();
}

void TulipPropertyTable::changeProperty(SuperGraph *sg,const std::string &name) {
  supergraph=sg;
  if (!supergraph->existProperty(name)) {
    editedProperty=0;
  } 
  else {
    editedProperty=supergraph->getProperty(name);
    editedPropertyName=name;
  }
  setNumCols(2);
  setColumnStretchable(1, true);
  updateNbElements();
  update();
}

void TulipPropertyTable::selectNodeOrEdge(bool b) {
  displayNode=b;
}

void TulipPropertyTable::changePropertyValue(int i,int j) {
  if (displayNode)
    changePropertyNodeValue(i,j);
  else
    changePropertyEdgeValue(i,j);
}

void TulipPropertyTable::changePropertyEdgeValue(int i,int j) {  
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (editedProperty==0) return;
  Observable::holdObservers();
  bool result=true;
  string str=this->text(i,j).ascii();
  SelectionProxy *tmpSel=supergraph->getProperty<SelectionProxy>("viewSelection");  
  Iterator<edge> *it=supergraph->getEdges();
  edge tmp;
  for (int pos=0; it->hasNext();) {
    tmp=it->next();
    if (!_filterSelection || tmpSel->getEdgeValue(tmp)) {
      if (pos==i) {
        result=editedProperty->setEdgeStringValue(tmp,str);
        break;
      }
      ++pos;
    }
  } delete it;
  
  if (!result) {
    QMessageBox::critical( 0, "Tulip Property Editor Change Failed",
                           QString("The value entered for this edge is not correct,\n"
                                   "The change won't be applied\n"
                                   "Modify the entered value to apply the changes.")
                           );
  }
  else {
    emit tulipEdgePropertyChanged(supergraph, tmp, editedPropertyName.c_str(), str.c_str());
  }
  
  this->setColumnWidth(1, this->horizontalHeader()->width() - this->columnWidth(0));
  Observable::unholdObservers();
}

void TulipPropertyTable::changePropertyNodeValue(int i,int j) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (editedProperty==0) return;
  Observable::holdObservers();
  bool result=true;
  string str=(this->text(i,j).latin1());
  
  SelectionProxy *tmpSel=supergraph->getProperty<SelectionProxy>("viewSelection");  
  Iterator<node> *it=supergraph->getNodes();
  node tmp;
  for (int pos=0; it->hasNext();) {
      tmp=it->next();
      if (!_filterSelection || tmpSel->getNodeValue(tmp)) {
        if (pos==i) {
          result=editedProperty->setNodeStringValue(tmp,str);
          break;
        }
        ++pos;
      }
  } delete it;
  
  if (!result) {
    QMessageBox::critical( 0, "Tulip Property Editor Change Failed",
                           QString("The value entered for this node is not correct,\n"
                                   "The change won't be applied\n"
                                   "Modify the entered value to apply the changes.")
                           );
  }
  else {
    emit tulipNodePropertyChanged(supergraph, tmp, editedPropertyName.c_str(), str.c_str());
  }  
  this->setColumnWidth(1, this->horizontalHeader()->width() - this->columnWidth(0));
  Observable::unholdObservers();
}

void TulipPropertyTable::filterSelection(bool b) {
  _filterSelection=b;
  updateNbElements();
  update();
}

void TulipPropertyTable::scroll(int i) {
  if (editedProperty==0) return;
  int curId=i/20;
  bool toUpdate=false;
  if (curId>vScrollPos+TABLEBUFSIZE/4) {
    if (vScrollPos+TABLEBUFSIZE/2!=nbElement) {
      vScrollPos=curId;
      if (curId+TABLEBUFSIZE/2>nbElement) curId=nbElement-TABLEBUFSIZE/2;
      toUpdate=true;
    }
  }
  if (curId<vScrollPos-TABLEBUFSIZE/4) {
    if (vScrollPos-TABLEBUFSIZE/2!=0) {
      vScrollPos=curId;
      if (curId-TABLEBUFSIZE/2<0) curId=TABLEBUFSIZE/2-1;
      toUpdate=true;
    }
  }
  if (toUpdate){update();}
}

void TulipPropertyTable::update() {
  if (supergraph==0) return;
  if (displayNode)
    updateNodes();
  else
    updateEdges();
  QTable::repaint();
}

void TulipPropertyTable::updateEdges() {
  if (editedProperty==0) {
    return;
  }
  SelectionProxy *tmpSel=supergraph->getProperty<SelectionProxy>("viewSelection");
  setNumRows(nbElement);
  Iterator<edge> *it=supergraph->getEdges();
  for (int i=0; it->hasNext();) {
    char buf[16];
    edge tmp=it->next();
    if (!_filterSelection || tmpSel->getEdgeValue(tmp)) {
      if ((i>=vScrollPos-TABLEBUFSIZE/2) && (i<=vScrollPos+TABLEBUFSIZE/2)) {
	sprintf (buf,"%d", tmp.id );
	setText(i,0,buf );
	setTulipEdgeItem(editedProperty, editedPropertyName, tmp, i, 1);
      }
      else if (i>vScrollPos+TABLEBUFSIZE/2) break;
      ++i;
    }
  } delete it;
  adjustColumn(0);
  setColumnWidth(1, horizontalHeader()->width() - columnWidth(0));
}

void TulipPropertyTable::updateNodes() {
  if (editedProperty==0) return;
  SelectionProxy *tmpSel=supergraph->getProperty<SelectionProxy>("viewSelection");
  setNumRows(nbElement);
  Iterator<node> *it=supergraph->getNodes();
  for (int i=0; it->hasNext();) {
    char buf[16];
    node tmp=it->next();
    if (!_filterSelection || tmpSel->getNodeValue(tmp)) {
      if ((i>=vScrollPos-TABLEBUFSIZE/2) && (i<=vScrollPos+TABLEBUFSIZE/2)) {
	sprintf (buf,"%d", tmp.id );
	setText(i,0,buf );
	setTulipNodeItem(editedProperty, editedPropertyName, tmp, i, 1);
      }
      else if (i>vScrollPos+TABLEBUFSIZE/2) break;
      ++i;
    }
  } delete it;
  adjustColumn(0);
  setColumnWidth(1, horizontalHeader()->width() - columnWidth(0));
}

void TulipPropertyTable::updateNbElements() {
  if (supergraph==0) return;
  unsigned int nbNode,nbEdge;
  if (!_filterSelection) {
    nbNode=supergraph->numberOfNodes();
    nbEdge=supergraph->numberOfEdges();
  }
  else {
    SelectionProxy *tmpSel=supergraph->getProperty<SelectionProxy>("viewSelection");
    nbNode=0;
    nbEdge=0;
    Iterator<node> *it=supergraph->getNodes();
    while (it->hasNext())
      if (tmpSel->getNodeValue(it->next())) nbNode++;
    delete it;
    Iterator<edge> *itE=supergraph->getEdges();
    while (itE->hasNext())
      if (tmpSel->getEdgeValue(itE->next())) nbEdge++;
    delete itE;
  }
  if (displayNode) nbElement=nbNode; else nbElement=nbEdge;
}

void TulipPropertyTable::setAll() {
  if (displayNode)
   setAllNodeValue();
  else
   setAllEdgeValue();
  QTable::update();
}

void TulipPropertyTable::setAllNodeValue() {
  if (editedProperty==0) return;
  Observable::holdObservers();
  bool ok=false;
  string tmpStr;

  if (editedPropertyName == "viewShape") {
    QStringList tmp;
    Iterator<string> *itS=GlGraph::glyphFactory.availablePlugins();
    while (itS->hasNext()){
      tmp.append(QString(itS->next().c_str()));
    }delete itS;
    
    QString shapeName = QInputDialog::getItem(string("Property \"" + editedPropertyName + "\": set all node value").c_str(),
                                              "Please choose a shape",
                                              tmp, 0, false, &ok, this);
    if (ok) {
      stringstream ss;
      ss << GlGraph::glyphId(shapeName.ascii());
      tmpStr = ss.str();
    }
  }
  else if (typeid((*editedProperty)) == typeid(MetricProxy)) {
    double d = QInputDialog::getDouble(string("Property \"" + editedPropertyName + "\": set all node value").c_str(),
                                       "Please enter your value",
                                       0, -2147483647, 2147483647, 10,
                                       &ok, this);
    if (ok) {
      stringstream ss;
      ss << d;
      tmpStr = ss.str();
    }
  }
  else if (typeid(*editedProperty) == typeid(ColorsProxy)) {
    QColor color = QColorDialog::getColor();
    if (color.isValid()) {
      stringstream ss;
      ss << "(" << color.red() << "," << color.green() << "," << color.blue() << ",255)";
      tmpStr = ss.str();
      ok = true;
    }
  }
  else {
    QString text = QInputDialog::getText(string("Property \"" + editedPropertyName + "\": set all node value").c_str(),
                                         "Please enter your value",
                                         QLineEdit::Normal,QString::null, &ok, this);
    if (ok) tmpStr = text.ascii();
    else ok = false;
  }
  
  if (ok) {
    bool result=true;
    SelectionProxy *tmpSel=supergraph->getProperty<SelectionProxy>("viewSelection");

    if (!_filterSelection && supergraph->existLocalProperty(editedPropertyName)) {
      result=editedProperty->setAllNodeStringValue(tmpStr);
    }
    else {
      Iterator<node> *it=supergraph->getNodes();
      for (int nbNode=0; it->hasNext();) {
        node tmp=it->next();
        if (!_filterSelection || tmpSel->getNodeValue(tmp)) {
          result=editedProperty->setNodeStringValue(tmp,tmpStr);
          if (!result) break;
        }
        if (_filterSelection && tmpSel->getNodeValue(tmp)) {
          this->setText(nbNode, 1, tmpStr.c_str());
          nbNode++;
        }
      }
      delete it;
    }

    if (!result) {
      QMessageBox::critical(0, "Tulip Property Editor : set all node value Failed",
                            QString("The value entered for the nodes is not correct,\n"
                                    "The change won't be applied\n")
                            );
    }
    else {
      this->update();
    }
  }
  
  Observable::unholdObservers();
}

void  TulipPropertyTable::setAllEdgeValue() {
  if (editedProperty==0) return;
  Observable::holdObservers();
  bool ok=false;
  string tmpStr;

  if (typeid(*editedProperty) == typeid(ColorsProxy)) {
    QColor color = QColorDialog::getColor();
    if (color.isValid()) {
      stringstream ss;
      ss << "(" << color.red() << "," << color.green() << "," << color.blue() << ",255)";
      tmpStr = ss.str();
      ok = true;
    }
  }
  else {
    QString text = QInputDialog::getText(string("Property \"" + editedPropertyName + "\": set all edge value").c_str(),
                                         "Please enter your value",
                                         QLineEdit::Normal, QString::null, &ok, this);
    if (ok) tmpStr = text.ascii();
    else ok = false;
  }
  
  if (ok) {
    bool result=true;
    SelectionProxy *tmpSel=supergraph->getProperty<SelectionProxy>("viewSelection");
    if (!_filterSelection && supergraph->existLocalProperty(editedPropertyName)) {
      result=editedProperty->setAllEdgeStringValue(tmpStr);
    }
    else {
      Iterator<edge> *itE=supergraph->getEdges();
      for (int nbEdge=0; itE->hasNext();) {
        edge tmp=itE->next();
        if (!_filterSelection || tmpSel->getEdgeValue(tmp)) {
          result=editedProperty->setEdgeStringValue(tmp, tmpStr);
          if (!result) break;
        }
        if (_filterSelection && tmpSel->getEdgeValue(tmp)) {
          this->setText(nbEdge, 1, tmpStr.c_str());
          nbEdge++;
        }
      }
      delete itE;
    }
    
    if (!result) {
      QMessageBox::critical(0, "Tulip Property Editor set all node value Failed",
                            QString("The value entered for the edges is not correct,\n"
                                    "change won't be applied\n")
                            );
    }
    else {
      this->update();
    }
  }
  Observable::unholdObservers();
}
