//**********************************************************************
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <map>
#include <vector>
#include <limits.h>
#include <fstream>
#include <iostream>
#include <unistd.h>

#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qapplication.h>
#include <qcolordialog.h>
#include <qfiledialog.h>
#include <qinputdialog.h>
#include <qworkspace.h>
#include <qmenubar.h>
#include <qtable.h>
#include <qvbox.h>
#include <qstatusbar.h>
#include <qpixmap.h>
#include <qstrlist.h>
#include <qimage.h>
#include <qpainter.h>
#include <qprogressdialog.h>
#include <qsettings.h>

#include <gzstream.h>
#include <tulip/TlpTools.h>
#include <tulip/Cluster.h>
#include <tulip/SubGraph.h>
#include <tulip/PropertyProxyContainer.h>
#include <tulip/Reflect.h>
#include <tulip/GlGraphWidget.h>
#include <tulip/TulipElementProperties.h>
#include <tulip/PropertyWidgets.h>
#include <tulip/ClusterTree.h>

#include "NavigateGlGraph.h"
#include "PropertyDialog.h"
#include "viewGl.h"
#include "Application.h"
#include "View3DSetup.h"
#include "QtProgress.h"

#define WITHQT3

using namespace std;
using namespace Tulip;

void myToolTip::maybeTip( const QPoint & p ){
  node tmpNode;
  edge tmpEdge;
  AtomType type;
  QString tmp;
  if (glWidget->getGlGraph()->doSelect(p.x(), p.y(), type, tmpNode, tmpEdge)) {
    switch(type) {
    case NODE: tip(QRect(p.x()-2,p.y()-2,4,4),tmp.setNum(tmpNode.id)); break;
    case EDGE: break;
    }
  }
}

//**********************************************************************
///Constructor of ViewGl
viewGl::viewGl(QWidget* parent,	const char* name,SuperGraph *superGraph):QMainWindow( parent, name )
{
  //   cerr << "Constructor viewGL" << endl;
  Observable::holdObservers();

  _superGraph=superGraph;
  _currentGraph=superGraph;

  /* restore app settings */
  QSettings settings;
  settings.insertSearchPath(QSettings::Windows, "/LaBRI");
  //settings.insertSearchPath(QSettings::Unix, TULIP_DATADIR); //FIXME:TULIP_DATADIR not defined
  QSize mySize(1130, 630);

  {
    int tmp = settings.readNumEntry("/Tulip/geometry/width", 1130);
    mySize.setWidth(tmp);
    tmp = settings.readNumEntry("/Tulip/geometry/height", 630);
    mySize.setHeight(tmp);
  }
  resize(mySize);
  /*
    QPixmap image0;
    image0.load( "../bitmap/logo32x32.bmp" );
    setIcon( image0 );
  */

  //  myCluster=superGraph->getClusterTree();
  setCaption(name);

  //+++++++++++++++++++++++++++
  //Building of the pop-up menu in the menuBar
#define FILL_MENU(type, menuName)                                                                       \
  {                                                                                                     \
    TemplateFactory<type##Factory, Type, PropertyContext>::ObjectCreator::const_iterator it;            \
    for (it = type##Proxy::factory.objMap.begin(); it != type##Proxy::factory.objMap.end(); ++it)     \
      menuName.insertItem(QString(it->first.c_str()));                                                  \
  }
  FILL_MENU(Int, intMenu);
  FILL_MENU(String, stringMenu);
  FILL_MENU(Sizes, sizesMenu);
  FILL_MENU(Colors, colorsMenu);
  FILL_MENU(Layout, layoutMenu);
  FILL_MENU(Metric, metricMenu);
  FILL_MENU(Selection, selectMenu);
#undef FILL_MENU
  {
    TemplateFactory<ClusteringFactory,Clustering,ClusterContext>::ObjectCreator::const_iterator it;
    for (it = TlpTools::clusteringFactory.objMap.begin(); it != TlpTools::clusteringFactory.objMap.end(); ++it)
      clusteringMenu.insertItem(it->first.c_str());
  }
  {
    TemplateFactory<ExportModuleFactory,ExportModule,ClusterContext>::ObjectCreator::const_iterator it;
    for (it = TlpTools::exportFactory.objMap.begin(); it != TlpTools::exportFactory.objMap.end(); ++it)
      exportGraphMenu.insertItem( it->first.c_str() );
  }

  //MDI mode
  QVBox* vb = new QVBox(this);
  vb->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
  workspace = new QWorkspace(vb);
  workspace->setScrollBarsEnabled(TRUE);
  setCentralWidget(vb);

  clusterTreeWidget = new ClusterTree(_superGraph,workspace);
  {
    int x, y, w, h;
    x = settings.readNumEntry("/Tulip/Cluster Tree/x", 550);
    y = settings.readNumEntry("/Tulip/Cluster Tree/y", 250);
    w = settings.readNumEntry("/Tulip/Cluster Tree/width", clusterTreeWidget->width());
    h = settings.readNumEntry("/Tulip/Cluster Tree/height", clusterTreeWidget->height());
    clusterTreeWidget->resize(QSize(w, h));
    clusterTreeWidget->move(x, y);
  }
  if (!settings.readBoolEntry("/Tulip/Cluster Tree/show", true)) clusterTreeWidget->hide();
  else clusterTreeWidget->show();

  updateSatutBar();

  textView = new QTextView(workspace,"Text Console");
  textView->setCaption(trUtf8("Text Console"));
  textView->setMinimumSize(200, 100);
  textView->setMaximumSize(32767, 32767);
  textView->show();
  {
    int x, y, w, h;
    x = settings.readNumEntry("/Tulip/Text Console/x", 554);
    y = settings.readNumEntry("/Tulip/Text Console/y", 0);
    w = settings.readNumEntry("/Tulip/Text Console/width", textView->width());
    h = settings.readNumEntry("/Tulip/Text Console/height", textView->height());
    textView->resize(QSize(w, h));
    textView->move(x, y);
  }
  if (!settings.readBoolEntry("/Tulip/Text Console/show", true)) textView->hide();
  else textView->show();
  
  glWidget1 = new NavigateGlGraph(workspace, "3D Graph View");
  glWidget1->setCaption(trUtf8("3D Graph View"));
  {
    int x, y, w, h;
    x = settings.readNumEntry("/Tulip/3D Graph View/x", 0);
    y = settings.readNumEntry("/Tulip/3D Graph View/y", 0);
    w = settings.readNumEntry("/Tulip/3D Graph View/width", glWidget1->width());
    h = settings.readNumEntry("/Tulip/3D Graph View/height", glWidget1->height());
    glWidget1->resize(QSize(w, h));
    glWidget1->move(x, y);
  }  
  glWidget = glWidget1->getGlGraphWidget();
  glWidget->setSuperGraph(_superGraph);
  glWidget->setQTextView(textView);
  glWidget1->setMinimumSize(0, 0);
  glWidget1->setMaximumSize(32767, 32767);
  glWidget1->setFocusPolicy(QWidget::NoFocus);
  glWidget1->setBackgroundMode(QWidget::PaletteBackground);
  
  glWidget->installEventFilter(this);
  //attach ToolTip to glWidget
  new myToolTip(glWidget, "toolTip", glWidget);
  if (!settings.readBoolEntry("/Tulip/3D Graph View/show", true)) glWidget->hide();
  else glWidget1->show();

  propD=new PropertyDialog(glWidget1->getGlGraphWidget()->getGlGraph(), workspace);
  propD->setCaption("Property Editor");
  {
    int x, y, w, h;
    x = settings.readNumEntry("/Tulip/Property Editor/x", 650);
    y = settings.readNumEntry("/Tulip/Property Editor/y", 0);
    w = settings.readNumEntry("/Tulip/Property Editor/width", propD->width());
    h = settings.readNumEntry("/Tulip/Property Editor/height", propD->height());
    propD->resize(QSize(w, h));
    propD->move(x, y);
  }
  propD->setSuperGraph(_superGraph);
  if (!settings.readBoolEntry("/Tulip/Property Editor/show", true)) propD->hide();
  else propD->show();

  QStringList listedProperties;
  Iterator<string> *itP = _superGraph->getPropertyProxyContainer()->getLocalProperties();
  while(itP->hasNext()) {
    const string &s = itP->next();
    if (s != "outdegree")
      listedProperties << s.c_str();
  }
  delete itP;
  
  nodeProperties = new TulipElementProperties(_superGraph, listedProperties, listedProperties, workspace);
  ((Application*)qApp)->nodeProperties = nodeProperties;
  nodeProperties->propertyTable->setGlyphTable(glWidget->getGlGraph()->getGlyphTable());
  {
    int x, y, w, h;
    x = settings.readNumEntry("/Tulip/Element Property/x", 700);
    y = settings.readNumEntry("/Tulip/Element Property/y", 150);
    w = settings.readNumEntry("/Tulip/Element Property/width", nodeProperties->width());
    h = settings.readNumEntry("/Tulip/Element Property/height", nodeProperties->height());
    textView->resize(QSize(w, h));
    textView->move(x, y);
  }
  if (!settings.readBoolEntry("/Tulip/Element Property/show", true)) nodeProperties->hide();
  else nodeProperties->show();

  //connect signals related to property change
  connect(nodeProperties,
          SIGNAL(tulipNodePropertyChanged(SuperGraph *, const node &, const QString &, const QString &)),
          propD,
          SLOT(updateNodes()));
  
  connect(propD,
          SIGNAL(tulipNodePropertyChanged(SuperGraph *, const node &, const QString &, const QString &)),
          nodeProperties,
          SLOT(nodePropertyChanged(SuperGraph *, const node &, const QString &, const QString &)));
  
  connect(glWidget, SIGNAL(nodeClicked(const node &)), nodeProperties, SLOT(setCurrentNode(const node &)));

  //connect signals related to supergraph replacement
  connect(clusterTreeWidget, SIGNAL(supergraphChanged(SuperGraph *)), glWidget, SLOT(setSuperGraph(SuperGraph *)));
  connect(clusterTreeWidget, SIGNAL(supergraphChanged(SuperGraph *)), propD, SLOT(setSuperGraph(SuperGraph *)));
  connect(clusterTreeWidget, SIGNAL(supergraphChanged(SuperGraph *)), nodeProperties, SLOT(setSuperGraph(SuperGraph *)));

  connect(clusterTreeWidget, SIGNAL(aboutToRemoveView(SuperGraph *)), this, SLOT(superGraphAboutToBeRemoved(SuperGraph *)));
  connect(clusterTreeWidget, SIGNAL(aboutToRemoveAllView(SuperGraph *)), this, SLOT(superGraphAboutToBeRemoved(SuperGraph *)));
  
  view3DSetup = new View3DSetup(glWidget->getGlGraph(),workspace);
  view3DSetup->setCaption(trUtf8("3D view setup"));
  {
    int x, y, w, h;
    x = settings.readNumEntry("/Tulip/3D Setup/x", 650);
    y = settings.readNumEntry("/Tulip/3D Setup/y", 300);
    w = settings.readNumEntry("/Tulip/3D Setup Console/width", view3DSetup->width());
    h = settings.readNumEntry("/Tulip/3D Setup/height", view3DSetup->height());
    view3DSetup->resize(QSize(w, h));
    view3DSetup->move(x, y);
  }

  QStrList listFormat=QImageIO::outputFormats();
  char *tmp=listFormat.first();
  while (tmp!=0) {
    exportImageMenu.insertItem(tmp);
    tmp=listFormat.next();
  }
  
  dialogMenu.insertItem(trUtf8("Cluster Tree"));
  dialogMenu.insertItem(trUtf8("Configuration"));
  dialogMenu.insertItem(trUtf8("Property Editor"));
  dialogMenu.insertItem(trUtf8("Node Properties"));
  dialogMenu.insertItem(trUtf8("Text Console"));
  dialogMenu.insertItem(trUtf8("View 3D"));
  
  //==============================================================
  //Menu building
  exportImageMenu.insertItem(trUtf8("EPS"));
  graphMenu.insertItem(trUtf8("&Save Graph as"), &exportGraphMenu);
  graphMenu.insertItem(trUtf8("&Save Picture as ") , &exportImageMenu); //this , SLOT( outputImage() ));
  graphMenu.insertItem(trUtf8("&Close"), this, SLOT(closeWin()));
  
  viewMenu.insertItem(trUtf8("Redraw View"), this, SLOT(redrawView()));
  viewMenu.insertItem(trUtf8("Center View"), this, SLOT(centerView()));
  viewMenu.insertItem(trUtf8("Restore View"), this, SLOT(restoreView()));
  viewMenu.insertItem(trUtf8("Dialogs"),  &dialogMenu);
  viewMenu.insertSeparator();
  viewMenu.insertItem(trUtf8("Save settings"), this, SLOT(saveDesktopLayout()));
  
  editMenu.insertItem(trUtf8("Deselect All"), this, SLOT(deselectALL()));
  editMenu.insertItem(trUtf8("Reverse Selection"), this, SLOT(reverseSelection()));
  editMenu.insertItem(trUtf8("Delete Selection"), glWidget, SLOT(delSelection()));
  editMenu.insertItem(trUtf8("New View"), this, SLOT(newView()));
  editMenu.insertItem(trUtf8("Reverse slected edges direction"), this, SLOT( reverseSelectedEdgeDirection() ));
  
  propertyMenu.insertItem(trUtf8("&Selection"), &selectMenu );
  propertyMenu.insertItem(trUtf8("&Layout"), &layoutMenu );
  propertyMenu.insertItem(trUtf8("&Metric"), &metricMenu );
  propertyMenu.insertItem(trUtf8("&Colors"), &colorsMenu );
  propertyMenu.insertItem(trUtf8("&Integer"), &intMenu );
  propertyMenu.insertItem(trUtf8("&String"), &stringMenu );
  propertyMenu.insertItem(trUtf8("&Sizes"), &sizesMenu );
  menuBar()->insertItem(trUtf8("&Graph"), &graphMenu );
  menuBar()->insertItem(trUtf8("&Edit"), &editMenu );
  menuBar()->insertItem(trUtf8("&Property"), &propertyMenu );
  menuBar()->insertItem(trUtf8("&Clustering"), &clusteringMenu );
  menuBar()->insertItem(trUtf8("&View"), &viewMenu );
  //+++++++++++++++++++++++++++
  //Connection of the menu
  connect(&stringMenu, SIGNAL(activated(int)),SLOT(changeString(int)));
  connect(&metricMenu, SIGNAL(activated(int)),SLOT(changeMetric(int)));
  connect(&layoutMenu, SIGNAL(activated(int)),SLOT(changeLayout(int)));
  connect(&selectMenu, SIGNAL(activated(int)),SLOT(changeSelection(int)));
  connect(&clusteringMenu, SIGNAL(activated(int)),SLOT(makeClustering(int)));
  connect(&sizesMenu, SIGNAL(activated(int)),SLOT(changeSizes(int)));
  connect(&intMenu, SIGNAL(activated(int)),SLOT(changeInt(int)));
  connect(&colorsMenu, SIGNAL(activated(int)),SLOT(changeColors(int)));
  connect(&exportGraphMenu, SIGNAL(activated(int)),SLOT(exportGraph(int)));
  connect(&exportImageMenu, SIGNAL(activated(int)),SLOT(exportImage(int)));
  connect(&dialogMenu, SIGNAL(activated(int)),SLOT(showDialog(int)));
  //+++++++++++++++++++++++++++
  windowsMenu = new QPopupMenu( this );
  windowsMenu->setCheckable( TRUE );
  connect(  windowsMenu , SIGNAL( aboutToShow() ), this, SLOT( windowsMenuAboutToShow() ) );
  menuBar()->insertItem( trUtf8("&Windows"), windowsMenu );

  restoreView();
  Observable::unholdObservers();
}
//**********************************************************************
///Destructor of viewGl
viewGl::~viewGl() {
#ifndef NDEBUG
  cerr << "viewGL:~viewGL" << endl; 
#endif
  if (_superGraph!=NULL) delete _superGraph;
}
//**********************************************************************
const GlGraphWidget *viewGl::getGlGraphWidget() const
{
  return glWidget;
}
//**********************************************************************
void viewGl::setParameters(const DataSet data)
{
  cerr << __PRETTY_FUNCTION__ << endl;
  glWidget->getGlGraph()->setParameters(data);
  view3DSetup->syncFromGlGraph();
  clusterTreeWidget->currentSuperGraphChanged(glWidget->getSuperGraph());
  nodeProperties->setSuperGraph(glWidget->getSuperGraph());
  propD->setSuperGraph(glWidget->getSuperGraph());
}
//**********************************************************************
void viewGl::saveDesktopLayout() {
  QSettings settings;
  settings.insertSearchPath(QSettings::Windows, "/LaBRI");
  {
    int tmp = this->width();
    settings.writeEntry("/Tulip/geometry/width", tmp);
    tmp = this->height();
    settings.writeEntry("/Tulip/geometry/height", tmp);
  }
  //***************
  //QWorkspace embed widgets into one with window decoration. See QWorkspace doc about MDI geometry.
#define TULIP_SAVE_CW(KEY, WIDGET)                                      \
  settings.writeEntry("/Tulip/" KEY "/x", WIDGET->parentWidget()->x()); \
  settings.writeEntry("/Tulip/" KEY "/y", WIDGET->parentWidget()->y()); \
  settings.writeEntry("/Tulip/" KEY "/width", WIDGET->width());         \
  settings.writeEntry("/Tulip/" KEY "/height", WIDGET->height());       \
  settings.writeEntry("/Tulip/" KEY "/show", WIDGET->isVisible());
  //***************
  TULIP_SAVE_CW("Cluster Tree", clusterTreeWidget);
  TULIP_SAVE_CW("Text Console", textView);
  TULIP_SAVE_CW("3D Graph View", glWidget1);
  TULIP_SAVE_CW("Property Editor", propD);
  TULIP_SAVE_CW("Element Property", nodeProperties);
  TULIP_SAVE_CW("3D Setup", view3DSetup)
#undef TULIP_SAVE_CW
}
//**********************************************************************
void viewGl::updateSatutBar(){
  char tmp[255];
  sprintf(tmp,trUtf8("Ready, Nodes:%d, Edges:%d"),clusterTreeWidget->getSuperGraph()->numberOfNodes(),clusterTreeWidget->getSuperGraph()->numberOfEdges());
  statusBar()->message(tmp);
}
//**********************************************************************
void viewGl::outputEPS() {
  QString s( QFileDialog::getSaveFileName());
  if (!s.isNull()) {
    glWidget->getGlGraph()->outputEPS(64000000,true,s.latin1());
  }
}
//**********************************************************************
void viewGl::exportImage(int id) {
  QString format(exportImageMenu.text(id));
  if (format == "EPS") {
    outputEPS();
    return;
  }
  QString filename(QFileDialog::getSaveFileName());
  if (filename.isNull()) return;
  QImage image(glWidget->grabFrameBuffer(false));
  image.save(filename, format);
}
//**********************************************************************
void viewGl::exportGraph(int id) {
  string filename(QFileDialog::getSaveFileName(this->caption()).latin1());
  if (filename == "") return;
  
  DataSet dataSet;

  string format(exportGraphMenu.text(id).latin1());
  ostream *os;
  if (filename.rfind(".gz") == (filename.length() - 3))
    os = new ogzstream(filename.c_str());
  else
    os = new ofstream(filename.c_str());
  
  StructDef parameter=TlpTools::exportFactory.getParam(format);

  dataSet.set("displaying", glWidget->getGlGraph()->getParameters());
  
  if (!TlpTools::exportGraph(clusterTreeWidget->getSuperGraph(), *os, format, dataSet, NULL)) {
    QMessageBox::critical( 0, trUtf8("Tulip export Failed"),
			   trUtf8("The file has not been exported")
			   );
  }
  delete os;
}
//**********************************************************************
void viewGl::windowsMenuActivated( int id ) {
  QWidget* w = workspace->windowList().at( id );
  if ( w ) w->setFocus();
}
//**********************************************************************
void viewGl::windowsMenuAboutToShow() {
  windowsMenu->clear();
  int cascadeId = windowsMenu->insertItem(trUtf8("&Cascade"), workspace, SLOT(cascade() ) );
  int tileId = windowsMenu->insertItem(trUtf8("&Tile"), workspace, SLOT(tile() ) );
  if ( workspace->windowList().isEmpty() ) {
    windowsMenu->setItemEnabled( cascadeId, FALSE );
    windowsMenu->setItemEnabled( tileId, FALSE );
  }
  windowsMenu->insertSeparator();
  QWidgetList windows = workspace->windowList();
  for ( int i = 0; i < int(windows.count()); ++i ) {
    int id = windowsMenu->insertItem(windows.at(i)->caption(),
				     this, SLOT( windowsMenuActivated( int ) ) );
    windowsMenu->setItemParameter( id, i );
    windowsMenu->setItemChecked( id, workspace->activeWindow() == windows.at(i) );
  }
}
//**********************************************************************
///Delete the windows
int viewGl::closeWin(){
  delete this;
  return true;
}
//**********************************************************************
bool viewGl::eventFilter(QObject *obj, QEvent *e) {
  if ((typeid(*obj) == typeid(GlGraphWidget)) &&
      (e->type() == QEvent::MouseButtonRelease)) {
    QMouseEvent *me = (QMouseEvent *) e;
    GlGraphWidget *glw = (GlGraphWidget *) obj;
    if (me->button()==RightButton) {
      glw->setContextCoord(me->x(), me->y());
      
      QPopupMenu *contextMenu=new QPopupMenu(this,"dd");
      contextMenu->insertItem(trUtf8("Delete"), glw, SLOT(contextDel()));
      contextMenu->insertItem(trUtf8("Select"), glw, SLOT(contextSelect()));
      contextMenu->insertItem(trUtf8("Add/Remove selection"), glw, SLOT(contextAddRemoveSelection()));
      contextMenu->insertItem(trUtf8("Properties"), glw, SLOT(contextProperties()));
      contextMenu->exec(me->globalPos());
      delete contextMenu;
      return true;
    }
    else {
      return false;
    }
  }
  return false;
}
//**********************************************************************
void viewGl::focusInEvent ( QFocusEvent * ){}
//**********************************************************************
///Make a new clustering of the view graph
void viewGl::makeClustering(int id) {
  Observable::holdObservers();
  string name(clusteringMenu.text(id).latin1());
  string erreurMsg;
  if (!TlpTools::clusterizeGraph(clusterTreeWidget->getSuperGraph(),erreurMsg,0,name)){
    QMessageBox::critical( 0, trUtf8("Tulip Algorithm Check Failed"),QString((name + "::" + erreurMsg).c_str()));
  }
  clusterTreeWidget->update();
  Observable::unholdObservers();
}

//**********************************************************************
///Change the current label use by the view
void viewGl::changeString(int id) {
  Observable::holdObservers();
  string name(stringMenu.text(id).latin1());
  string erreurMsg;
  bool resultBool,cached;
  StringProxy *myString=getLocalProxy<StringProxy>(clusterTreeWidget->getSuperGraph(),name,cached,resultBool,erreurMsg);
  if (cached) resultBool=myString->recompute(erreurMsg);
  if (!resultBool) {
    QMessageBox::critical( 0, trUtf8("Tulip Algorithm Check Failed"),QString((name + "::" + erreurMsg).c_str()));
  }
  else {
    *getLocalProxy<StringProxy>(clusterTreeWidget->getSuperGraph(),"viewLabel")=*myString;
    glWidget->setSuperGraph(clusterTreeWidget->getSuperGraph());
  }
  clusterTreeWidget->getSuperGraph()->getPropertyProxyContainer()->delLocalProxy(name);
  propD->setSuperGraph(clusterTreeWidget->getSuperGraph());
  Observable::unholdObservers();
}

//**********************************************************************
///Change the current Selection use by the view
void viewGl::changeSelection(int id) {
  Observable::holdObservers();
  string name(selectMenu.text(id).latin1());
  string erreurMsg;
  bool resultBool,cached;
  SelectionProxy *mySelection=getLocalProxy<SelectionProxy>(clusterTreeWidget->getSuperGraph(),name,cached,resultBool,erreurMsg);
  if (cached) resultBool=mySelection->recompute(erreurMsg);
  if (!resultBool) {
    QMessageBox::critical( 0, trUtf8("Tulip Algorithm Check Failed"),QString((name + "::" + erreurMsg).c_str()));
  }
  else {
    *getLocalProxy<SelectionProxy>(clusterTreeWidget->getSuperGraph(),"viewSelection")=*mySelection;
    glWidget->setSuperGraph(clusterTreeWidget->getSuperGraph());
  }
  clusterTreeWidget->getSuperGraph()->getPropertyProxyContainer()->delLocalProxy(name);
  propD->setSuperGraph(clusterTreeWidget->getSuperGraph());
  Observable::unholdObservers();
}
//**********************************************************************
///Change the current Metric use by the view
void viewGl::changeMetric(int id) {
  Observable::holdObservers();
  string name(metricMenu.text(id).latin1());
  string erreurMsg;
  bool cached;
  bool resultBool;
  string name2=trUtf8("Computing Metric ").latin1()+name;
  MyProgress myProgress(this,name2);
  myProgress.hide();
  MetricProxy *myMetric=getLocalProxy<MetricProxy>(clusterTreeWidget->getSuperGraph(),name,cached,resultBool,erreurMsg,&myProgress);
  if (cached) resultBool=myMetric->recompute(erreurMsg);
  if (!resultBool) {
    QMessageBox::critical( 0, trUtf8("Tulip Algorithm Check Failed"), QString((name + "::" + erreurMsg).c_str()) );
  }
  else {
    PropertyProxyContainer *ppC=clusterTreeWidget->getSuperGraph()->getPropertyProxyContainer();
    if (myProgress.compute==false) {
      ppC->delLocalProxy(name);
    }
    else
    if (view3DSetup->isAutoMapMetric()) {
      *getLocalProxy<MetricProxy>(clusterTreeWidget->getSuperGraph(),"viewMetric") = *myMetric;

      ColorsProxy *myColorMap = getLocalProxy<ColorsProxy>(clusterTreeWidget->getSuperGraph(),
							 view3DSetup->getColorMapping(), cached, resultBool, erreurMsg);

      if (cached) resultBool=myColorMap->recompute(erreurMsg);
      
      getLocalProxy<ColorsProxy>(clusterTreeWidget->getSuperGraph(),"viewColors")->setAllNodeValue(view3DSetup->getNodeColor());
      getLocalProxy<ColorsProxy>(clusterTreeWidget->getSuperGraph(),"viewColors")->setAllEdgeValue(view3DSetup->getEdgeColor());
      
      *getLocalProxy<ColorsProxy>(clusterTreeWidget->getSuperGraph(),"viewColors") = *myColorMap;

      glWidget->setSuperGraph(clusterTreeWidget->getSuperGraph());
      ppC->delLocalProxy(name);
      ppC->delLocalProxy(view3DSetup->getColorMapping());
    }
  }
  propD->setSuperGraph(clusterTreeWidget->getSuperGraph());
  Observable::unholdObservers();
}
//**********************************************************************
///Change the current Layout use by the view
void viewGl::changeLayout(int id) {
  Observable::holdObservers();
  string name(layoutMenu.text(id).latin1());
  string erreurMsg;
  bool resultBool,cached;
  string name2=trUtf8("Computing Layout ").latin1()+name;
  MyProgress myProgress(this,name2);
  myProgress.hide();
  LayoutProxy *myLayout=getLocalProxy<LayoutProxy>(clusterTreeWidget->getSuperGraph(),name,cached,resultBool,erreurMsg,&myProgress);
  if (cached)  resultBool=myLayout->recompute(erreurMsg);
  if (!resultBool) {
    clusterTreeWidget->getSuperGraph()->getPropertyProxyContainer()->delLocalProxy(name);
    QMessageBox::critical( 0, trUtf8("Tulip Algorithm Check Failed"), QString((name + "::" + erreurMsg).c_str()) );
  }
  else {
    if(myProgress.compute==true) {
      if (view3DSetup->isAutoScale()) myLayout->perfectAspectRatio();
      *getLocalProxy<LayoutProxy>(clusterTreeWidget->getSuperGraph(),"viewLayout")=*myLayout;
      glWidget->setSuperGraph(clusterTreeWidget->getSuperGraph());
      restoreView();
    }
    clusterTreeWidget->getSuperGraph()->getPropertyProxyContainer()->delLocalProxy(name);
    propD->setSuperGraph(clusterTreeWidget->getSuperGraph());
  }
  Observable::unholdObservers();
}
//**********************************************************************
void viewGl::changeInt(int id) {
  Observable::holdObservers();
  string name(layoutMenu.text(id).latin1());
  string erreurMsg;
  bool resultBool,cached; 
  IntProxy *myInt=getLocalProxy<IntProxy>(clusterTreeWidget->getSuperGraph(),name,cached,resultBool,erreurMsg);
  if (cached)  resultBool=myInt->recompute(erreurMsg);
  if (!resultBool) {
    QMessageBox::critical( 0, trUtf8("Tulip Algorithm Check Failed"), QString((name + "::" + erreurMsg).c_str()) );
  }
  else {
    *getLocalProxy<IntProxy>(clusterTreeWidget->getSuperGraph(),"viewInt")=*myInt;
    glWidget->setSuperGraph(clusterTreeWidget->getSuperGraph());
  }
  clusterTreeWidget->getSuperGraph()->getPropertyProxyContainer()->delLocalProxy(name);
  propD->setSuperGraph(clusterTreeWidget->getSuperGraph());
  Observable::unholdObservers();
}
//**********************************************************************
void viewGl::changeColors(int id) {
  Observable::holdObservers();
  string name(colorsMenu.text(id).latin1());
  string erreurMsg;
  bool resultBool,cached; 
  ColorsProxy *myColors=getLocalProxy<ColorsProxy>(clusterTreeWidget->getSuperGraph(),name,cached,resultBool,erreurMsg);
  if (cached)  resultBool=myColors->recompute(erreurMsg);
  if (!resultBool) {
    QMessageBox::critical( 0, trUtf8("Tulip Algorithm Check Failed"), QString((name + "::" + erreurMsg).c_str()) );
  }
  else {
    *getLocalProxy<ColorsProxy>(clusterTreeWidget->getSuperGraph(),"viewColors")=*myColors;
    glWidget->setSuperGraph(clusterTreeWidget->getSuperGraph());
  }
  clusterTreeWidget->getSuperGraph()->getPropertyProxyContainer()->delLocalProxy(name);
  propD->setSuperGraph(clusterTreeWidget->getSuperGraph());
  Observable::unholdObservers();
  //  cerr << "viewGl::changeColors Ok" << endl;
}
//**********************************************************************
void viewGl::changeSizes(int id)
{
  //  cerr << "viewGl::changeSizes" << endl;
  Observable::holdObservers();
  string name(sizesMenu.text(id).latin1());
  string erreurMsg;
  bool resultBool,cached; 
  SizesProxy *mySizes=getLocalProxy<SizesProxy>(clusterTreeWidget->getSuperGraph(),name,cached,resultBool,erreurMsg);
  if (cached)  resultBool=mySizes->recompute(erreurMsg);
  if (!resultBool) {
      QMessageBox::critical( 0, trUtf8("Tulip Algorithm Check Failed"), QString((name + "::" + erreurMsg).c_str()) );
    }
  else {
    *getLocalProxy<SizesProxy>(clusterTreeWidget->getSuperGraph(),"viewSize")=*mySizes;
    glWidget->setSuperGraph(clusterTreeWidget->getSuperGraph());
  }
  clusterTreeWidget->getSuperGraph()->getPropertyProxyContainer()->delLocalProxy(name);
  propD->setSuperGraph(clusterTreeWidget->getSuperGraph());
  Observable::unholdObservers();
  //  cerr << "viewGl::changeLayout Ok" << endl;
}

//===========================================================================
//Fonction de paramtrage de la fentre Couleurs ect ...

///Change the base node color of the node in the graph view
void viewGl::showDialog(int id){
  QString name(dialogMenu.text(id));
  if (name==trUtf8("Cluster Tree")) {
    clusterTreeWidget->show();
    clusterTreeWidget->raise();
    clusterTreeWidget->setFocus();
  }
  if (name==trUtf8("Configuration")) {
    view3DSetup->show();
    view3DSetup->raise();
    view3DSetup->setFocus();
  }
  if (name==trUtf8("Property Editor")) {
    propD->show();
    propD->raise();
    propD->setFocus();
  }
  if (name==trUtf8("Node Properties")) {
    nodeProperties->show();
    nodeProperties->raise();
    nodeProperties->setFocus();
  }
  if (name==trUtf8("Text Console")) {
    textView->show();
    textView->raise();
    textView->setFocus();
  }
  if (name==trUtf8("View 3D")) {
    glWidget1->show();
    glWidget1->raise();
    glWidget1->setFocus();
  }
}

//======================================================================
//Fonction du Menu de vue
//======================================================================
///Redraw the view of the graph
void  viewGl::redrawView()
{
  Observable::holdObservers();
  getProxy<LayoutProxy>(clusterTreeWidget->getSuperGraph(),"viewLayout")->notifyObservers();
  Observable::unholdObservers();
}

///Reccenter the layout of the graph
void viewGl::centerView()
{
  Observable::holdObservers();
  glWidget->getGlGraph()->init();
  getProxy<LayoutProxy>(clusterTreeWidget->getSuperGraph(),"viewLayout")->notifyObservers();
  Observable::unholdObservers();
}
///Restore the view of the graph
void viewGl::restoreView()
{
  Observable::holdObservers();
  LayoutProxy *layout=getProxy<LayoutProxy>(clusterTreeWidget->getSuperGraph(),"viewLayout");
  layout->resetBoundingBox();  
  layout->center();
  layout->notifyObservers();
  glWidget->getGlGraph()->init();

  updateSatutBar();

  Observable::unholdObservers();
}
//===========================================================
//Fonction du Menu Edit
//===========================================================
///Deselect all entries in the glGraph current selection Proxy
void viewGl::deselectALL()
{
  Observable::holdObservers();
   string tmp;
   getProxy<SelectionProxy>(clusterTreeWidget->getSuperGraph(),"viewSelection")->setAllNodeValue(false);
   getProxy<SelectionProxy>(clusterTreeWidget->getSuperGraph(),"viewSelection")->setAllEdgeValue(false);
  Observable::unholdObservers();
}

///Reverse all entries in the glGraph current selection Proxy
void viewGl::reverseSelection()
{
  Observable::holdObservers();
  string tmp;
  getProxy<SelectionProxy>(clusterTreeWidget->getSuperGraph(),"viewSelection")->reverse();
  Observable::unholdObservers();
}

///Unhide all the nodes which are see by glGraph
void viewGl::newView() {
  bool ok = FALSE;
  string tmp;
  bool verifGraph=true;
  SelectionProxy *sel1 = getProxy<SelectionProxy>(clusterTreeWidget->getSuperGraph(),"viewSelection");
  Iterator<edge>*itE = clusterTreeWidget->getSuperGraph()->getEdges();
  Observable::holdObservers();
  while (itE->hasNext()) {
    edge ite = itE->next();
    if (sel1->getEdgeValue(ite)) {
      if (!sel1->getNodeValue(clusterTreeWidget->getSuperGraph()->source(ite))) {sel1->setNodeValue(clusterTreeWidget->getSuperGraph()->source(ite),true);verifGraph=false;}
      if (!sel1->getNodeValue(clusterTreeWidget->getSuperGraph()->target(ite))) {sel1->setNodeValue(clusterTreeWidget->getSuperGraph()->target(ite),true);verifGraph=false;}
    }
  }
  delete itE;
  Observable::unholdObservers();
  
  if(!verifGraph) 
    QMessageBox::critical( 0, trUtf8("Tulip Warning") ,trUtf8("The selection wasn't a graph, missing nodes have been added"));
  QString text = QInputDialog::getText( trUtf8("View building") ,  trUtf8("Please enter view name") , QLineEdit::Normal,QString::null, &ok, this );
  if (ok && !text.isEmpty()) {
    sel1 = getProxy<SelectionProxy>(clusterTreeWidget->getSuperGraph(),"viewSelection");
    clusterTreeWidget->getSuperGraph()->addView(text.latin1(), sel1);
    clusterTreeWidget->update();
  }
  else if (ok) {
    sel1 = getProxy<SelectionProxy>(clusterTreeWidget->getSuperGraph(),"viewSelection");
    clusterTreeWidget->getSuperGraph()->addView(trUtf8("new view").latin1(),sel1);
    clusterTreeWidget->update();
  }
}
//==============================================================
void viewGl::reverseSelectedEdgeDirection()
{
  Observable::holdObservers();
  getProxy<SelectionProxy>(clusterTreeWidget->getSuperGraph(),"viewSelection")->reverseEdgeDirection();  
  Observable::unholdObservers();
}

void viewGl::superGraphAboutToBeRemoved(SuperGraph *sg) {
    glWidget->setSuperGraph(_superGraph);
    propD->setSuperGraph(_superGraph);
    nodeProperties->setSuperGraph(_superGraph);
}
