;;; ttcn3.el --- a major mode for editing TTCN-3 core language files

;; Copyright (C) 2000, 2001 W. Borgert <debacle@debian.org>

;; Author:     2000 W. Borgert <debacle@debian.org>
;; Maintainer: W. Borgert <debacle@debian.org>
;; Created:    2000-03-26
;; Version:    $Id: ttcn3.el,v 1.11 2004/01/21 20:26:57 debacle Exp $
;; Keywords:   TTCN, languages, ASN.1

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or (at
;; your option) any later version.

;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

(require 'cc-mode)			; ttcn-3-mode inherits from cc-mode
(require 'cc-langs)			; e.g. c-make-inherited-keymap
(require 'compile)			; and this for compile-interal,
(require 'easymenu)			; and uses easymenu,
(require 'font-lock)			; font-lock,
(require 'imenu)			; and imenu

(defconst c-TTCN3-conditional-key "do\\|else\\|for\\|if\\|while")
(defconst c-TTCN3-comment-start-regexp "/\\([*][*]?\\)")
(defconst c-TTCN3-defun-prompt-regexp "\\<function\\>")
(defvar c-ttcn3-menu nil)

(defvar ttcn3-mode-abbrev-table nil
  "Abbreviation table used in TTCN-3 buffers.")
(define-abbrev-table 'ttcn3-mode-abbrev-table ())

(defvar ttcn3-mode-map ()
  "Keymap used in TTCN-3 buffers.")
(if ttcn3-mode-map
    nil
  (setq ttcn3-mode-map (c-make-inherited-keymap))
  ;; add bindings which are only useful for TTCN-3
  )

(defvar ttcn3-mode-syntax-table nil
  "Syntax table used in TTCN-3 buffers.")
(if ttcn3-mode-syntax-table
    ()
  (setq ttcn3-mode-syntax-table (make-syntax-table))
  (c-populate-syntax-table ttcn3-mode-syntax-table))
(modify-syntax-entry ?_ "w" ttcn3-mode-syntax-table)

(easy-menu-define c-ttcn3-menu ttcn3-mode-map "TTCN-3 Mode Commands"
		  (c-mode-menu "TTCN-3"))

(defvar ttcn3-imenu-generic-expression nil
  "Imenu generic expression for TTCN-3 mode.  See `imenu-generic-expression'.")
(setq ttcn3-imenu-generic-expression
      '(("Constants"
         "^[ \t]*\\(external[ \t]+\\)?\\<const\\>[ \t]+\\sw+[ \t]+\\(\\sw+\\)"
         2)
        ("Module Parameters"
         "^[ \t]*\\<modulepar\\>[ \t]+\\sw+[ \t]+\\(\\sw+\\)"
         1)
        ("Variables"
	 "^[ \t]*\\<var\\>[ \t]+\\(\\(record\\|set\\)[ \t]+of[ \t]+\\)?\\sw+\\sw+[ \t]+\\(\\sw+\\)"
	 3)
	("Timers"
	 "^[ \t]*\\<timer\\>[ \t]+\\(\\sw+\\)" 1)
	("Templates"
	 "^[ \t]*\\<template\\>[ \t]+\\sw+\[ \t]+\\(\\sw+\\)" 1)
	("Types"
	 "^[ \t]*\\<type\\>[ \t]+\\(\\(record\\|set\\)[ \t]+of[ \t]+\\)?\\sw+\\sw+[ \t]+\\(\\sw+\\)"
	 3)
	("Named Alts"
         "^[ \t]*\\<named alt\\>[ \t]+\\(\\sw+\\)" 1)
        ("Altsteps"
         "^[ \t]*\\<altstep\\>[ \t]+\\(\\sw+\\)" 1)
        ("Functions"
	 "^[ \t]*\\(external[ \t]+\\)?\\<function\\>[ \t]+\\(\\sw+\\)"
	 2)
	("Test Cases"
	 "^[ \t]*\\<testcase\\>[ \t]+\\(\\sw+\\)" 1)
	("Groups"
	 "^[ \t]*\\<group\\>[ \t]+\\(\\sw+\\)" 1)
	("Modules"
	 "^[ \t]*\\<module\\>[ \t]+\\(\\sw+\\)" 1)))

(defvar ttcn3-font-lock-keywords nil
  "Expressions to highlight in TTCN-3 mode.")

; Different Emacsen - different font-lock-faces!

; GNU Emacs 20.7 has: builtin comment constant function-name keyword
;   reference string type variable-name warning

; GNU Emacs 21.0 has: builtin comment constant doc function-name
;   keyword reference string type variable-name warning

; XEmacs 21.1 has: comment doc-string function-name keyword
;   preprocessor reference string type variable-name

; Therefore, some aliases:
(if (and (not (boundp 'font-lock-builtin-face))
	 (boundp 'font-lock-doc-string-face))
    (defun ttcn3-builtin-face ()
      "builtin face for XEmacs"
      font-lock-doc-string-face)
  (defun ttcn3-builtin-face ()
    "builtin face for GNU Emacs"
    font-lock-builtin-face))
(if (and (not (boundp 'font-lock-constant-face))
	 (boundp 'font-lock-preprocessor-face))
    (defun ttcn3-constant-face ()
      "constant face for XEmacs"
      font-lock-preprocessor-face)
  (defun ttcn3-constant-face ()
    "constant face for GNU Emacs"
    font-lock-constant-face))

(setq ttcn3-font-lock-keywords
      (eval-when-compile
	(list
	 ;; TTCN-3 functions, modules, and testcases
	 (list (concat
                "\\<\\(" "function" "\\|" "group" "\\|" "language"
                "\\|" "module" "\\|" "named alt" "\\|" "altstep"
                "\\|" "testcase"
                "\\)\\>" "[ \t]+\\(\\sw+\\)?")
               '(1 font-lock-keyword-face)
	       '(2 font-lock-function-name-face nil t))
	 ;; TTCN-3 keywords
	 (list
	  (concat
	   "\\<"
	   (regexp-opt
	    '("action" "activate" "all" "alt" "and" "and4b" "any"
              "call" "catch" "check" "clear" "connect" "const"
              "control" "create" "deactivate" "disconnect" "display"
              "do" "done" "else" "encode" "error" "except" "exception"
              "execute" "expand" "extension" "external" "fail" "false"
              "for" "from" "get" "getcall" "getreply" "getverdict" "goto" "if"
              "ifpresent" "import" "in" "inconc" "infinity" "inout"
              "interleave" "label" "length" "log" "map" "match"
              "message" "mixed" "mod" "modifies" "modulepar" "mtc" "none"
              "nonrecursive" "not" "not4b" "nowait" "null" "omit"
              "optional" "or" "or4b" "out" "param" "pass" "pattern"
              "procedure" "raise" "read" "receive" "rem" "repeat"
              "reply" "return" "running" "runs on" "self" "send"
              "sender" "setverdict" "signature" "start" "stop" "sut.action"
              "system" "template" "timeout" "timer" "to" "trigger"
              "true" "type" "unmap" "value" "valueof" "var"
              "verdict.get" "verdict.set" "while" "with" "xor"
              "xor4b") t) "\\>")
	  '(1 font-lock-keyword-face))
	 ;; TTCN-3 predefined (built-in) functions
	 (list
	  (concat
	   "\\<"
	   (regexp-opt
	    '("bit2int" "char2int" "hex2int" "int2bit" "int2char"
	      "int2hex" "int2oct" "int2str" "int2unichar" "ischosen"
	      "ispresent" "lengthof" "oct2int" "sizeof" "str2int"
	      "unichar2int") t) "\\>")
	  '(1 (ttcn3-builtin-face)))
	 ;; TTCN-3 types
	 (list
	  (concat
	   "\\<"
	   (regexp-opt
	    '("address" "anytype" "bitstring" "boolean" "char" "charstring"
	      "component" "enumerated" "float" "hexstring" "integer"
	      "objid" "octetstring" "port" "record" "record of" "set"
	      "set of" "union" "universal charstring" "verdicttype") t)
	      "\\>")
	  '(1 font-lock-type-face))
	 ;; user-defined types
	 (list (concat "\\<\\(type\\)\\>[ \t]+\\(\\(record\\|set\\)[ \t]+"
		       "of[ \t]+\\)?\\(\\sw+\\)[ \t]+\\(\\sw+\\)")
	       '(1 font-lock-keyword-face)
	       '(5 font-lock-type-face nil t))
	 ;; TTCN-3 constants
	 (list (concat "\\<\\(const\\)\\>"
		       "[ \t]+\\(\\sw+\\)?[ \t]+\\(\\sw+\\)?")
	       '(1 font-lock-keyword-face)
	       '(2 font-lock-type-face)
	       '(3 (ttcn3-constant-face) nil t))
	 ;; TTCN-3 templates, and variables
	 (list (concat "\\<\\(template\\|var\\)\\>[ \t]+"
		       "\\(\\(record\\|set\\)[ \t]+of[ \t]+\\)?"
		       "\\(\\sw+\\)[ \t]+\\(\\sw+\\)")
	       '(1 font-lock-keyword-face)
	       '(4 font-lock-type-face)
	       '(5 font-lock-variable-name-face nil t))
	 ;; ASN.1 keywords, not to be used as identifiers in TTCN-3
	 (list
	  (concat
	   "\\<"
	   (regexp-opt
	    '("ABSENT" "ABSTRACT-SYNTAX" "ALL" "APPLICATION"
	      "AUTOMATIC" "BEGIN" "BIT" "BMPSTRING" "BOOLEAN" "BY"
	      "CHARACTER" "CHOICE" "CLASS" "COMPONENT" "COMPONENTS"
	      "CONSTRAINED" "DEFAULT" "DEFINITIONS" "EMBEDDED" "END"
	      "ENUMERATED" "EXCEPT" "EXPLICIT" "EXPORTS" "EXTERNAL"
	      "FALSE" "FROM" "GeneralizedTime" "GeneralString"
	      "IA5String" "IDENTIFIER" "IMPLICIT" "IMPORTS" "INCLUDES"
	      "INSTANCE" "INTEGER" "INTERSECTION" "ISO646String" "MAX"
	      "MIN" "MINUS-INFINITY" "NULL" "NumericString" "OBJECT"
	      "ObjectDescriptor" "OCTET" "OF" "OPTIONAL" "PDV"
	      "PLUS-INFINITY" "PRESENT" "PrintableString" "PRIVATE"
	      "REAL" "SEQUENCE" "SET" "SIZE" "STRING" "SYNTAX"
	      "T61String" "TAGS" "TeletexString" "TRUE"
	      "TYPE-IDENTIFIER" "UNION" "UNIQUE" "UNIVERSAL"
	      "UniversalString" "UTCTime" "VideotexString"
	      "VisibleString" "WITH") t) "\\>")
	  '(1 font-lock-reference-face)))))

;; Handle TTCN-3 alternatives simlilar to switch/case in C
(make-variable-buffer-local 'c-switch-label-key)
(defconst c-TTCN3-alternative-key "\\(\\[.*\\]\\)")

;; Support for the TTCN3Parser and ttthreeparser
(defvar ttcn3-parse-command "TTCN3Parser"
  "The default command for \\[ttcn3-parse], e.g. TTCN3Parser or
ttthreeparser.")

(defvar ttcn3-parse-history '("TTCN3Parser" "ttthreeparser"))

(defun ttcn3-parse (command-args)
  "Run a TTCN-3 parser, with user-specified args, and collect output
in a buffer.  While the parser runs asynchronously, you can use
\\[next-error] (M-x next-error), or
\\<compilation-minor-mode-map>\\[compile-goto-error] in the parser
output buffer, to go to the lines where the parser found problems.

This command uses a special history list for its arguments, so you can
easily repeat a parse."
  (interactive
   (let ((ttcn3-parse-default
	  (or (car ttcn3-parse-history) ttcn3-parse-command)))
     (list (read-from-minibuffer
	    "Run the TTCN-3 parser (like this): "
	    (or ttcn3-parse-default ttcn3-parse-command)
	    nil nil 'ttcn3-parse-history))))
  (let* ((buf (compile-internal
	       (concat command-args " " (buffer-file-name))
	       "No more errors"
	       "TTCN-3 parse")))))

;;;###autoload
(defun ttcn-3-mode ()
  "Major mode for editing TTCN-3 core language.  Reference: rev. 5 of
the BNF with changes until 2001-10.

This mode is based on `CC Mode'.  Please look for further information
in the info documenation for that mode."
  (interactive)
  (c-initialize-cc-mode)
  (kill-all-local-variables)
  (set-syntax-table ttcn3-mode-syntax-table)
  (setq major-mode 'ttcn-3-mode
 	mode-name "TTCN-3"
 	local-abbrev-table ttcn3-mode-abbrev-table)
  (use-local-map ttcn3-mode-map)
  (c-common-init)
  (setq comment-start "/* "
 	comment-end   " */"
 	c-conditional-key c-TTCN3-conditional-key
 	c-comment-start-regexp c-TTCN3-comment-start-regexp
	c-method-key nil
	c-switch-label-key c-TTCN3-alternative-key
 	c-baseclass-key nil
	c-recognize-knr-p nil
	defun-prompt-regexp c-TTCN3-defun-prompt-regexp
	imenu-generic-expression ttcn3-imenu-generic-expression
	imenu-case-fold-search nil)
  (imenu-add-to-menubar "Module-Index")
  (c-set-offset 'substatement-open 0)
  (c-set-offset 'statement-cont 0)
  (run-hooks 'c-mode-common-hook)
  (run-hooks 'ttcn3-mode-hook)
  (set (make-local-variable 'font-lock-defaults)
       '(ttcn3-font-lock-keywords nil nil ((?_ . "w"))))
  (c-update-modeline))

(provide 'ttcn3)

;;; ttcn3.el ends here
