// cpu_main.c - TMS9900 Emulator - Main
//
// Copyright (c) 2001, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "ti99/defs.h"
#include "emu/socket.h"

extern SOCKET *dbg_File;

#define INC_PC PC += 2
#define JUMP   PC += ((int8)opCode << 1) // Implies Format II field

#define F8I src = ti99_ReadW(PC); INC_PC

void ti99_BuildCPU(register TI99_CPU *ti99cpu, uint32 mode)
{
	uint32 idx;

	extern TI99_INST ti99_Inst[];

	// Allocate memory space for opcode table.
	if (ti99cpu->Opcodes == NULL)
		ti99cpu->Opcodes = malloc(65536 * sizeof(uint32 *));

	// Initialize default illegal instructions
	for (idx = 0; idx < 65536; idx++)
		ti99cpu->Opcodes[idx] = ti99_opIllegal;

	// Now build opcode table for execution.
	for (idx = 0; ti99_Inst[idx].Name; idx++) {
		uint16 opidx = ti99_Inst[idx].opCode;
		uint16 opcnt = ~ti99_Inst[idx].opMask + 1;

		if (ti99_Inst[idx].Execute == NULL)
			continue;

		while(opcnt--)
			ti99cpu->Opcodes[opidx++] = ti99_Inst[idx].Execute;
	}
}

int ti99_Execute(MAP_DEVICE *map)
{
	TI99_SYSTEM       *ti99sys = (TI99_SYSTEM *)map->Device;
	register TI99_CPU *ti99cpu = ti99sys->Processor;
	register uint16   opCode;

	// Initialization
	IPS = 0;

	ts10_SetAlarm(ts10_TickRealTimer);
	ts10_StartTimer();
	EnableTimer;

	for (;;) {

#ifdef DEBUG
		if (dbg_Check(DBG_TRACE)) {
			uint32 addr = PC;
			d99_Disasm(TI99, dbg_File, &addr, 0);
		}
#endif /* DEBUG */

		// Fetch next instruction
		faultPC = PC;
		IR      = ReadW(PC);
		INC_PC;

		IPS++;  // Count instruction for meter

		// Execute emulated instruction
		ti99cpu->Opcodes[IR](ti99cpu);
	}
}
