// panel.c - Panel Control routines (screen area)
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS10 Emulator.
// See ReadMe for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <termios.h>
#include <unistd.h>
#include "emu/defs.h"
#include "emu/socket.h"

#define Stdin  STDIN_FILENO
#define Stdout STDOUT_FILENO
#define Stderr STDERR_FILENO

int    ts10_pkbBuffer;
char   ts10_kbBuffer[256];
char   *ts10_Prompt;
SOCKET *ts10_Stdin;
SOCKET *ts10_Stdout;
SOCKET *ts10_Stderr;

void panel_Input(SOCKET *, char *, int);
void panel_Eof(SOCKET *, int, int);

// Initialize Panel Control Handler
void InitControlPanel(void)
{
	struct termios kbAttr;

	ts10_Prompt    = "TS10> ";
	ts10_pkbBuffer = 0;

	// Set up all standard I/O on socket slots.
	if (ts10_Stdin = sock_Open("<stdin>", Stdin, NET_STDIO)) {
		ts10_Stdin->Eof     = panel_Eof;
		ts10_Stdin->Process = panel_Input;

		// Turn edit and echo mode off for keyboard.
		tcgetattr(Stdin, &kbAttr);
		kbAttr.c_lflag &= ~(ICANON | ECHO);
		tcsetattr(Stdin, TCSAFLUSH, &kbAttr);
	}

	ts10_Stdout = sock_Open("<stdout>", Stdout, NET_STDIO);
	ts10_Stderr = sock_Open("<stderr>", Stderr, NET_STDIO);
}

void panel_Cleanup(void)
{
	struct termios kbAttr;

	tcgetattr(Stdin, &kbAttr);
	kbAttr.c_lflag |= (ICANON | ECHO);
	tcsetattr(Stdin, TCSANOW, &kbAttr);
}

void panel_Eof(SOCKET *Socket, int nBytes, int nError)
{
	// Do nothing
}

void panel_Input(SOCKET *Socket, char *inBuffer, int lenBuffer)
{
	int idx;

	for (idx = 0; idx < lenBuffer; idx++) {
		switch(inBuffer[idx]) {
			case '\b':   // Backspace key (ASCII BS)
			case '\x7F': // Delete key (ASCII DEL)
				if (ts10_pkbBuffer > 0) {
					ts10_pkbBuffer--;
					sock_Send(ts10_Stdout->idSocket, "\b \b", 3);
				}
				break;

			case '\n': // Linefeed Key (ASCII LF)
				sock_Send(ts10_Stdout->idSocket, "\r\n", 2);
				ts10_kbBuffer[ts10_pkbBuffer] = '\0';
				console_Prompt(ts10_kbBuffer);
				ts10_pkbBuffer = 0;
				sock_Send(ts10_Stdout->idSocket, ts10_Prompt, 0);
				break;

			case '?': // List current commands
				if (ts10_pkbBuffer == 0) {
					ListAllCommands();
					sock_Send(ts10_Stdout->idSocket, ts10_Prompt, 0);
					break;
				}

			default: // Append any character to keyboard buffer.
				if (ts10_pkbBuffer < 255) {
					ts10_kbBuffer[ts10_pkbBuffer++] = inBuffer[idx];
					sock_Send(ts10_Stdout->idSocket, &inBuffer[idx], 1);
				}
		}
	}
}

