
/*
 * Licensed Materials - Property of IBM
 *
 * trousers - An open source TCG Software Stack
 *
 * (C) Copyright International Business Machines Corp. 2004-2006
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <syslog.h>
#include <string.h>
#include <netdb.h>

#include "trousers/tss.h"
#include "spi_internal_types.h"
#include "tcs_internal_types.h"
#include "tcs_tsp.h"
#include "tcs_utils.h"
#include "tcs_int_literals.h"
#include "capabilities.h"
#include "tcslog.h"
#include "tcsd_wrap.h"
#include "tcsd.h"
#include "tcs_utils.h"

pthread_mutex_t tcsp_lock = PTHREAD_MUTEX_INITIALIZER;

void
LoadBlob_Auth_Special(UINT64 *offset, BYTE *blob, TPM_AUTH *auth)
{
	LoadBlob(offset, TCPA_SHA1BASED_NONCE_LEN, blob, auth->NonceEven.nonce, NULL);
	LoadBlob_BOOL(offset, auth->fContinueAuthSession, blob, NULL);
	LoadBlob(offset, TCPA_SHA1BASED_NONCE_LEN, blob, (BYTE *)&auth->HMAC, NULL);
}

void
UnloadBlob_Auth_Special(UINT64 *offset, BYTE *blob, TPM_AUTH *auth)
{
	UnloadBlob_UINT32(offset, &auth->AuthHandle, blob, NULL);
	UnloadBlob(offset, TCPA_SHA1BASED_NONCE_LEN, blob, auth->NonceOdd.nonce, NULL);
	UnloadBlob_BOOL(offset, &auth->fContinueAuthSession, blob, NULL);
	UnloadBlob(offset, TCPA_SHA1BASED_NONCE_LEN, blob, (BYTE *)&auth->HMAC, NULL);
}

void
LoadBlob_KM_KEYINFO(UINT64 *offset, BYTE *blob, TSS_KM_KEYINFO *info)
{
	LoadBlob_VERSION(offset, blob, (TCPA_VERSION *)&(info->versionInfo));
	LoadBlob_UUID(offset, blob, info->keyUUID);
	LoadBlob_UUID(offset, blob, info->parentKeyUUID);
	LoadBlob_BYTE(offset, info->bAuthDataUsage, blob, NULL);
	LoadBlob_BOOL(offset, info->fIsLoaded, blob, NULL);
	LoadBlob_UINT32(offset, info->ulVendorDataLength, blob, NULL);
	LoadBlob(offset, info->ulVendorDataLength, blob, info->rgbVendorData, NULL);
}

void
UnloadBlob_KM_KEYINFO(UINT64 *offset, BYTE *blob, TSS_KM_KEYINFO *info)
{
	UnloadBlob_VERSION(offset, blob, (TCPA_VERSION *)&(info->versionInfo));
	UnloadBlob_UUID(offset, blob, &info->keyUUID);
	UnloadBlob_UUID(offset, blob, &info->parentKeyUUID);
	UnloadBlob_BYTE(offset, blob, &info->bAuthDataUsage, NULL);
	UnloadBlob_BOOL(offset, &info->fIsLoaded, blob, NULL);
	UnloadBlob_UINT32(offset, &info->ulVendorDataLength, blob, NULL);
	UnloadBlob(offset, info->ulVendorDataLength, info->rgbVendorData, blob, NULL);
}

void
LoadBlob_LOADKEY_INFO(UINT64 *offset, BYTE *blob, TCS_LOADKEY_INFO *info)
{
	LoadBlob_UUID(offset, blob, info->keyUUID);
	LoadBlob_UUID(offset, blob, info->parentKeyUUID);
	LoadBlob(offset, TCPA_DIGEST_SIZE, blob, info->paramDigest.digest, NULL);
	LoadBlob_UINT32(offset, info->authData.AuthHandle, blob, NULL);
	LoadBlob(offset, TCPA_NONCE_SIZE, blob, info->authData.NonceOdd.nonce, NULL);
	LoadBlob(offset, TCPA_NONCE_SIZE, blob, info->authData.NonceEven.nonce, NULL);
	LoadBlob_BOOL(offset, info->authData.fContinueAuthSession, blob, NULL);
	LoadBlob(offset, TCPA_AUTHDATA_SIZE, blob, (BYTE *)&info->authData.HMAC, NULL);
}

void
UnloadBlob_LOADKEY_INFO(UINT64 *offset, BYTE *blob, TCS_LOADKEY_INFO *info)
{
	UnloadBlob_UUID(offset, blob, &info->keyUUID);
	UnloadBlob_UUID(offset, blob, &info->parentKeyUUID);
	UnloadBlob(offset, TCPA_DIGEST_SIZE, blob, info->paramDigest.digest, NULL);
	UnloadBlob_UINT32(offset, &info->authData.AuthHandle, blob, NULL);
	UnloadBlob(offset, TCPA_NONCE_SIZE, blob, (BYTE *)&info->authData.NonceOdd.nonce, NULL);
	UnloadBlob(offset, TCPA_NONCE_SIZE, blob, (BYTE *)&info->authData.NonceEven.nonce, NULL);
	UnloadBlob_BOOL(offset, &info->authData.fContinueAuthSession, blob, NULL);
	UnloadBlob(offset, TCPA_DIGEST_SIZE, blob, (BYTE *)&info->authData.HMAC, NULL);
}

void
LoadBlob_PCR_EVENT(UINT64 *offset, BYTE *blob, TSS_PCR_EVENT *event)
{
	LoadBlob_VERSION(offset, blob, (TCPA_VERSION *)&(event->versionInfo));
	LoadBlob_UINT32(offset, event->ulPcrIndex, blob, NULL);
	LoadBlob_UINT32(offset, event->eventType, blob, NULL);

	LoadBlob_UINT32(offset, event->ulPcrValueLength, blob, NULL);
	if (event->ulPcrValueLength > 0)
		LoadBlob(offset, event->ulPcrValueLength, blob, event->rgbPcrValue, NULL);

	LoadBlob_UINT32(offset, event->ulEventLength, blob, NULL);
	if (event->ulEventLength > 0)
		LoadBlob(offset, event->ulEventLength, blob, event->rgbEvent, NULL);

}

TSS_RESULT
UnloadBlob_PCR_EVENT(UINT64 *offset, BYTE *blob, TSS_PCR_EVENT *event)
{
	UnloadBlob_VERSION(offset, blob, (TCPA_VERSION *)&(event->versionInfo));
	UnloadBlob_UINT32(offset, &event->ulPcrIndex, blob, NULL);
	UnloadBlob_UINT32(offset, &event->eventType, blob, NULL);

	UnloadBlob_UINT32(offset, &event->ulPcrValueLength, blob, NULL);
	if (event->ulPcrValueLength > 0) {
		event->rgbPcrValue = malloc(event->ulPcrValueLength);
		if (event->rgbPcrValue == NULL) {
			LogError("malloc of %d bytes failed.", event->ulPcrValueLength);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}

		UnloadBlob(offset, event->ulPcrValueLength, blob, event->rgbPcrValue, NULL);
	} else {
		event->rgbPcrValue = NULL;
	}

	UnloadBlob_UINT32(offset, &event->ulEventLength, blob, NULL);
	if (event->ulEventLength > 0) {
		event->rgbEvent = malloc(event->ulEventLength);
		if (event->rgbEvent == NULL) {
			LogError("malloc of %d bytes failed.", event->ulEventLength);
			free(event->rgbPcrValue);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}

		UnloadBlob(offset, event->ulEventLength, blob, event->rgbEvent, NULL);
	} else {
		event->rgbEvent = NULL;
	}

	return TSS_SUCCESS;
}

void
initData(struct tcsd_comm_data *comm, int parm_count)
{
	/* min packet size should be the size of the header */
	memset(&comm->hdr, 0, sizeof(struct tcsd_packet_hdr));
	comm->hdr.packet_size = sizeof(struct tcsd_packet_hdr);
	if (parm_count > 0) {
		comm->hdr.type_offset = sizeof(struct tcsd_packet_hdr);
		comm->hdr.parm_offset = comm->hdr.type_offset +
					(sizeof(TCSD_PACKET_TYPE) * parm_count);
		comm->hdr.packet_size = comm->hdr.parm_offset;
	}

	memset(comm->buf, 0, comm->buf_size);
}

int
loadData(UINT64 *offset, TCSD_PACKET_TYPE data_type, void *data, int data_size, BYTE *blob)
{
	switch (data_type) {
	case TCSD_PACKET_TYPE_BYTE:
		LoadBlob_BYTE(offset, *((BYTE *) (data)), blob, NULL);
		break;
	case TCSD_PACKET_TYPE_BOOL:
		LoadBlob_BOOL(offset, *((TSS_BOOL *) (data)), blob, NULL);
		break;
	case TCSD_PACKET_TYPE_UINT16:
		LoadBlob_UINT16(offset, *((UINT16 *) (data)), blob, NULL);
		break;
	case TCSD_PACKET_TYPE_UINT32:
		LoadBlob_UINT32(offset, *((UINT32 *) (data)), blob, NULL);
		break;
	case TCSD_PACKET_TYPE_PBYTE:
		LoadBlob(offset, data_size, blob, data, NULL);
		break;
	case TCSD_PACKET_TYPE_NONCE:
		LoadBlob(offset, sizeof(TCPA_NONCE), blob, ((TCPA_NONCE *)data)->nonce, NULL);
		break;
	case TCSD_PACKET_TYPE_DIGEST:
		LoadBlob(offset, sizeof(TCPA_DIGEST), blob, ((TCPA_DIGEST *)data)->digest, NULL);
		break;
	case TCSD_PACKET_TYPE_AUTH:
		LoadBlob_Auth_Special(offset, blob, ((TPM_AUTH *)data));
		break;
	case TCSD_PACKET_TYPE_UUID:
		LoadBlob_UUID(offset, blob, *((TSS_UUID *)data));
		break;
	case TCSD_PACKET_TYPE_ENCAUTH:
		LoadBlob(offset, sizeof(TCPA_ENCAUTH), blob, ((TCPA_ENCAUTH *)data)->authdata, NULL);
		break;
	case TCSD_PACKET_TYPE_VERSION:
		LoadBlob_VERSION(offset, blob, ((TCPA_VERSION *)data));
		break;
	case TCSD_PACKET_TYPE_KM_KEYINFO:
		LoadBlob_KM_KEYINFO(offset, blob, ((TSS_KM_KEYINFO *)data));
		break;
	case TCSD_PACKET_TYPE_LOADKEY_INFO:
		LoadBlob_LOADKEY_INFO(offset, blob, ((TCS_LOADKEY_INFO *)data));
		break;
	case TCSD_PACKET_TYPE_PCR_EVENT:
		LoadBlob_PCR_EVENT(offset, blob, ((TSS_PCR_EVENT *)data));
		break;
	default:
		LogError("TCSD packet type unknown! (0x%x)", data_type & 0xff);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	return TSS_SUCCESS;
}

int
setData(TCSD_PACKET_TYPE dataType, int index, void *theData, int theDataSize, struct tcsd_comm_data *comm)
{
	UINT64 old_offset, offset;
	TSS_RESULT result;
	TCSD_PACKET_TYPE *type;

	/* Calculate the size of the area needed (use NULL for blob address) */
	offset = 0;
	if ((result = loadData(&offset, dataType, theData, theDataSize, NULL)) != TSS_SUCCESS)
		return result;
	if (((int)comm->hdr.packet_size + (int)offset) < 0) {
		LogError("Too much data to be transmitted!");
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (((int)comm->hdr.packet_size + (int)offset) > comm->buf_size) {
		/* reallocate the buffer */
		BYTE *buffer;
		int buffer_size = comm->hdr.packet_size + offset;
#if 0
		static int realloc_scalar = 1;

		realloc_scalar *= TCSD_COMMBUF_REALLOC_SCALAR;

		if (((int)offset * realloc_scalar) + buffer_size < 0)
			buffer_size = INT_MAX;
		else
			buffer_size += (int)offset * realloc_scalar;

		LogDebug("Increasing communication buffer by %d bytes.",
			 (int)offset * realloc_scalar);
#endif
		LogDebug("Increasing communication buffer to %d bytes.", buffer_size);
		buffer = realloc(comm->buf, buffer_size);
		if (buffer == NULL) {
			LogError("realloc of %d bytes failed.", buffer_size);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		comm->buf_size = buffer_size;
		comm->buf = buffer;
	}

	offset = old_offset = comm->hdr.parm_offset + comm->hdr.parm_size;
	if ((result = loadData(&offset, dataType, theData, theDataSize, comm->buf)) != TSS_SUCCESS)
		return result;
	type = (TCSD_PACKET_TYPE *)(comm->buf + comm->hdr.type_offset) + index;
	*type = dataType;
	comm->hdr.type_size += sizeof(TCSD_PACKET_TYPE);
	comm->hdr.parm_size += (offset - old_offset);

	comm->hdr.packet_size = offset;
	comm->hdr.num_parms++;

	return TSS_SUCCESS;
}

UINT32
getData(TCSD_PACKET_TYPE dataType, int index, void *theData, int theDataSize, struct tcsd_comm_data *comm)
{
	TSS_RESULT result;
	UINT64 old_offset, offset;
	TCSD_PACKET_TYPE *type = (TCSD_PACKET_TYPE *)(comm->buf + comm->hdr.type_offset) +
				 index;

	if ((UINT32)index >= comm->hdr.num_parms || dataType != *type) {
		LogDebug("Data type of TCS packet element %d doesn't match.", index);
		return TSS_TCP_RPC_BAD_PACKET_TYPE;
	}
	old_offset = offset = comm->hdr.parm_offset;
	switch (dataType) {
	case TCSD_PACKET_TYPE_BYTE:
		UnloadBlob_BYTE(&offset, (BYTE *) (theData), comm->buf, NULL);
		break;
	case TCSD_PACKET_TYPE_BOOL:
		UnloadBlob_BOOL(&offset, (TSS_BOOL *) (theData), comm->buf, NULL);
		break;
	case TCSD_PACKET_TYPE_UINT16:
		UnloadBlob_UINT16(&offset, (UINT16 *) (theData), comm->buf, NULL);
		break;
	case TCSD_PACKET_TYPE_UINT32:
		UnloadBlob_UINT32(&offset, (UINT32 *) (theData), comm->buf, NULL);
		break;
	case TCSD_PACKET_TYPE_PBYTE:
		UnloadBlob(&offset, theDataSize, comm->buf, theData, NULL);
		break;
	case TCSD_PACKET_TYPE_NONCE:
		UnloadBlob(&offset, sizeof(TCPA_NONCE), comm->buf,
			   ((TCPA_NONCE *) (theData))->nonce, NULL);
		break;
	case TCSD_PACKET_TYPE_DIGEST:
		UnloadBlob(&offset, sizeof(TCPA_DIGEST), comm->buf,
			   ((TCPA_DIGEST *) (theData))->digest, NULL);
		break;
	case TCSD_PACKET_TYPE_AUTH:
		UnloadBlob_Auth_Special(&offset, comm->buf, ((TPM_AUTH *) theData));
		break;
	case TCSD_PACKET_TYPE_UUID:
		UnloadBlob_UUID(&offset, comm->buf, (TSS_UUID *) theData);
		break;
	case TCSD_PACKET_TYPE_ENCAUTH:
		UnloadBlob(&offset, sizeof(TCPA_ENCAUTH), comm->buf,
			   ((TCPA_ENCAUTH *) theData)->authdata, NULL);
		break;
	case TCSD_PACKET_TYPE_VERSION:
		UnloadBlob_VERSION(&offset, comm->buf, ((TCPA_VERSION *) theData));
		break;
	case TCSD_PACKET_TYPE_KM_KEYINFO:
		UnloadBlob_KM_KEYINFO(&offset, comm->buf, ((TSS_KM_KEYINFO*)theData));
		break;
	case TCSD_PACKET_TYPE_LOADKEY_INFO:
		UnloadBlob_LOADKEY_INFO(&offset, comm->buf, ((TCS_LOADKEY_INFO *)theData));
		break;
	case TCSD_PACKET_TYPE_PCR_EVENT:
		if ((result = UnloadBlob_PCR_EVENT(&offset, comm->buf, ((TSS_PCR_EVENT *)theData))))
			return result;
		break;
	default:
		LogError("TCSD packet type unknown! (0x%x)", dataType & 0xff);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	comm->hdr.parm_offset = offset;
	comm->hdr.parm_size -= (offset - old_offset);

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_Error(struct tcsd_thread_data *data)
{
	LogError("%s reached.", __FUNCTION__);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = TCSERR(TSS_E_FAIL);

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_OpenContext(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;

	LogDebug("thread %x servicing a %s request", (UINT32)pthread_self(), __FUNCTION__);

	result = TCS_OpenContext_Internal(&hContext);
	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);

		/* Set the context in the thread's object. Later, if something goes wrong
		 * and the connection can't be closed cleanly, we'll still have a reference
		 * to what resources need to be freed. */
		data->context = hContext;
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetRandom(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	UINT32 bytesRequested;
	BYTE *randomBytes = NULL;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &bytesRequested, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_GetRandom_Internal(hContext, &bytesRequested, &randomBytes);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 2);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &bytesRequested, 0, &data->comm)) {
			free(randomBytes);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 1, randomBytes, bytesRequested, &data->comm)) {
			free(randomBytes);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(randomBytes);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_CreateEndorsementKeyPair(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCPA_NONCE antiReplay;
	UINT32 eKPtrSize;
	BYTE *eKPtr;
	UINT32 eKSize;
	BYTE* eK;
	TCPA_DIGEST checksum;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_NONCE, 1, &antiReplay, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &eKPtrSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (eKPtrSize == 0)
		eKPtr = NULL;
	else {
		eKPtr = calloc(1, eKPtrSize);
		if (eKPtr == NULL) {
			LogError("malloc of %d bytes failed.", eKPtrSize);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 3, eKPtr, eKPtrSize, &data->comm)) {
			free(eKPtr);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_CreateEndorsementKeyPair_Internal(hContext, antiReplay, eKPtrSize, eKPtr, &eKSize, &eK, &checksum);

	pthread_mutex_unlock(&tcsp_lock);

	free(eKPtr);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 3);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &eKSize, 0, &data->comm)) {
			free(eK);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 1, eK, eKSize, &data->comm)) {
			free(eK);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(eK);
		if (setData(TCSD_PACKET_TYPE_DIGEST, 2, &checksum, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
	else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_ReadPubek(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCPA_NONCE antiReplay;
	UINT32 pubEKSize;
	BYTE *pubEK;
	TCPA_DIGEST checksum;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_NONCE, 1, &antiReplay, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_ReadPubek_Internal(hContext, antiReplay, &pubEKSize, &pubEK,
				    &checksum);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 3);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &pubEKSize, 0, &data->comm)) {
			free(pubEK);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 1, pubEK, pubEKSize, &data->comm)) {
			free(pubEK);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(pubEK);
		if (setData(TCSD_PACKET_TYPE_DIGEST, 2, &checksum, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_OwnerReadPubek(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	UINT32 pubEKSize;
	BYTE *pubEK;
	TSS_RESULT result;
	TPM_AUTH auth;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_AUTH, 1, &auth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_OwnerReadPubek_Internal(hContext, &auth, &pubEKSize, &pubEK);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 3);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &auth, 0, &data->comm)) {
			free(pubEK);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_UINT32, 1, &pubEKSize, 0, &data->comm)) {
			free(pubEK);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 2, pubEK, pubEKSize, &data->comm)) {
			free(pubEK);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(pubEK);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_DisablePubekRead(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TPM_AUTH auth;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_AUTH, 1, &auth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_DisablePubekRead_Internal(hContext, &auth);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &auth, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_CloseContext(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	result = TCS_CloseContext_Internal(hContext);

	/* This will signal the thread that the connection has been closed cleanly */
	if (result == TSS_SUCCESS)
		data->context = NULL_TCS_HANDLE;

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_EvictKey(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE hKey;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &hKey, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = key_mgr_evict(hContext, hKey);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetPubkey(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE hKey;
	TPM_AUTH auth;
	TPM_AUTH *pAuth;
	UINT32 pubKeySize;
	BYTE *pubKey;
	TSS_RESULT result;
	int i;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &hKey, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	result = getData(TCSD_PACKET_TYPE_AUTH, 2, &auth, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pAuth = NULL;
	else if (result)
		return TCSERR(TSS_E_INTERNAL_ERROR);
	else
		pAuth = &auth;

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_GetPubKey_Internal(hContext, hKey, pAuth, &pubKeySize, &pubKey);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		i = 0;
		initData(&data->comm, 3);
		if (pAuth != NULL)
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pAuth, 0, &data->comm)) {
				free(pubKey);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &pubKeySize, 0, &data->comm)) {
			free(pubKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, pubKey, pubKeySize, &data->comm)) {
			free(pubKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(pubKey);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_OIAP(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_AUTHHANDLE authHandle;
	TCPA_NONCE n0;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	pthread_mutex_lock(&tcsp_lock);

	result = auth_mgr_oiap(hContext, &authHandle, &n0);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 2);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &authHandle, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		if (setData(TCSD_PACKET_TYPE_NONCE, 1, &n0, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_TerminateHandle(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_AUTHHANDLE authHandle;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &authHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_TerminateHandle_Internal(hContext, authHandle);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_Extend(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	UINT32 pcrIndex;
	TCPA_DIGEST inDigest;
	TSS_RESULT result;
	TCPA_DIGEST outDigest;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &pcrIndex, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_DIGEST, 2, &inDigest, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_Extend_Internal(hContext, pcrIndex, inDigest, &outDigest);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_DIGEST, 0, &outDigest, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_PcrRead(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	UINT32 pcrIndex;
	TCPA_DIGEST digest;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &pcrIndex, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_PcrRead_Internal(hContext, pcrIndex, &digest);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_DIGEST, 0, &digest, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetCapability(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCPA_CAPABILITY_AREA capArea;
	UINT32 subCapSize;
	BYTE *subCap;
	UINT32 respSize;
	BYTE *resp;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &capArea, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &subCapSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (subCapSize == 0)
		subCap = NULL;
	else {
		subCap = calloc(1, subCapSize);
		if (subCap == NULL) {
			LogError("malloc of %d bytes failed.", subCapSize);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		if (getData(TCSD_PACKET_TYPE_PBYTE, 3, subCap, subCapSize, &data->comm)) {
			free(subCap);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_GetCapability_Internal(hContext, capArea, subCapSize, subCap,
					&respSize, &resp);

	pthread_mutex_unlock(&tcsp_lock);

	free(subCap);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 2);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &respSize, 0, &data->comm)) {
			free(resp);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 1, resp, respSize, &data->comm)) {
			free(resp);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(resp);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetCapabilityOwner(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TPM_AUTH ownerAuth;
	TCPA_VERSION version;
	UINT32 nonVol;
	UINT32 vol;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_AUTH, 1, &ownerAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_GetCapabilityOwner_Internal(hContext, &ownerAuth, &version,
					     &nonVol, &vol);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 4);
		if (setData(TCSD_PACKET_TYPE_VERSION, 0, &version, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		if (setData(TCSD_PACKET_TYPE_UINT32, 1, &nonVol, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		if (setData(TCSD_PACKET_TYPE_UINT32, 2, &vol, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		if (setData(TCSD_PACKET_TYPE_AUTH, 3, &ownerAuth, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_TCSGetCapability(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCPA_CAPABILITY_AREA capArea;
	UINT32 subCapSize;
	BYTE *subCap;
	UINT32 respSize;
	BYTE *resp;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &capArea, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &subCapSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	subCap = calloc(1, subCapSize);
	if (subCap == NULL) {
		LogError("malloc of %d bytes failed.", subCapSize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, subCap, subCapSize, &data->comm)) {
		free(subCap);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	result = TCS_GetCapability_Internal(hContext, capArea, subCapSize, subCap,
				       &respSize, &resp);
	free(subCap);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 2);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &respSize, 0, &data->comm)) {
			free(resp);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 1, resp, respSize, &data->comm)) {
			free(resp);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(resp);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_PhysicalSetDeactivated(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_BOOL state;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_BOOL, 1, &state, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_PhysicalSetDeactivated_Internal(hContext, state);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_TakeOwnership(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	UINT16 protocolID;
	UINT32 encOwnerAuthSize;
	BYTE *encOwnerAuth;
	UINT32 encSrkAuthSize;
	BYTE *encSrkAuth;
	UINT32 srkInfoSize;
	BYTE *srkInfo;
	TPM_AUTH ownerAuth;

	UINT32 srkKeySize;
	BYTE *srkKey;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT16, 1, &protocolID, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &encOwnerAuthSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	encOwnerAuth = calloc(1, encOwnerAuthSize);
	if (encOwnerAuth == NULL) {
		LogError("malloc of %d bytes failed.", encOwnerAuthSize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, encOwnerAuth, encOwnerAuthSize, &data->comm)) {
		free(encOwnerAuth);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_UINT32, 4, &encSrkAuthSize, 0, &data->comm)) {
		free(encOwnerAuth);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	encSrkAuth = calloc(1, encSrkAuthSize);
	if (encSrkAuth == NULL) {
		LogError("malloc of %d bytes failed.", encSrkAuthSize);
		free(encOwnerAuth);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 5, encSrkAuth, encSrkAuthSize, &data->comm)) {
		free(encOwnerAuth);
		free(encSrkAuth);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_UINT32, 6, &srkInfoSize, 0, &data->comm)) {
		free(encOwnerAuth);
		free(encSrkAuth);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	srkInfo = calloc(1, srkInfoSize);
	if (srkInfo == NULL) {
		LogError("malloc of %d bytes failed.", srkInfoSize);
		free(encOwnerAuth);
		free(encSrkAuth);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 7, srkInfo, srkInfoSize, &data->comm)) {
		free(encOwnerAuth);
		free(encSrkAuth);
		free(srkInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_AUTH, 8, &ownerAuth, 0, &data->comm)) {
		free(encOwnerAuth);
		free(encSrkAuth);
		free(srkInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_TakeOwnership_Internal(hContext, protocolID, encOwnerAuthSize,
					encOwnerAuth, encSrkAuthSize,
					encSrkAuth, srkInfoSize, srkInfo,
					&ownerAuth, &srkKeySize, &srkKey);

	pthread_mutex_unlock(&tcsp_lock);

	free(encOwnerAuth);
	free(encSrkAuth);
	free(srkInfo);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 3);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &ownerAuth, 0, &data->comm)) {
			free(srkKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_UINT32, 1, &srkKeySize, 0, &data->comm)) {
			free(srkKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 2, srkKey, srkKeySize, &data->comm)) {
			free(srkKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(srkKey);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_OwnerClear(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TPM_AUTH auth;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_AUTH, 1, &auth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_OwnerClear_Internal(hContext, &auth);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &auth, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_DisableOwnerClear(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TPM_AUTH auth;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_AUTH, 1, &auth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_DisableOwnerClear_Internal(hContext, &auth);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &auth, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_ForceClear(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_ForceClear_Internal(hContext);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_DisableForceClear(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_DisableForceClear_Internal(hContext);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_PhysicalEnable(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_PhysicalEnable_Internal(hContext);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_RegisterKey(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_UUID WrappingKeyUUID;
	TSS_UUID KeyUUID;
	UINT32 cKeySize;
	BYTE *rgbKey;
	UINT32 cVendorData;
	BYTE *gbVendorData;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UUID, 1, &WrappingKeyUUID, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UUID, 2, &KeyUUID, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 3, &cKeySize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	rgbKey = calloc(1, cKeySize);
	if (rgbKey == NULL) {
		LogError("malloc of %d bytes failed.", cKeySize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 4, rgbKey, cKeySize, &data->comm)) {
		free(rgbKey);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_UINT32, 5, &cVendorData, 0, &data->comm)) {
		free(rgbKey);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (cVendorData == 0)
		gbVendorData = NULL;
	else {
		gbVendorData = calloc(1, cVendorData);
		if (gbVendorData == NULL) {
			LogError("malloc of %d bytes failed.", cVendorData);
			free(rgbKey);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}

		if (getData(TCSD_PACKET_TYPE_PBYTE, 6, gbVendorData, cVendorData, &data->comm)) {
			free(rgbKey);
			free(gbVendorData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
	}

	result = TCS_RegisterKey_Internal(hContext, &WrappingKeyUUID, &KeyUUID,
				     cKeySize, rgbKey, cVendorData,
				     gbVendorData);
	free(rgbKey);
	free(gbVendorData);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_UnregisterKey(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_UUID uuid;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UUID, 1, &uuid, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	result = TCSP_UnregisterKey_Internal(hContext, uuid);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetRegisteredKeyBlob(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_UUID uuid;
	UINT32 pcKeySize;
	BYTE *prgbKey;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UUID, 1, &uuid, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	result = TCS_GetRegisteredKeyBlob_Internal(hContext, &uuid, &pcKeySize,
					      &prgbKey);
	if (result == TSS_SUCCESS) {
		initData(&data->comm, 2);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &pcKeySize, 0, &data->comm)) {
			free(prgbKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 1, prgbKey, pcKeySize, &data->comm)) {
			free(prgbKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(prgbKey);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_LoadKeyByBlob(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE hUnwrappingKey;
	UINT32 cWrappedKeyBlob;
	BYTE *rgbWrappedKeyBlob;

	TPM_AUTH auth;

	TCS_KEY_HANDLE phKeyTCSI;
	TCS_KEY_HANDLE phKeyHMAC;

	TPM_AUTH *pAuth;
	TSS_RESULT result;
	int i;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &hUnwrappingKey, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &cWrappedKeyBlob, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	rgbWrappedKeyBlob = calloc(1, cWrappedKeyBlob);
	if (rgbWrappedKeyBlob == NULL) {
		LogError("malloc of %d bytes failed.", cWrappedKeyBlob);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, rgbWrappedKeyBlob, cWrappedKeyBlob, &data->comm)) {
		free(rgbWrappedKeyBlob);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	result = getData(TCSD_PACKET_TYPE_AUTH, 4, &auth, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pAuth = NULL;
	else if (result) {
		free(rgbWrappedKeyBlob);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		pAuth = &auth;

	pthread_mutex_lock(&tcsp_lock);

	result = key_mgr_load_by_blob(hContext, hUnwrappingKey, cWrappedKeyBlob, rgbWrappedKeyBlob,
				      pAuth, &phKeyTCSI, &phKeyHMAC);

	if (!result)
		result = ctx_mark_key_loaded(hContext, phKeyTCSI);

	pthread_mutex_unlock(&tcsp_lock);

	free(rgbWrappedKeyBlob);

	if (result == TSS_SUCCESS) {
		i = 0;
		initData(&data->comm, 3);
		if (pAuth != NULL) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pAuth, 0, &data->comm))
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &phKeyTCSI, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &phKeyHMAC, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_LoadKeyByUUID(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_UUID uuid;
	TCS_LOADKEY_INFO info, *pInfo;
	TCS_KEY_HANDLE phKeyTCSI;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UUID, 1, &uuid, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	result = getData(TCSD_PACKET_TYPE_LOADKEY_INFO, 2, &info, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pInfo = NULL;
	else if (result)
		return TCSERR(TSS_E_INTERNAL_ERROR);
	else
		pInfo = &info;

	pthread_mutex_lock(&tcsp_lock);

	result = key_mgr_load_by_uuid(hContext, &uuid, pInfo, &phKeyTCSI);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 2);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &phKeyTCSI, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		if (pInfo != NULL) {
			if (setData(TCSD_PACKET_TYPE_LOADKEY_INFO, 1, pInfo, 0, &data->comm))
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
	} else {
		if (result == TCSERR(TCS_E_KM_LOADFAILED) && pInfo != NULL) {
			initData(&data->comm, 1);
			if (setData(TCSD_PACKET_TYPE_LOADKEY_INFO, 0, pInfo, 0, &data->comm))
				return TCSERR(TSS_E_INTERNAL_ERROR);
		} else
			initData(&data->comm, 0);
	}

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_OSAP(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCPA_ENTITY_TYPE entityType;
	UINT32 entityValue;
	TCPA_NONCE nonceOddOSAP;

	TCS_AUTHHANDLE authHandle;
	TCPA_NONCE nonceEven;
	TCPA_NONCE nonceEvenOSAP;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT16, 1, &entityType, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &entityValue, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_NONCE, 3, &nonceOddOSAP, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = auth_mgr_osap(hContext, entityType, entityValue, nonceOddOSAP,
			       &authHandle, &nonceEven, &nonceEvenOSAP);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 3);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &authHandle, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		if (setData(TCSD_PACKET_TYPE_NONCE, 1, &nonceEven, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		if (setData(TCSD_PACKET_TYPE_NONCE, 2, &nonceEvenOSAP, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_Sign(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE hKey;
	UINT32 areaToSignSize;
	BYTE *areaToSign;

	TPM_AUTH auth;
	TPM_AUTH *pAuth;

	UINT32 sigSize;
	BYTE *sig;
	TSS_RESULT result;

	int i;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &hKey, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &areaToSignSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	areaToSign = calloc(1, areaToSignSize);
	if (areaToSign == NULL) {
		LogError("malloc of %d bytes failed.", areaToSignSize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, areaToSign, areaToSignSize, &data->comm)) {
		free(areaToSign);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	result = getData(TCSD_PACKET_TYPE_AUTH, 4, &auth, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pAuth = NULL;
	else if (result) {
		free(areaToSign);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		pAuth = &auth;

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_Sign_Internal(hContext, hKey, areaToSignSize, areaToSign,
			       pAuth, &sigSize, &sig);

	pthread_mutex_unlock(&tcsp_lock);

	free(areaToSign);

	if (result == TSS_SUCCESS) {
		i = 0;
		initData(&data->comm, 3);
		if (pAuth != NULL) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, &auth, 0, &data->comm)) {
				free(sig);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &sigSize, 0, &data->comm)) {
			free(sig);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, sig, sigSize, &data->comm)) {
			free(sig);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(sig);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_DirWriteAuth(struct tcsd_thread_data *data)
{
	TSS_RESULT result;
	TSS_HCONTEXT hContext;
	TCPA_DIRINDEX dirIndex;
	TCPA_DIGEST dirDigest;
	TPM_AUTH auth;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &dirIndex, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_DIGEST, 2, &dirDigest, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_AUTH, 3, &auth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_DirWriteAuth_Internal(hContext, dirIndex, dirDigest, &auth);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &auth, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_DirRead(struct tcsd_thread_data *data)
{
	TSS_RESULT result;
	TSS_HCONTEXT hContext;
	TCPA_DIRINDEX dirIndex;
	TCPA_DIRVALUE dirValue;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &dirIndex, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_DirRead_Internal(hContext, dirIndex, &dirValue);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_DIGEST, 0, &dirValue, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_Seal(struct tcsd_thread_data *data)
{
	TSS_RESULT result;
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE keyHandle;
	TCPA_ENCAUTH KeyUsageAuth;
	UINT32 PCRInfoSize, inDataSize;
	BYTE *PCRInfo = NULL, *inData = NULL;
	TPM_AUTH emptyAuth, pubAuth, *pAuth;
	UINT32 outDataSize;
	BYTE *outData;

	int i = 0;

	memset(&emptyAuth, 0, sizeof(TPM_AUTH));
	memset(&pubAuth, 0, sizeof(TPM_AUTH));

	if (getData(TCSD_PACKET_TYPE_UINT32, i++, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, i++, &keyHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_ENCAUTH, i++, &KeyUsageAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, i++, &PCRInfoSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (PCRInfoSize > 0) {
		PCRInfo = calloc(1, PCRInfoSize);
		if (PCRInfo == NULL) {
			LogError("malloc of %u bytes failed.", PCRInfoSize);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}

		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, PCRInfo, PCRInfoSize, &data->comm)) {
			free(PCRInfo);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
	}

	if (getData(TCSD_PACKET_TYPE_UINT32, i++, &inDataSize, 0, &data->comm)) {
		free(PCRInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (inDataSize > 0) {
		inData = calloc(1, inDataSize);
		if (inData == NULL) {
			LogError("malloc of %u bytes failed.", inDataSize);
			free(PCRInfo);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}

		if (getData(TCSD_PACKET_TYPE_PBYTE, i++, inData, inDataSize, &data->comm)) {
			free(inData);
			free(PCRInfo);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
	}

	result = getData(TCSD_PACKET_TYPE_AUTH, i++, &pubAuth, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pAuth = NULL;
	else if (result) {
		free(inData);
		free(PCRInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		pAuth = &pubAuth;

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_Seal_Internal(hContext, keyHandle, KeyUsageAuth, PCRInfoSize, PCRInfo,
			inDataSize, inData, pAuth, &outDataSize, &outData);

	pthread_mutex_unlock(&tcsp_lock);

	free(inData);
	free(PCRInfo);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 3);
		if (pAuth != NULL) {
			if (setData(TCSD_PACKET_TYPE_AUTH, 0, pAuth, 0, &data->comm)) {
				free(outData);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}

		if (setData(TCSD_PACKET_TYPE_UINT32, 1, &outDataSize, 0, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 2, outData, outDataSize, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(outData);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_UnSeal(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE parentHandle;
	UINT32 inDataSize;
	BYTE *inData;

	TPM_AUTH parentAuth, dataAuth, emptyAuth;
	TPM_AUTH *pParentAuth, *pDataAuth;

	UINT32 outDataSize;
	BYTE *outData;
	TSS_RESULT result;

	memset(&emptyAuth, 0, sizeof(TPM_AUTH));

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &parentHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &inDataSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	inData = calloc(1, inDataSize);
	if (inData == NULL) {
		LogError("malloc of %d bytes failed.", inDataSize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}

	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, inData, inDataSize, &data->comm)) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
#if 0
	if (getData(TCSD_PACKET_TYPE_AUTH, 4, &parentAuth, 0, &data->comm)) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (!memcmp(&emptyAuth, &parentAuth, sizeof(TPM_AUTH)))
		pParentAuth = NULL;
	else
		pParentAuth = &parentAuth;

	if (getData(TCSD_PACKET_TYPE_AUTH, 5, &dataAuth, 0, &data->comm)) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
#else
	result = getData(TCSD_PACKET_TYPE_AUTH, 4, &parentAuth, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pParentAuth = NULL;
	else if (result) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		pParentAuth = &parentAuth;

	result = getData(TCSD_PACKET_TYPE_AUTH, 5, &dataAuth, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE) {
		pDataAuth = pParentAuth;
		pParentAuth = NULL;
	} else if (result) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		pDataAuth = &dataAuth;
#endif
	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_Unseal_Internal(hContext, parentHandle, inDataSize, inData,
				 pParentAuth, pDataAuth, &outDataSize, &outData);

	pthread_mutex_unlock(&tcsp_lock);

	free(inData);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 4);
		if (pParentAuth != NULL) {
			if (setData(TCSD_PACKET_TYPE_AUTH, 0, pParentAuth, 0, &data->comm)) {
				free(outData);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		} else {
			if (setData(TCSD_PACKET_TYPE_AUTH, 0, &emptyAuth, 0, &data->comm)) {
				free(outData);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}

		if (setData(TCSD_PACKET_TYPE_AUTH, 1, &dataAuth, 0, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_UINT32, 2, &outDataSize, 0, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 3, outData, outDataSize, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(outData);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_UnBind(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE keyHandle;
	UINT32 inDataSize;
	BYTE *inData;

	TPM_AUTH privAuth;
	TPM_AUTH *pPrivAuth;

	UINT32 outDataSize;
	BYTE *outData;
	TSS_RESULT result;

	int i;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &keyHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &inDataSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	inData = calloc(1, inDataSize);
	if (inData == NULL) {
		LogError("malloc of %d bytes failed.", inDataSize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, inData, inDataSize, &data->comm)) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
#if 0
	if (getData(TCSD_PACKET_TYPE_AUTH, 4, &privAuth, 0, &data->comm))
		pPrivAuth = NULL;
	else
		pPrivAuth = &privAuth;
#else
	result = getData(TCSD_PACKET_TYPE_AUTH, 4, &privAuth, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pPrivAuth = NULL;
	else if (result) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		pPrivAuth = &privAuth;
#endif
	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_UnBind_Internal(hContext, keyHandle, inDataSize, inData,
				 pPrivAuth, &outDataSize, &outData);

	pthread_mutex_unlock(&tcsp_lock);

	free(inData);

	if (result == TSS_SUCCESS) {
		i = 0;
		initData(&data->comm, 3);
		if (pPrivAuth != NULL) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pPrivAuth, 0, &data->comm))
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &outDataSize, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, outData, outDataSize, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
		free(outData);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_StirRandom(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	UINT32 inDataSize;
	BYTE *inData;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &inDataSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	inData = calloc(1, inDataSize);
	if (inData == NULL) {
		LogError("malloc of %d bytes failed.", inDataSize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 2, inData, inDataSize, &data->comm)) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_StirRandom_Internal(hContext, inDataSize, inData);

	pthread_mutex_unlock(&tcsp_lock);

	free(inData);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_CreateWrapKey(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE hWrappingKey;
	TCPA_ENCAUTH KeyUsageAuth;
	TCPA_ENCAUTH KeyMigrationAuth;
	UINT32 keyInfoSize;
	BYTE *keyInfo;

	TPM_AUTH pAuth;

	UINT32 keyDataSize;
	BYTE *keyData;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &hWrappingKey, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_ENCAUTH, 2, &KeyUsageAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_ENCAUTH, 3, &KeyMigrationAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 4, &keyInfoSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	keyInfo = calloc(1, keyInfoSize);
	if (keyInfo == NULL) {
		LogError("malloc of %d bytes failed.", keyInfoSize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 5, keyInfo, keyInfoSize, &data->comm)) {
		free(keyInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_AUTH, 6, &pAuth, 0, &data->comm)) {
		free(keyInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_CreateWrapKey_Internal(hContext, hWrappingKey, KeyUsageAuth,
					KeyMigrationAuth, keyInfoSize, keyInfo,
					&keyDataSize, &keyData, &pAuth);

	pthread_mutex_unlock(&tcsp_lock);

	free(keyInfo);
	if (result == TSS_SUCCESS) {
		initData(&data->comm, 3);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &keyDataSize, 0, &data->comm)) {
			free(keyData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 1, keyData, keyDataSize, &data->comm)) {
			free(keyData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(keyData);
		if (setData(TCSD_PACKET_TYPE_AUTH, 2, &pAuth, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_ChangeAuth(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE parentHandle;
	TCPA_PROTOCOL_ID protocolID;
	TCPA_ENCAUTH newAuth;
	TCPA_ENTITY_TYPE entityType;
	UINT32 encDataSize;
	BYTE *encData;

	TPM_AUTH ownerAuth;
	TPM_AUTH entityAuth;

	UINT32 outDataSize;
	BYTE *outData;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &parentHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT16, 2, &protocolID, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_ENCAUTH, 3, &newAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT16, 4, &entityType, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 5, &encDataSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	encData = calloc(1, encDataSize);
	if (encData == NULL) {
		LogError("malloc of %d bytes failed.", encDataSize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 6, encData, encDataSize, &data->comm)) {
		free(encData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_AUTH, 7, &ownerAuth, 0, &data->comm)) {
		free(encData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_AUTH, 8, &entityAuth, 0, &data->comm)) {
		free(encData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_ChangeAuth_Internal(hContext, parentHandle, protocolID,
				     newAuth, entityType, encDataSize, encData,
				     &ownerAuth, &entityAuth, &outDataSize,
				     &outData);

	pthread_mutex_unlock(&tcsp_lock);

	free(encData);
	if (result == TSS_SUCCESS) {
		initData(&data->comm, 4);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &ownerAuth, 0, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_AUTH, 1, &entityAuth, 0, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_UINT32, 2, &outDataSize, 0, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 3, outData, outDataSize, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(outData);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_ChangeAuthOwner(struct tcsd_thread_data *data)
{

	TCS_CONTEXT_HANDLE hContext;
	TCPA_PROTOCOL_ID protocolID;
	TCPA_ENCAUTH newAuth;
	TCPA_ENTITY_TYPE entityType;

	TPM_AUTH ownerAuth;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT16, 1, &protocolID, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_ENCAUTH, 2, &newAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT16, 3, &entityType, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_AUTH, 4, &ownerAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_ChangeAuthOwner_Internal(hContext, protocolID, newAuth,
					  entityType, &ownerAuth);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &ownerAuth, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_Quote(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE hKey;
	TCPA_NONCE antiReplay;
	UINT32 pcrDataSizeIn;
	BYTE *pcrDataIn;

	TPM_AUTH privAuth;
	TPM_AUTH *pPrivAuth;

	UINT32 pcrDataSizeOut;
	BYTE *pcrDataOut;
	UINT32 sigSize;
	BYTE *sig;
	TSS_RESULT result;

	int i;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &hKey, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_NONCE, 2, &antiReplay, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 3, &pcrDataSizeIn, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	pcrDataIn = (BYTE *)calloc(1, pcrDataSizeIn);
	if (pcrDataIn == NULL) {
		LogError("malloc of %d bytes failed.", pcrDataSizeIn);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 4, pcrDataIn, pcrDataSizeIn, &data->comm)) {
		free(pcrDataIn);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
#if 0
	if (getData(TCSD_PACKET_TYPE_AUTH, 5, &privAuth, 0, &data->comm))
		pPrivAuth = NULL;
	else
		pPrivAuth = &privAuth;
#else
	result = getData(TCSD_PACKET_TYPE_AUTH, 5, &privAuth, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pPrivAuth = NULL;
	else if (result) {
		free(pcrDataIn);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		pPrivAuth = &privAuth;
#endif

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_Quote_Internal(hContext, hKey, antiReplay, pcrDataSizeIn,
				pcrDataIn, pPrivAuth, &pcrDataSizeOut,
				&pcrDataOut, &sigSize, &sig);

	pthread_mutex_unlock(&tcsp_lock);

	free(pcrDataIn);
	if (result == TSS_SUCCESS) {
		i = 0;
		initData(&data->comm, 5);
		if (pPrivAuth != NULL) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pPrivAuth, 0, &data->comm)) {
				free(pcrDataOut);
				free(sig);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &pcrDataSizeOut, 0, &data->comm)) {
			free(pcrDataOut);
			free(sig);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, pcrDataOut, pcrDataSizeOut, &data->comm)) {
			free(pcrDataOut);
			free(sig);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &sigSize, 0, &data->comm)) {
			free(pcrDataOut);
			free(sig);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, sig, sigSize, &data->comm)) {
			free(pcrDataOut);
			free(sig);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		free(pcrDataOut);
		free(sig);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_MakeIdentity(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCPA_ENCAUTH identityAuth;
	TCPA_CHOSENID_HASH privCAHash;
	UINT32 idKeyInfoSize;
	BYTE *idKeyInfo = NULL;

	TPM_AUTH auth1, auth2;
	TPM_AUTH *pSRKAuth, *pOwnerAuth;

	UINT32 idKeySize;
	BYTE *idKey = NULL;
	UINT32 pcIDBindSize;
	BYTE *prgbIDBind = NULL;
	UINT32 pcECSize;
	BYTE *prgbEC = NULL;
	UINT32 pcPlatCredSize;
	BYTE *prgbPlatCred = NULL;
	UINT32 pcConfCredSize;
	BYTE *prgbConfCred = NULL;
	TSS_RESULT result;

	int i;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_ENCAUTH, 1, &identityAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_DIGEST, 2, &privCAHash, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_UINT32, 3, &idKeyInfoSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	idKeyInfo = (BYTE *) calloc(1, idKeyInfoSize);
	if (idKeyInfo == NULL) {
		LogError("malloc of %d bytes failed.", idKeyInfoSize);
		return TCSERR(TSS_E_OUTOFMEMORY);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 4, idKeyInfo, idKeyInfoSize, &data->comm)) {
		free(idKeyInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_AUTH, 5, &auth1, 0, &data->comm)) {
		free(idKeyInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
#if 0
	/* if the next one is missing, then the previous was really the owner auth */
	if (getData(TCSD_PACKET_TYPE_AUTH, 6, &ownerAuth, 0, &data->comm)) {
		LogDebug("Failed to get ownerAuth.  SRK auth is really NULL "
			  "and single auth is ownerAuth");
		pSRKAuth = NULL;
		memcpy(&ownerAuth, &srkAuth, sizeof (TPM_AUTH));
	} else {
		LogDebug("two Auth");
		pSRKAuth = &srkAuth;
	}
#else
	result = getData(TCSD_PACKET_TYPE_AUTH, 6, &auth2, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE) {
		pOwnerAuth = &auth1;
		pSRKAuth = NULL;
	} else if (result) {
		free(idKeyInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	} else {
		pOwnerAuth = &auth2;
		pSRKAuth = &auth1;
	}
#endif

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_MakeIdentity_Internal(hContext, identityAuth, privCAHash,
				       idKeyInfoSize, idKeyInfo, pSRKAuth,
				       pOwnerAuth, &idKeySize, &idKey,
				       &pcIDBindSize, &prgbIDBind, &pcECSize,
				       &prgbEC, &pcPlatCredSize, &prgbPlatCred,
				       &pcConfCredSize, &prgbConfCred);

	pthread_mutex_unlock(&tcsp_lock);

	free(idKeyInfo);

	if (result == TSS_SUCCESS) {
		i = 0;
		initData(&data->comm, 12);
		if (pSRKAuth) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pSRKAuth, 0, &data->comm))
				goto internal_error;
		}
		if (setData(TCSD_PACKET_TYPE_AUTH, i++, pOwnerAuth, 0, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &idKeySize, 0, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, idKey, idKeySize, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &pcIDBindSize, 0, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, prgbIDBind, pcIDBindSize, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &pcECSize, 0, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, prgbEC, pcECSize, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &pcPlatCredSize, 0, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, prgbPlatCred, pcPlatCredSize, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &pcConfCredSize, 0, &data->comm))
			goto internal_error;
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, prgbConfCred, pcConfCredSize, &data->comm))
			goto internal_error;

		free(idKey);
		free(prgbIDBind);
		free(prgbEC);
		free(prgbPlatCred);
		free(prgbConfCred);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;

internal_error:
	free(idKey);
	free(prgbIDBind);
	free(prgbEC);
	free(prgbPlatCred);
	free(prgbConfCred);
	return TCSERR(TSS_E_INTERNAL_ERROR);
}

TSS_RESULT
tcs_wrap_EnumRegisteredKeys(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_UUID uuid, *pUuid;
	UINT32 cKeyHierarchySize;
	TSS_KM_KEYINFO *pKeyHierarchy;
	unsigned int i, j;
	TSS_RESULT result;

	/* Receive */
	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

#if 0
	if (getData(TCSD_PACKET_TYPE_UUID , 1, &uuid, 0, &data->comm))
		pUuid = NULL;
	else
		pUuid = &uuid;
#else
	result = getData(TCSD_PACKET_TYPE_UUID , 1, &uuid, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pUuid = NULL;
	else if (result)
		return TCSERR(TSS_E_INTERNAL_ERROR);
	else
		pUuid = &uuid;
#endif
	result = TCS_EnumRegisteredKeys_Internal(
			hContext,
			pUuid,
			&cKeyHierarchySize,
			&pKeyHierarchy);
	if (result == TSS_SUCCESS) {
		i = 0;
		initData(&data->comm, 1 + cKeyHierarchySize);
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &cKeyHierarchySize, 0, &data->comm)) {
			free(pKeyHierarchy);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		for (j = 0; j < cKeyHierarchySize; j++) {
			if (setData(TCSD_PACKET_TYPE_KM_KEYINFO, i++, &pKeyHierarchy[j], 0, &data->comm)) {
				free(pKeyHierarchy);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}
		free(pKeyHierarchy);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_LogPcrEvent(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_PCR_EVENT event;
	TSS_RESULT result;
	UINT32 number;

	/* Receive */
	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_PCR_EVENT , 1, &event, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	result = TCS_LogPcrEvent_Internal(hContext, event, &number);
	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &number, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetPcrEvent(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_PCR_EVENT *pEvent = NULL;
	TSS_RESULT result;
	UINT32 pcrIndex, number, totalSize;
	BYTE lengthOnly;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &pcrIndex, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &number, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_BYTE, 3, &lengthOnly, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (lengthOnly)
		result = TCS_GetPcrEvent_Internal(hContext, pcrIndex, &number, NULL);
	else
		result = TCS_GetPcrEvent_Internal(hContext, pcrIndex, &number, &pEvent);

	if (result == TSS_SUCCESS) {
		if (lengthOnly == FALSE)
			totalSize = get_pcr_event_size(pEvent);
		else
			totalSize = 0;

		initData(&data->comm, 2);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &number, 0, &data->comm)) {
			if (lengthOnly == FALSE)
				free_external_events(1, pEvent);
			free(pEvent);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		if (lengthOnly == FALSE) {
			if (setData(TCSD_PACKET_TYPE_PCR_EVENT, 1, pEvent, 0, &data->comm)) {
				free_external_events(1, pEvent);
				free(pEvent);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
			free_external_events(1, pEvent);
			free(pEvent);
		}
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetPcrEventsByPcr(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_PCR_EVENT *ppEvents = NULL;
	TSS_RESULT result;
	UINT32 firstEvent, eventCount, totalSize, pcrIndex, i, j;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &pcrIndex, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &firstEvent, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_UINT32, 3, &eventCount, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	result = TCS_GetPcrEventsByPcr_Internal(hContext, pcrIndex, firstEvent, &eventCount, &ppEvents);
	if (result == TSS_SUCCESS) {
		for (i = 0, totalSize = 0; i < eventCount; i++)
			totalSize += get_pcr_event_size(&(ppEvents[i]));

		initData(&data->comm, 1 + eventCount);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &eventCount, 0, &data->comm)) {
			free_external_events(eventCount, ppEvents);
			free(ppEvents);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		i = 1;
		for (j = 0; j < eventCount; j++) {
			if (setData(TCSD_PACKET_TYPE_PCR_EVENT, i++, &(ppEvents[j]), 0, &data->comm)) {
				free_external_events(eventCount, ppEvents);
				free(ppEvents);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}

		free_external_events(eventCount, ppEvents);
		free(ppEvents);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetPcrEventLog(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_PCR_EVENT *ppEvents;
	TSS_RESULT result;
	UINT32 eventCount, totalSize, i, j;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	result = TCS_GetPcrEventLog_Internal(hContext, &eventCount, &ppEvents);
	if (result == TSS_SUCCESS) {
		for (i = 0, totalSize = 0; i < eventCount; i++)
			totalSize += get_pcr_event_size(&(ppEvents[i]));

		initData(&data->comm, 1 + eventCount);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &eventCount, 0, &data->comm)) {
			free_external_events(eventCount, ppEvents);
			free(ppEvents);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		i = 1;
		for (j = 0; j < eventCount; j++) {
			if (setData(TCSD_PACKET_TYPE_PCR_EVENT, i++, &(ppEvents[j]), 0, &data->comm)) {
				free_external_events(eventCount, ppEvents);
				free(ppEvents);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}

		free_external_events(eventCount, ppEvents);
		free(ppEvents);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_SelfTestFull(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x servicing a %s request", (UINT32)pthread_self(), __FUNCTION__);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_SelfTestFull_Internal(hContext);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_CertifySelfTest(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	UINT32 sigSize;
	BYTE *sigData = NULL;
	TCS_KEY_HANDLE hKey;
	TCPA_NONCE antiReplay;
	TPM_AUTH privAuth;
	TPM_AUTH *pPrivAuth;
	int i;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x servicing a %s request", (UINT32)pthread_self(), __FUNCTION__);
        if (getData(TCSD_PACKET_TYPE_UINT32, 1, &hKey, 0, &data->comm))
                return TCSERR(TSS_E_INTERNAL_ERROR);
        if (getData(TCSD_PACKET_TYPE_NONCE, 2, &antiReplay, 0, &data->comm))
                return TCSERR(TSS_E_INTERNAL_ERROR);
#if 0
        if (getData(TCSD_PACKET_TYPE_AUTH, 3, &privAuth, 0, &data->comm))
                pPrivAuth = NULL;
        else
                pPrivAuth = &privAuth;
#else
        result = getData(TCSD_PACKET_TYPE_AUTH, 3, &privAuth, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
                pPrivAuth = NULL;
	else if (result)
                return TCSERR(TSS_E_INTERNAL_ERROR);
        else
                pPrivAuth = &privAuth;

#endif
	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_CertifySelfTest_Internal(hContext, hKey, antiReplay, pPrivAuth, &sigSize, &sigData);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		i = 0;
		initData(&data->comm, 3);
                if (pPrivAuth != NULL) {
                        if (setData(TCSD_PACKET_TYPE_AUTH, i++, pPrivAuth, 0, &data->comm)) {
                                free(sigData);
                                return TCSERR(TSS_E_INTERNAL_ERROR);
                        }
                }

		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &sigSize, 0, &data->comm)) {
			free(sigData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, sigData, sigSize, &data->comm)) {
			free(sigData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(sigData);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetTestResult(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	UINT32 resultDataSize;
	BYTE *resultData = NULL;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x servicing a %s request", (UINT32)pthread_self(), __FUNCTION__);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_GetTestResult_Internal(hContext, &resultDataSize, &resultData);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 2);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &resultDataSize, 0, &data->comm)) {
			free(resultData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 1, resultData, resultDataSize, &data->comm)) {
			free(resultData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(resultData);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_SetOwnerInstall(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_BOOL state;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_BOOL, 1, &state, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_SetOwnerInstall_Internal(hContext, state);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_OwnerSetDisable(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_BOOL disableState;
	TPM_AUTH ownerAuth;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_BOOL, 1, &disableState, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_AUTH, 2, &ownerAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_OwnerSetDisable_Internal(hContext, disableState, &ownerAuth);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &ownerAuth, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_PhysicalDisable(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_PhysicalDisable_Internal(hContext);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_PhysicalPresence(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TCPA_PHYSICAL_PRESENCE phyPresFlags;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT16, 1, &phyPresFlags, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_PhysicalPresence_Internal(hContext, phyPresFlags);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_SetTempDeactivated(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_SetTempDeactivated_Internal(hContext);

	pthread_mutex_unlock(&tcsp_lock);

	initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_CertifyKey(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE certHandle, keyHandle;
	TPM_AUTH *pCertAuth = NULL, *pKeyAuth = NULL, certAuth, keyAuth, nullAuth;
	UINT32 CertifyInfoSize, outDataSize;
	BYTE *CertifyInfo, *outData;
	TCPA_NONCE antiReplay;
	TSS_RESULT result;

	memset(&nullAuth, 0, sizeof(TPM_AUTH));
	memset(&certAuth, 0, sizeof(TPM_AUTH));
	memset(&keyAuth, 0, sizeof(TPM_AUTH));

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &certHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &keyHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_NONCE, 3, &antiReplay, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_AUTH, 4, &certAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_AUTH, 5, &keyAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (memcmp(&nullAuth, &certAuth, sizeof(TPM_AUTH)))
		pCertAuth = &certAuth;

	if (memcmp(&nullAuth, &keyAuth, sizeof(TPM_AUTH)))
		pKeyAuth = &keyAuth;

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_CertifyKey_Internal(hContext, certHandle, keyHandle,
			antiReplay, pCertAuth, pKeyAuth, &CertifyInfoSize,
			&CertifyInfo, &outDataSize, &outData);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		int i = 0;
		initData(&data->comm, 6);
		if (pCertAuth) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pCertAuth, 0, &data->comm)) {
				free(CertifyInfo);
				free(outData);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}
		if (pKeyAuth) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pKeyAuth, 0, &data->comm)) {
				free(CertifyInfo);
				free(outData);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &CertifyInfoSize, 0, &data->comm)) {
			free(CertifyInfo);
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, CertifyInfo, CertifyInfoSize, &data->comm)) {
			free(CertifyInfo);
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(CertifyInfo);
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &outDataSize, 0, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, outData, outDataSize, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(outData);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_GetRegisteredKeyByPublicInfo(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	UINT32 algId, ulPublicInfoLength, keySize;
	BYTE *rgbPublicInfo, *keyBlob;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &algId, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &ulPublicInfoLength, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	rgbPublicInfo = (BYTE *)calloc(1, ulPublicInfoLength);
	if (rgbPublicInfo == NULL) {
		LogError("malloc of %d bytes failed.", ulPublicInfoLength);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, rgbPublicInfo, ulPublicInfoLength, &data->comm)) {
		free(rgbPublicInfo);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_GetRegisteredKeyByPublicInfo_Internal(hContext, algId,
			ulPublicInfoLength, rgbPublicInfo, &keySize, &keyBlob);

	pthread_mutex_unlock(&tcsp_lock);

	free(rgbPublicInfo);
	if (result == TSS_SUCCESS) {
		initData(&data->comm, 2);
		if (setData(TCSD_PACKET_TYPE_UINT32, 0, &keySize, 0, &data->comm)) {
			free(keyBlob);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 1, keyBlob, keySize, &data->comm)) {
			free(keyBlob);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(keyBlob);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_ActivateIdentity(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TCS_KEY_HANDLE idKeyHandle;
	TPM_AUTH *pIdKeyAuth = NULL, *pOwnerAuth = NULL, auth1, auth2;
	UINT32 SymmetricKeySize, blobSize;
	BYTE *SymmetricKey, *blob;
	TSS_RESULT result;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebug("thread %x context %x: %s", (UINT32)pthread_self(), hContext, __FUNCTION__);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &idKeyHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &blobSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if ((blob = malloc(blobSize)) == NULL)
		return TCSERR(TSS_E_OUTOFMEMORY);

	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, blob, blobSize, &data->comm)) {
		free(blob);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (getData(TCSD_PACKET_TYPE_AUTH, 4, &auth1, 0, &data->comm)) {
		free(blob);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
#if 0
	if (getData(TCSD_PACKET_TYPE_AUTH, 5, &auth2, 0, &data->comm)) {
		LogDebugFn("No auth for identity key");
		pOwnerAuth = &auth1;
	} else {
		pIdKeyAuth = &auth1;
		pOwnerAuth = &auth2;
	}
#else
	result = getData(TCSD_PACKET_TYPE_AUTH, 5, &auth2, 0, &data->comm);
	if (result == TSS_TCP_RPC_BAD_PACKET_TYPE)
		pOwnerAuth = &auth1;
	else if (result) {
		free(blob);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	} else {
		pIdKeyAuth = &auth1;
		pOwnerAuth = &auth2;
	}
#endif
	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_ActivateTPMIdentity_Internal(hContext, idKeyHandle, blobSize,
						   blob, pIdKeyAuth, pOwnerAuth,
						   &SymmetricKeySize,
						   &SymmetricKey);

	pthread_mutex_unlock(&tcsp_lock);

	free(blob);

	if (result == TSS_SUCCESS) {
		int i = 0;
		initData(&data->comm, 4);
		if (pIdKeyAuth) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pIdKeyAuth, 0, &data->comm)) {
				free(SymmetricKey);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}
		if (setData(TCSD_PACKET_TYPE_AUTH, i++, pOwnerAuth, 0, &data->comm)) {
			free(SymmetricKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &SymmetricKeySize, 0, &data->comm)) {
			free(SymmetricKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, SymmetricKey, SymmetricKeySize, &data->comm)) {
			free(SymmetricKey);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		free(SymmetricKey);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_CreateMigrationBlob(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TCS_KEY_HANDLE parentHandle;
	TSS_MIGRATE_SCHEME migrationType;
	UINT32 MigrationKeyAuthSize, encDataSize, randomSize, outDataSize;
	BYTE *MigrationKeyAuth, *encData, *random, *outData;
	TPM_AUTH auth1, auth2, *pParentAuth, *pEntityAuth;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebugFn("thread %x context %x", (UINT32)pthread_self(), hContext);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &parentHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);
	if (getData(TCSD_PACKET_TYPE_UINT16, 2, &migrationType, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_UINT32, 3, &MigrationKeyAuthSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	MigrationKeyAuth = (BYTE *)malloc(MigrationKeyAuthSize);
	if (MigrationKeyAuth == NULL) {
		LogError("malloc of %d bytes failed.", MigrationKeyAuthSize);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 4, MigrationKeyAuth, MigrationKeyAuthSize, &data->comm)) {
		free(MigrationKeyAuth);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (getData(TCSD_PACKET_TYPE_UINT32, 5, &encDataSize, 0, &data->comm)) {
		free(MigrationKeyAuth);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	encData = (BYTE *)malloc(encDataSize);
	if (encData == NULL) {
		free(MigrationKeyAuth);
		LogError("malloc of %d bytes failed.", encDataSize);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 6, encData, encDataSize, &data->comm)) {
		free(MigrationKeyAuth);
		free(encData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (getData(TCSD_PACKET_TYPE_AUTH, 7, &auth1, 0, &data->comm)) {
		free(MigrationKeyAuth);
		free(encData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (getData(TCSD_PACKET_TYPE_AUTH, 8, &auth2, 0, &data->comm)) {
		/* If loading the 2nd auth fails, the first one was entity auth */
		pParentAuth = NULL;
		pEntityAuth = &auth1;
	} else {
		/* If loading the 2nd auth succeeds, the first one was parent auth */
		pParentAuth = &auth1;
		pEntityAuth = &auth2;
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_CreateMigrationBlob_Internal(hContext, parentHandle, migrationType,
						   MigrationKeyAuthSize, MigrationKeyAuth,
						   encDataSize, encData, pParentAuth, pEntityAuth,
						   &randomSize, &random, &outDataSize, &outData);

	pthread_mutex_unlock(&tcsp_lock);

	free(MigrationKeyAuth);
	free(encData);
	if (result == TSS_SUCCESS) {
		int i = 0;
		initData(&data->comm, 6);
		if (pParentAuth) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pParentAuth, 0, &data->comm)) {
				free(random);
				free(outData);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}

		if (setData(TCSD_PACKET_TYPE_AUTH, i++, pEntityAuth, 0, &data->comm)) {
			free(random);
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &randomSize, 0, &data->comm)) {
			free(random);
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (randomSize > 0) {
			if (setData(TCSD_PACKET_TYPE_PBYTE, i++, random, randomSize, &data->comm)) {
				free(random);
				free(outData);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}

		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &outDataSize, 0, &data->comm)) {
			free(random);
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, outData, outDataSize, &data->comm)) {
			free(random);
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		free(random);
		free(outData);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_ConvertMigrationBlob(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TCS_KEY_HANDLE parentHandle;
	UINT32 outDataSize, randomSize, inDataSize;
	BYTE *outData, *random, *inData;
	TPM_AUTH parentAuth, *pParentAuth;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebugFn("thread %x context %x", (UINT32)pthread_self(), hContext);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &parentHandle, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &inDataSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	inData = (BYTE *)malloc(inDataSize);
	if (inData == NULL) {
		LogError("malloc of %d bytes failed.", inDataSize);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, inData, inDataSize, &data->comm)) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (getData(TCSD_PACKET_TYPE_UINT32, 4, &randomSize, 0, &data->comm)) {
		free(inData);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	random = (BYTE *)malloc(randomSize);
	if (random == NULL) {
		free(inData);
		LogError("malloc of %d bytes failed.", randomSize);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 5, random, randomSize, &data->comm)) {
		free(inData);
		free(random);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (getData(TCSD_PACKET_TYPE_AUTH, 6, &parentAuth, 0, &data->comm))
		pParentAuth = NULL;
	else
		pParentAuth = &parentAuth;


	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_ConvertMigrationBlob_Internal(hContext, parentHandle, inDataSize, inData,
						    randomSize, random, pParentAuth, &outDataSize,
						    &outData);

	pthread_mutex_unlock(&tcsp_lock);

	free(inData);
	free(random);
	if (result == TSS_SUCCESS) {
		int i = 0;
		initData(&data->comm, 3);
		if (pParentAuth) {
			if (setData(TCSD_PACKET_TYPE_AUTH, i++, pParentAuth, 0, &data->comm)) {
				free(outData);
				return TCSERR(TSS_E_INTERNAL_ERROR);
			}
		}

		if (setData(TCSD_PACKET_TYPE_UINT32, i++, &outDataSize, 0, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, i++, outData, outDataSize, &data->comm)) {
			free(outData);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		free(outData);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_AuthorizeMigrationKey(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TSS_MIGRATE_SCHEME migrateScheme;
	UINT32 MigrationKeySize, MigrationKeyAuthSize;
	BYTE *MigrationKey, *MigrationKeyAuth;
	TPM_AUTH ownerAuth;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebugFn("thread %x context %x", (UINT32)pthread_self(), hContext);

	if (getData(TCSD_PACKET_TYPE_UINT16, 1, &migrateScheme, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &MigrationKeySize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	MigrationKey = (BYTE *)malloc(MigrationKeySize);
	if (MigrationKey == NULL) {
		LogError("malloc of %d bytes failed.", MigrationKeySize);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, MigrationKey, MigrationKeySize, &data->comm)) {
		free(MigrationKey);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (getData(TCSD_PACKET_TYPE_AUTH, 4, &ownerAuth, 0, &data->comm)) {
		free(MigrationKey);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_AuthorizeMigrationKey_Internal(hContext, migrateScheme, MigrationKeySize,
						     MigrationKey, &ownerAuth,
						     &MigrationKeyAuthSize, &MigrationKeyAuth);

	pthread_mutex_unlock(&tcsp_lock);

	free(MigrationKey);
	if (result == TSS_SUCCESS) {
		initData(&data->comm, 3);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &ownerAuth, 0, &data->comm)) {
			free(MigrationKeyAuth);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		if (setData(TCSD_PACKET_TYPE_UINT32, 1, &MigrationKeyAuthSize, 0, &data->comm)) {
			free(MigrationKeyAuth);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 2, MigrationKeyAuth, MigrationKeyAuthSize,
			    &data->comm)) {
			free(MigrationKeyAuth);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		free(MigrationKeyAuth);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_KillMaintenanceFeature(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TPM_AUTH ownerAuth;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebugFn("thread %x context %x", (UINT32)pthread_self(), hContext);

	if (getData(TCSD_PACKET_TYPE_AUTH, 1, &ownerAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_KillMaintenanceFeature_Internal(hContext, &ownerAuth);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &ownerAuth, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_CreateMaintenanceArchive(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TPM_AUTH ownerAuth;
	TSS_BOOL generateRandom;
	UINT32 randomSize, archiveSize;
	BYTE *random, *archive;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebugFn("thread %x context %x", (UINT32)pthread_self(), hContext);

	if (getData(TCSD_PACKET_TYPE_BOOL, 1, &generateRandom, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_AUTH, 2, &ownerAuth, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_CreateMaintenanceArchive_Internal(hContext, generateRandom, &ownerAuth,
							&randomSize, &random, &archiveSize,
							&archive);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 5);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &ownerAuth, 0, &data->comm)) {
			free(random);
			free(archive);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		if (setData(TCSD_PACKET_TYPE_UINT32, 1, &randomSize, 0, &data->comm)) {
			free(random);
			free(archive);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 2, random, randomSize, &data->comm)) {
			free(random);
			free(archive);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		if (setData(TCSD_PACKET_TYPE_UINT32, 3, &archiveSize, 0, &data->comm)) {
			free(random);
			free(archive);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 4, archive, archiveSize, &data->comm)) {
			free(random);
			free(archive);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		free(random);
		free(archive);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_LoadMaintenanceArchive(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TPM_AUTH ownerAuth;
	UINT32 dataInSize, dataOutSize;
	BYTE *dataIn, *dataOut;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebugFn("thread %x context %x", (UINT32)pthread_self(), hContext);

	if (getData(TCSD_PACKET_TYPE_UINT32, 1, &dataInSize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	dataIn = (BYTE *)malloc(dataInSize);
	if (dataIn == NULL) {
		LogError("malloc of %d bytes failed.", dataInSize);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}
	if (getData(TCSD_PACKET_TYPE_PBYTE, 2, dataIn, dataInSize, &data->comm)) {
		free(dataIn);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (getData(TCSD_PACKET_TYPE_AUTH, 3, &ownerAuth, 0, &data->comm)) {
		free(dataIn);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_LoadMaintenanceArchive_Internal(hContext, dataInSize, dataIn, &ownerAuth,
							&dataOutSize, &dataOut);

	pthread_mutex_unlock(&tcsp_lock);
	free(dataIn);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 3);
		if (setData(TCSD_PACKET_TYPE_AUTH, 0, &ownerAuth, 0, &data->comm)) {
			free(dataOut);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		if (setData(TCSD_PACKET_TYPE_UINT32, 1, &dataOutSize, 0, &data->comm)) {
			free(dataOut);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
		if (setData(TCSD_PACKET_TYPE_PBYTE, 2, dataOut, dataOutSize, &data->comm)) {
			free(dataOut);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}

		free(dataOut);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}

TSS_RESULT
tcs_wrap_LoadManuMaintPub(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	UINT32 pubKeySize;
	BYTE *pubKey;
	TCPA_NONCE antiReplay;
	TCPA_DIGEST checksum;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebugFn("thread %x context %x", (UINT32)pthread_self(), hContext);

	if (getData(TCSD_PACKET_TYPE_NONCE, 1, &antiReplay, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	if (getData(TCSD_PACKET_TYPE_UINT32, 2, &pubKeySize, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pubKey = (BYTE *)malloc(pubKeySize);
	if (pubKey == NULL) {
		LogError("malloc of %d bytes failed.", pubKeySize);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	if (getData(TCSD_PACKET_TYPE_PBYTE, 3, pubKey, pubKeySize, &data->comm)) {
		free(pubKey);
		return TCSERR(TSS_E_INTERNAL_ERROR);
	}

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_LoadManuMaintPub_Internal(hContext, antiReplay, pubKeySize, pubKey,
						&checksum);

	pthread_mutex_unlock(&tcsp_lock);
	free(pubKey);

	if (result == TSS_SUCCESS) {
#if 0
		*hdr = calloc(1, size + sizeof(TCPA_DIGEST));
		if (*hdr == NULL) {
			LogError("malloc of %zu bytes failed.", size + sizeof(TCPA_DIGEST));
			return TCSERR(TSS_E_OUTOFMEMORY);
		}

		if (setData(TCSD_PACKET_TYPE_DIGEST, 0, &checksum, 0, *hdr)) {
			free(*hdr);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
#else
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_DIGEST, 0, &checksum, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
#endif
	} else {
#if 0
		*hdr = calloc(1, size);
		if (*hdr == NULL) {
			LogError("malloc of %u bytes failed.", size);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		(*hdr)->packet_size = size;
#else
		initData(&data->comm, 0);
#endif
	}
#if 0
	(*hdr)->result = result;
#else
	data->comm.hdr.u.result = result;
#endif

	return TSS_SUCCESS;
}
#if 0
TSS_RESULT
tcs_wrap_ReadManuMaintPub(struct tcsd_thread_data *data,
			  struct tsp_packet *tsp_data,
			  struct tcsd_packet_hdr **hdr)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TCPA_NONCE antiReplay;
	TCPA_DIGEST checksum;
	UINT32 size = sizeof(struct tcsd_packet_hdr);

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, tsp_data))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebugFn("thread %x context %x", (UINT32)pthread_self(), hContext);

	if (getData(TCSD_PACKET_TYPE_NONCE, 1, &antiReplay, 0, tsp_data))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_ReadManuMaintPub_Internal(hContext, antiReplay, &checksum);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		*hdr = calloc(1, size + sizeof(TCPA_DIGEST));
		if (*hdr == NULL) {
			LogError("malloc of %zu bytes failed.", size + sizeof(TCPA_DIGEST));
			return TCSERR(TSS_E_OUTOFMEMORY);
		}

		if (setData(TCSD_PACKET_TYPE_DIGEST, 0, &checksum, 0, *hdr)) {
			free(*hdr);
			return TCSERR(TSS_E_INTERNAL_ERROR);
		}
	} else {
		*hdr = calloc(1, size);
		if (*hdr == NULL) {
			LogError("malloc of %u bytes failed.", size);
			return TCSERR(TSS_E_OUTOFMEMORY);
		}
		(*hdr)->packet_size = size;
	}
	(*hdr)->result = result;

	return TSS_SUCCESS;
}
#else
TSS_RESULT
tcs_wrap_ReadManuMaintPub(struct tcsd_thread_data *data)
{
	TCS_CONTEXT_HANDLE hContext;
	TSS_RESULT result;
	TCPA_NONCE antiReplay;
	TCPA_DIGEST checksum;

	if (getData(TCSD_PACKET_TYPE_UINT32, 0, &hContext, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	LogDebugFn("thread %x context %x", (UINT32)pthread_self(), hContext);

	if (getData(TCSD_PACKET_TYPE_NONCE, 1, &antiReplay, 0, &data->comm))
		return TCSERR(TSS_E_INTERNAL_ERROR);

	pthread_mutex_lock(&tcsp_lock);

	result = TCSP_ReadManuMaintPub_Internal(hContext, antiReplay, &checksum);

	pthread_mutex_unlock(&tcsp_lock);

	if (result == TSS_SUCCESS) {
		initData(&data->comm, 1);
		if (setData(TCSD_PACKET_TYPE_DIGEST, 0, &checksum, 0, &data->comm))
			return TCSERR(TSS_E_INTERNAL_ERROR);
	} else
		initData(&data->comm, 0);

	data->comm.hdr.u.result = result;

	return TSS_SUCCESS;
}
#endif

/* Dispatch */
typedef struct tdDispatchTable {
	TSS_RESULT (*Func) (struct tcsd_thread_data *);
	const char *name;
} DispatchTable;

DispatchTable tcs_func_table[TCSD_MAX_NUM_ORDS] = {
	{tcs_wrap_Error,"Error"},   /* 0 */
	{tcs_wrap_OpenContext,"OpenContext"},
	{tcs_wrap_CloseContext,"CloseContext"},
	{tcs_wrap_Error,"Error"},
	{tcs_wrap_TCSGetCapability,"TCSGetCapability"},
	{tcs_wrap_RegisterKey,"RegisterKey"}, /* 5 */
	{tcs_wrap_UnregisterKey,"UnregisterKey"},
	{tcs_wrap_EnumRegisteredKeys,"EnumRegisteredKeys"},
	{tcs_wrap_Error,"Error"},
	{tcs_wrap_GetRegisteredKeyBlob,"GetRegisteredKeyBlob"},
	{tcs_wrap_GetRegisteredKeyByPublicInfo,"GetRegisteredKeyByPublicInfo"}, /* 10 */
	{tcs_wrap_LoadKeyByBlob,"LoadKeyByBlob"},
	{tcs_wrap_LoadKeyByUUID,"LoadKeyByUUID"},
	{tcs_wrap_EvictKey,"EvictKey"},
	{tcs_wrap_CreateWrapKey,"CreateWrapKey"},
	{tcs_wrap_GetPubkey,"GetPubkey"}, /* 15 */
	{tcs_wrap_MakeIdentity,"MakeIdentity"},
	{tcs_wrap_LogPcrEvent,"LogPcrEvent"},
	{tcs_wrap_GetPcrEvent,"GetPcrEvent"},
	{tcs_wrap_GetPcrEventsByPcr,"GetPcrEventsByPcr"},
	{tcs_wrap_GetPcrEventLog,"GetPcrEventLog"}, /* 20 */
	{tcs_wrap_SetOwnerInstall,"SetOwnerInstall"},
	{tcs_wrap_TakeOwnership,"TakeOwnership"},
	{tcs_wrap_OIAP,"OIAP"},
	{tcs_wrap_OSAP,"OSAP"},
	{tcs_wrap_ChangeAuth,"ChangeAuth"}, /* 25 */
	{tcs_wrap_ChangeAuthOwner,"ChangeAuthOwner"},
	{tcs_wrap_Error,"Error"},
	{tcs_wrap_Error,"Error"},
	{tcs_wrap_TerminateHandle,"TerminateHandle"},
	{tcs_wrap_ActivateIdentity,"ActivateIdentity"}, /* 30 */
	{tcs_wrap_Extend,"Extend"},
	{tcs_wrap_PcrRead,"PcrRead"},
	{tcs_wrap_Quote,"Quote"},
	{tcs_wrap_DirWriteAuth,"DirWriteAuth"},
	{tcs_wrap_DirRead,"DirRead"}, /* 35 */
	{tcs_wrap_Seal,"Seal"},
	{tcs_wrap_UnSeal,"UnSeal"},
	{tcs_wrap_UnBind,"UnBind"},
	{tcs_wrap_CreateMigrationBlob,"CreateMigrationBlob"},
	{tcs_wrap_ConvertMigrationBlob,"ConvertMigrationBlob"}, /* 40 */
	{tcs_wrap_AuthorizeMigrationKey,"AuthorizeMigrationKey"},
	{tcs_wrap_CertifyKey,"CertifyKey"},
	{tcs_wrap_Sign,"Sign"},
	{tcs_wrap_GetRandom,"GetRandom"},
	{tcs_wrap_StirRandom,"StirRandom"}, /* 45 */
	{tcs_wrap_GetCapability,"GetCapability"},
	{tcs_wrap_Error,"Error"},
	{tcs_wrap_GetCapabilityOwner,"GetCapabilityOwner"},
	{tcs_wrap_CreateEndorsementKeyPair,"CreateEndorsementKeyPair"},
	{tcs_wrap_ReadPubek,"ReadPubek"}, /* 50 */
	{tcs_wrap_DisablePubekRead,"DisablePubekRead"},
	{tcs_wrap_OwnerReadPubek,"OwnerReadPubek"},
	{tcs_wrap_SelfTestFull,"SelfTestFull"},
	{tcs_wrap_CertifySelfTest,"CertifySelfTest"},
	{tcs_wrap_Error,"Error"}, /* 55 */
	{tcs_wrap_GetTestResult,"GetTestResult"},
	{tcs_wrap_OwnerSetDisable,"OwnerSetDisable"},
	{tcs_wrap_OwnerClear,"OwnerClear"},
	{tcs_wrap_DisableOwnerClear,"DisableOwnerClear"},
	{tcs_wrap_ForceClear,"ForceClear"}, /* 60 */
	{tcs_wrap_DisableForceClear,"DisableForceClear"},
	{tcs_wrap_PhysicalDisable,"PhysicalDisable"},
	{tcs_wrap_PhysicalEnable,"PhysicalEnable"},
	{tcs_wrap_PhysicalSetDeactivated,"PhysicalSetDeactivated"},
	{tcs_wrap_SetTempDeactivated,"SetTempDeactivated"}, /* 65 */
	{tcs_wrap_PhysicalPresence,"PhysicalPresence"},
	{tcs_wrap_Error,"Error"},
	{tcs_wrap_Error,"Error"},
	{tcs_wrap_CreateMaintenanceArchive,"CreateMaintenanceArchive"},
	{tcs_wrap_LoadMaintenanceArchive,"LoadMaintenanceArchive"}, /* 70 */
	{tcs_wrap_KillMaintenanceFeature,"KillMaintenanceFeature"},
	{tcs_wrap_LoadManuMaintPub,"LoadManuMaintPub"},
	{tcs_wrap_ReadManuMaintPub,"ReadManuMaintPub"}
};

int
access_control(struct tcsd_thread_data *thread_data)
{
	int i = 0;
	struct hostent *local_hostent = NULL;

	if ((local_hostent = gethostbyname("localhost")) == NULL) {
		LogError("Error resolving localhost: %s", hstrerror(h_errno));
		return 1;
	}

	/* if the request comes from localhost, or is in the accepted ops list,
	 * approve it */
	if (!strncmp(thread_data->hostname, local_hostent->h_name,
		     MIN((size_t)local_hostent->h_length, strlen(thread_data->hostname)))) {
		return 0;
	} else {
		while (tcsd_options.remote_ops[i]) {
			if ((UINT32)tcsd_options.remote_ops[i] == thread_data->comm.hdr.u.ordinal) {
				LogInfo("Accepted %s operation from %s",
					tcs_func_table[thread_data->comm.hdr.u.ordinal].name,
					thread_data->hostname);
				return 0;
			}
			i++;
		}
	}

	return 1;
}

TSS_RESULT
dispatchCommand(struct tcsd_thread_data *data)
{
	UINT64 offset;
	TSS_RESULT result;

	/* First, check the ordinal bounds */
	if (data->comm.hdr.u.ordinal >= TCSD_MAX_NUM_ORDS) {
		LogError("Illegal TCSD Ordinal");
		return TCSERR(TSS_E_FAIL);
	}

	LogDebug("Dispatching ordinal %u", data->comm.hdr.u.ordinal);
	if (access_control(data)) {
		LogWarn("Denied %s operation from %s",
			tcs_func_table[data->comm.hdr.u.ordinal].name, data->hostname);

		/* set platform header */
		memset(&data->comm.hdr, 0, sizeof(data->comm.hdr));
		data->comm.hdr.packet_size = sizeof(struct tcsd_packet_hdr);
		data->comm.hdr.u.result = TCSERR(TSS_E_FAIL);

		/* set the comm buffer */
		memset(data->comm.buf, 0, data->comm.buf_size);
		offset = 0;
		LoadBlob_UINT32(&offset, data->comm.hdr.packet_size, data->comm.buf, NULL);
		LoadBlob_UINT32(&offset, data->comm.hdr.u.result, data->comm.buf, NULL);

		return TSS_SUCCESS;
	}

	/* Now, dispatch */
	if ((result = tcs_func_table[data->comm.hdr.u.ordinal].Func(data)) == TSS_SUCCESS) {
		/* set the comm buffer */
		offset = 0;
		LoadBlob_UINT32(&offset, data->comm.hdr.packet_size, data->comm.buf, NULL);
		LoadBlob_UINT32(&offset, data->comm.hdr.u.result, data->comm.buf, NULL);
		LoadBlob_UINT32(&offset, data->comm.hdr.num_parms, data->comm.buf, NULL);
		LoadBlob_UINT32(&offset, data->comm.hdr.type_size, data->comm.buf, NULL);
		LoadBlob_UINT32(&offset, data->comm.hdr.type_offset, data->comm.buf, NULL);
		LoadBlob_UINT32(&offset, data->comm.hdr.parm_size, data->comm.buf, NULL);
		LoadBlob_UINT32(&offset, data->comm.hdr.parm_offset, data->comm.buf, NULL);
	}

	return result;
}

TSS_RESULT
getTCSDPacket(struct tcsd_thread_data *data)
{
	/* make sure the all the data is present */
	if (data->comm.hdr.num_parms > 0 &&
	    data->comm.hdr.packet_size !=
		    (UINT32)(data->comm.hdr.parm_offset + data->comm.hdr.parm_size))
		return TSPERR(TSS_E_INTERNAL_ERROR);

	/* dispatch the command to the TCS */
	return dispatchCommand(data);
}
