//@HEADER
// ************************************************************************
// 
//            NOX: An Object-Oriented Nonlinear Solver Package
//                 Copyright (2002) Sandia Corporation
// 
//            LOCA: Library of Continuation Algorithms Package
//                 Copyright (2005) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
// 
// Questions? Contact Roger Pawlowski (rppawlo@sandia.gov) or 
// Eric Phipps (etphipp@sandia.gov), Sandia National Laboratories.
// ************************************************************************
//  CVS Information
//  $Source: /space/CVS/Trilinos/packages/nox/test/epetra/PelletTransport/PelletTransport.H,v $
//  $Author: rhoope $
//  $Date: 2007/06/25 13:50:17 $
//  $Revision: 1.1 $
// ************************************************************************
//@HEADER

// Finite Element Problem Class
/* Provides function (F) and Jacobian evaluations for the following equation 
 * via a 1D linear finite element discretization with Epetra objects.
 *
 * d2u 
 * --- - k * u**2 = 0
 * dx2
 *
 * subject to @ x=0, u=1
 */

#ifndef _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H
#define _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H

// This is needed to get the FillType enum values from NOX_Epetra_Interface
#include "NOX_Epetra.H"
#include "Teuchos_RCP.hpp"

// Forward Declarations
class Epetra_Comm;
class Epetra_Map; 
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Flag to tell how to order the unknowns - Currently not used
enum StoreUnknowns {Interleaved, Staggered};

// Finite Element Problem Class
class PelletTransport { 

public:

  enum ACTIVE_REGIONS {
                        UO2     ,
                        HELIUM  ,
                        CLAD    ,
                        NONE
                      };

  // Constructor
  PelletTransport( int NumGlobalElementsUO2  , double xminUO2  , double xmaxUO2  ,
                   int NumGlobalElementsHe   , double xminHe   , double xmaxHe   ,
                   int NumGlobalElementsClad , double xminClad , double xmaxClad ,
                   Epetra_Comm& Comm, bool restart = false                         );
  
  // Destructor
  ~PelletTransport();

  // Reset problem for next parameter (time) step.
  // For now, this simply updates oldsoln with the given Epetra_Vector
  void reset(const Epetra_Vector& x);

  // Set initial condition for solution vector
  void initializeSoln();

  // Evaluates the function (F) and/or the Jacobian using the solution 
  // values in solnVector.
  bool evaluate(NOX::Epetra::Interface::Required::FillType fType, 
                const Epetra_Vector *solnVector, 
		Epetra_Vector *rhsVector, Epetra_RowMatrix *matrix);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by the PelletTransport class.
  Teuchos::RCP<Epetra_Vector> getSolution();
  
  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by the PelletTransport class.
  Teuchos::RCP<Epetra_CrsMatrix> getJacobian();

  // Return a reference to the Epetra_Vector with the mesh positions
  Teuchos::RCP<Epetra_Vector> getMesh();

  // Accesor function for time step
  double getdt();

  // Return a reference to the Epetra_Vector with the old solution
  Epetra_Vector& getOldSoln();

  // Return a reference to the Epetra_CrsGraph that is generated by
  // the PelletTransport problem class.
  Teuchos::RCP<Epetra_CrsGraph> getGraph();

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraphUsingNodes(Epetra_CrsGraph& AA);

private:

  int    NumGlobalElementsUO2 ;
  int    NumGlobalElementsHe  ;
  int    NumGlobalElementsClad;
  double xminUO2 , xmaxUO2 ;
  double xminHe  , xmaxHe  ;
  double xminClad, xmaxClad;
  double dt;

  bool restart;

  Epetra_Map *StandardNodeMap; 
  Epetra_Map *OverlapNodeMap;
  Epetra_Map *StandardMap; 
  Epetra_Map *OverlapMap;
  Epetra_Import *Importer;
  Epetra_Import *nodeImporter;
  Teuchos::RCP<Epetra_IntVector> elemTypes;
  Teuchos::RCP<Epetra_Vector> xptr;
  Teuchos::RCP<Epetra_Vector> initialSolution;
  Epetra_Vector *oldSolution;
  Epetra_Vector *rhs;
  Teuchos::RCP<Epetra_CrsGraph> AA;
  Teuchos::RCP<Epetra_CrsMatrix> A;
  Epetra_Comm *Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes

  int NumSpecies;         // Number of unknowns owned by this process
  enum { NUMSPECIES = 2}; // Needed for SGI build
  int NumMyNodes;         // Number of nodes owned by this process
  int NumGlobalNodes;     // Total Number of nodes
  int NumMyUnknowns;      // Number of unknowns owned by this process
  int NumGlobalUnknowns;  // Total Number of unknowns

};
#endif




