//@HEADER
// ************************************************************************
// 
//            NOX: An Object-Oriented Nonlinear Solver Package
//                 Copyright (2002) Sandia Corporation
// 
//            LOCA: Library of Continuation Algorithms Package
//                 Copyright (2005) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
// 
// Questions? Contact Roger Pawlowski (rppawlo@sandia.gov) or 
// Eric Phipps (etphipp@sandia.gov), Sandia National Laboratories.
// ************************************************************************
//  CVS Information
//  $Source: /space/CVS/Trilinos/packages/nox/test/epetra/LOCA_Brusselator/Brusselator.H,v $
//  $Author: rppawlo $
//  $Date: 2006/08/22 00:01:53 $
//  $Revision: 1.3 $
// ************************************************************************
//@HEADER

// Finite Element Problem Class
/* Provides function (F) and Jacobian evaluations for the following equation 
 * via a 1D linear finite element discretization with Epetra objects.
 *
 * d2u 
 * --- - k * u**2 = 0
 * dx2
 *
 * subject to @ x=0, u=1
 */

#ifndef _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H
#define _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H

// This is needed to get the FillType enum values from NOX_Epetra_Interface
#include "NOX_Epetra.H"

// Forward Declarations
class Epetra_Comm;
class Epetra_Map; 
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Flag to tell the evaluate routine what objects to fill
enum FillType { F_ONLY, MATRIX_ONLY, ALL }; 

// Finite Element Problem Class
class Brusselator { 

public:

  // Constructor
  Brusselator( int NumGlobalUnknowns, Epetra_Comm& Comm);
  
  // Destructor
  ~Brusselator();

  // Reset problem for next parameter (time) step.
  // For now, this simply updates oldsoln with the given Epetra_Vector
  void reset(const Epetra_Vector& x);

  // Set initial condition for solution vector
  void initializeSoln();

  // Set parameters alpha, beta
  void setParameters(double alpha_, double beta_, double D1_, double D2_);

  // Evaluates the function (F) and/or the Jacobian using the solution 
  // values in solnVector.
  bool evaluate(FillType fType, 
                const Epetra_Vector *solnVector, 
		Epetra_Vector *rhsVector, Epetra_RowMatrix *matrix,
		double jac_coeff = 1.0, 
		double mass_coeff = 0.0);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by the Brusselator class.
  Epetra_Vector& getSolution();
  
  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by the Brusselator class.
  Epetra_CrsMatrix& getJacobian();

  // Return a reference to the Epetra_Vector with the mesh positions
  Epetra_Vector& getMesh();

  // Return a reference to the Epetra_CrsGraph that is generated by
  // the Brusselator problem class.
  Epetra_CrsGraph& getGraph();

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraph(Epetra_CrsGraph& AA);

  // Private to prohibit copying
  Brusselator(const Brusselator&);
  Brusselator& operator=(const Brusselator&);

private:

  double xmin;
  double xmax;
  Epetra_Comm *Comm;
  int NumGlobalNodes;     // Total Number of nodes
  Epetra_Import *ColumnToOverlapImporter;
  Epetra_CrsGraph *AA;
  Epetra_CrsMatrix *A;
  double alpha, beta, D1, D2;     // Model Parameters

  // Default initialization for these
  double dx;
  Epetra_Map *StandardNodeMap; 
  Epetra_Map *OverlapNodeMap;
  Epetra_Map *StandardMap; 
  Epetra_Map *OverlapMap;
  Epetra_Import *Importer;
  Epetra_Import *nodeImporter;
  Epetra_Vector *xptr;
  Epetra_Vector *initialSolution;
  Epetra_Vector *rhs;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes

  int NumSpecies;         // Number of unknowns owned by this process
  enum { NUMSPECIES = 2}; // Needed for SGI build
  int NumMyNodes;         // Number of nodes owned by this process
  int NumMyUnknowns;      // Number of unknowns owned by this process
  int NumGlobalUnknowns;  // Total Number of unknowns

};
#endif




